/* This is a script that lists the day's callers.
 *
 * Author: Eric Oulashin (AKA Nightfox)
 * BBS: Digital Distortion
 * BBS address: digdist.bbsindex.com
 *
 * Date       User              Description
 * 2009-04-30 Eric Oulashin     Created
 * 2009-05-01 Eric Oulashin     Continued work
 * 2009-05-02 Eric Oulashin     Continued work - Used substring to make
 *                              sure field values don't exceed the field
 *                              widths.  Updated to read logon.lst in the
 *                              data directory rather than going through
 *                              the user records and matching their last
 *                              logon date.
 * 2009-05-03 Eric Oulashin     Refactored the script a bit.  Added a
 *                              boolean, pauseAtEnd, which will control
 *                              whether or not the script will pause at
 *                              the end, and set up argv[0] to change
 *                              that too.  Also, added a Synchronet
 *                              version check.
 * 2009-05-04 Eric Oulashin     Changed the User variable's name from user
 *                              to theUser to avoid stomping on the user
 *                              object declared in sbbsdefs.js.  Also,
 *                              updated to read through the system user
 *                              records if unable to read the logon.lst file.
 *                              Either way, the caller information is now
 *                              placed in an array, and the last X callers
 *                              are displayed at the end.  Also, made use
 *                              of the newly-added variable DIGDIST_INCLUDE_SYSOP
 *                              in DigitalDistortionDefs.js.
 * 2009-05-09 Eric Oulashin     Updated to only add the caller information to
 *                              the array if the user's handle & logon time
 *                              read from logon.lst are not blank.  This fixed
 *                              a bug that was causing the script to occasionally
 *                              show a single line in the logon list for the day's
 *                              first caller.
 * 2009-05-10 Eric Oulashin     Removed the "donePrompt" color from the colors
 *                              array, because it wasn't being used.
 * 2009-05-13 Eric Oulashin     Updated to be free of DigitalDistortionDefs.js.
 *                              Added VERSION and VER_DATE, although they aren't
 *                              displayed to the user.
 * 2009-06-16 Eric Oulashin     Version 1.02
 *                              - Put the caller listing code into a class so that
 *                                the caller list can either be executed in this
 *                                script (default), or this script can be loaded
 *                                into another script, and the user may use the
 *                                class as desired.
 *                              - Added parameters to the user listing functions
 *                                to specify whether to return the caller list as
 *                                as a string rather than outputting it to the
 *                                screen.
 */

/* These are the command-line arguments:
 * 0: Whether or not to pause at the end of the listing (true/false)
 * 1: The maximum number of callers to list
 * 2: Whether or not to execute the caller list (true/false).  This
 *    dfaults to true.  You would want to pass false if you are including
 *    this script in another JavaScript file and wish to execute it in your
 *    own way.
 */

load("sbbsdefs.js");


// This script requires Synchronet version 3.10 or higher.
// Exit if the Synchronet version is below the minimum.
if (system.version_num < 31000)
{
	var message = "nhyi* Warning:nhw Digital Distortion List Today's Callers "
	             + "requires version g3.10w or\r\n"
	             + "higher of Synchronet.  This BBS is using version g" + system.version
	             + "w.  Please notify the sysop.";
	console.crlf();
	console.print(message);
	console.crlf();
	console.print("p");
	exit();
}

// Version and version date.
// These are declared with "var" rather than "const" to avoid
// an error about re-definitions.
var VERSION = "1.02";
var VER_DATE = "2009-06-16";

// Single-line box drawing/border characters.
// These are declared with "var" rather than "const" to avoid
// an error about re-definitions.
var UPPER_LEFT_SINGLE = "";
var HORIZONTAL_SINGLE = "";
var UPPER_RIGHT_SINGLE = "";
var VERTICAL_SINGLE = "";
var LOWER_LEFT_SINGLE = "";
var LOWER_RIGHT_SINGLE = "";
var T_SINGLE = "";
var LEFT_T_SINGLE = "";
var RIGHT_T_SINGLE = "";
var BOTTOM_T_SINGLE = "";
var CROSS_SINGLE = "";
// Double-line box drawing/border characters.
var UPPER_LEFT_DOUBLE = "";
var HORIZONTAL_DOUBLE = "";
var UPPER_RIGHT_DOUBLE = "";
var VERTICAL_DOUBLE = "";
var LOWER_LEFT_DOUBLE = "";
var LOWER_RIGHT_DOUBLE = "";
var T_DOUBLE = "";
var LEFT_T_DOUBLE = "";
var RIGHT_T_DOUBLE = "";
var BOTTOM_T_DOUBLE = "";
var CROSS_DOUBLE = "";

///////////////////////////
// Main script execution //
///////////////////////////

// The 3rd program argument specifies whether or not to execute the script code.
var executeScriptCode = true;
if ((typeof(argv[2]) != "undefined") && (argv[2] != null))
   executeScriptCode = argv[2];

if (executeScriptCode)
{
   var callerLister = new DDTodaysCallerLister();

   // The first program argument can change pauseAtEnd.
   if ((typeof(argv[0]) != "undefined") && (argv[0] != null))
      callerLister.pauseAtEnd = argv[0];
   // The second program argument can change MAX_CALLERS.
   if ((typeof(argv[1]) != "undefined") && (argv[1] != null))
      callerLister.MAX_CALLERS = argv[1];

   callerLister.listTodaysCallers(false);
}

// End of script execution


//////////////////////////////////////////////////////////////////////////////////
// Class & functions

// This is the constructor for the CallerInfo class, which contains
// information about a caller.
function CallerInfo()
{
   this.nodeNumber = "1";
   this.handle = "";
   this.location = "";
   this.logonTime = "0";
   this.connectionType = "";
   this.numCallsToday = "0";
}

// This is the constructor for the DDTodaysCallerLister class, which
// performs a listing of the day's callers.
function DDTodaysCallerLister()
{
   // Colors
   this.colors = new Array();
   this.colors["veryTopHeader"] = "nhw";
   this.colors["border"] = "nc";
   this.colors["colLabelText"] = "n4hw";
   this.colors["timeOn"] = "hc";
   this.colors["userName"] = "hy";
   this.colors["location"] = "hm";
   this.colors["nodeNumber"] = "hr";
   this.colors["connectionType"] = "hb";
   this.colors["numCalls"] = "hg";

   // pauseAtEnd controls whether or not to pause at the end of the listing.
   this.pauseAtEnd = true;
   // The maximum number of callers to display
   this.MAX_CALLERS = 6;
   // Whether or not to include the sysop in the list.
   this.includeSysop = true;
   // Whether or not to output the list as HTML (using html_encode()).
   this.doHTML = false;

   // Column inner lengths
   this.TIME_ON_LEN = 7;
   this.USER_NAME_LEN = 18;
   this.LOCATION_LEN = 25;
   this.NODE_NUM_LEN = 5;
   this.CONNECTION_TYPE_LEN = 10;
   this.NUM_CALLS_LEN = 6;

   // This is the message that is displayed if the user is the first caller of the day.
   this.firstCallerMessage = this.colors["veryTopHeader"] + "You are the first caller of the day!";

   // callerInfoArray is an array of CallerInfo objects containing
   // the day's caller information.
   this.callerInfoArray = new Array();

   // Member methods
   this.populateCallerInfoArray = DDTodaysCallerLister_PopulateCallerInfoArray;
   this.listTodaysCallers = DDTodaysCallerLister_ListTodaysCallers;
   this.writeHeader = DDTodaysCallerLister_WriteHeader;
   this.writeEndBorder = DDTodaysCallerLister_WriteEndBorder;
}
// For the DDTodaysCallerLister class: Populates the array of callers.
function DDTodaysCallerLister_PopulateCallerInfoArray()
{
   // Clear the caller info array
   this.callerInfoArray = new Array();

   // If the user is not the first caller, then open logon.lst in the data dir.
   // If the file can't be opened, then assume the user is the first caller of
   // the day.
   var loginFile = null;
   var loginFileOpened = false;
   if (system.stats.logons_today > 0)
   {
      loginFile = new File(system.data_dir + "logon.lst");
      if (loginFile.exists)
         loginFileOpened = loginFile.open("r", true);
   }
   
   // See if the user is the first caller.
   if (system.stats.logons_today == 0)
   {
      // Do nothing.
   }
   // The user is not the first caller.  If the login file was opened, read the
   // caller information from it.
   else if (loginFileOpened && (loginFile != null))
   {
      // Read the lines from the file and insert the caller information into
      // this.callerInfoArray.
      var fileLine = "";
      var userHandle = ""; // For testing for valid data
      var logonTime = "";  // For testing for valid data
      var callerInfo = null;
      while (!loginFile.eof)
      {
         // Read the line from the file, stripping control characters.
         fileLine = strip_ctrl(loginFile.readln(512));
      
         // Skip blank lines
         if (fileLine == "")
            continue;
         
         /* Fields
            ------
            0-2: Node #
            3-9: Total # of calls to the system?
            10-35: User handle
            36-60: User location
            61-65: Login time
            67-74: Connection type
            76-80: The user's number of calls today
         */
         // Only add the caller info if the user's handle & logon time are
         // not blank.
         userHandle = trimSpaces(fileLine.substring(10, 36), false, false, true);
         userLogonTime = trimSpaces(fileLine.substring(61, 66), false, false, true);
         if ((userHandle != "") && (userLogonTime != ""))
         {
            callerInfo = new CallerInfo();
            callerInfo.nodeNumber = trimSpaces(fileLine.substring(0, 2), false, false, true);
            callerInfo.handle = userHandle;
            callerInfo.location = trimSpaces(fileLine.substring(36, 61), false, false, true);
            callerInfo.logonTime = userLogonTime;
            callerInfo.connectionType = trimSpaces(fileLine.substring(67, 75), false, false, true);
            callerInfo.numCallsToday = trimSpaces(fileLine.substring(76, 81), false, false, true);
            this.callerInfoArray.push(callerInfo);
         }
      }
      loginFile.close();
   }
   // The user is not the first caller, and the login file was not opened.  Look through
   // the user records for users who called today.
   else
   {
      // Go through each user record.  If their last logon date was today, then
      // add their user information to this.callerInfoArray.
      var callerInfo = null;
      for (var userNum = 1; userNum <= system.stats.total_users; ++userNum)
      {
         // If the sysop is not to be included, then skip user # 1 (user #1
         // is more than likely the sysop).
         if (!this.includeSysop && (userNum == 1))
            continue;
   
         theUser = new User(userNum);
         // If the user logged on today, then add the user information to
         // this.callerInfoArray.
         if (theUser.stats.logons_today > 0)
         {
            callerInfo = new CallerInfo();
            callerInfo.nodeNumber = "";
            callerInfo.handle = theUser.alias;
            callerInfo.location = theUser.location;
            callerInfo.logonTime = strftime("%H:%M", theUser.stats.laston_date);
            callerInfo.connectionType = theUser.connection;
            callerInfo.numCallsToday = theUser.stats.logons_today;
            this.callerInfoArray.push(callerInfo);
         }
      }
   
      // Sort the array by user logon time
      this.callerInfoArray.sort(function(pA, pB)
      {
         // Get both users' logon times, converted to numbers without the colons.
         var userALogonTime = +pA.logonTime.replace(":", "");
         var userBLogonTime = +pB.logonTime.replace(":", "");
         
         // Return -1, 0, or 1, depending on whether user A's logon
         // time is before, equal, or after user B's logon time.
         var returnValue = 0;
         if (userALogonTime < userBLogonTime)
            returnValue = -1;
         else if (userALogonTime > userBLogonTime)
            returnValue = 1;
            
         return returnValue;
      });
   }
}
// For the DDTodaysCallerLister class: Lists the day's callers.
//
// Parameters:
//  pReturnInString: Boolean - Whether or not to return the caller listing
//                   as a string rather than outputting it to the scren.
//                   This is options.
function DDTodaysCallerLister_ListTodaysCallers(pReturnInString)
{
   var returnInString = false;
   if (pReturnInString != null)
      returnInString = pReturnInString;

   if (!returnInString)
      console.clear();

   var returnStr = "";

   // Populate the caller info array
   this.populateCallerInfoArray();
   
   // If this.callerInfoArray has any caller information, then display it.
   if (this.callerInfoArray.length > 0)
   {
      var lineText = "";

      // The following line which sets console.line_counter to 0 is a
      // kludge to disable Synchronet's automatic pausing after a
      // screenful of text.
      if (!returnInString)
         console.line_counter = 0;
      
      // Write the caller list header
      returnStr = this.writeHeader(returnInString);
      
      // We want to display the last MAX_CALLERS number of callers.  First,
      // calculate the starting array index based on MAX_CALLERS.  If
      // MAX_CALLERS is 0 or negative, then display all the callers in the
      // array.
      var startIndex = 0;
      if ((this.MAX_CALLERS > 0) && (this.callerInfoArray.length > this.MAX_CALLERS))
         startIndex = this.callerInfoArray.length - this.MAX_CALLERS;
      // Display the callers.
      for (var i = startIndex; i < this.callerInfoArray.length; ++i)
      {
         lineText = this.colors["border"] + VERTICAL_SINGLE;
         lineText += this.colors["timeOn"] + centerText(this.callerInfoArray[i].logonTime, this.TIME_ON_LEN);
         lineText += this.colors["border"] + VERTICAL_SINGLE;
         lineText += this.colors["userName"] + centerText(this.callerInfoArray[i].handle.substring(0, this.USER_NAME_LEN), this.USER_NAME_LEN);
         lineText += this.colors["border"] + VERTICAL_SINGLE;
         lineText += this.colors["location"] + centerText(this.callerInfoArray[i].location.substring(0, this.LOCATION_LEN), this.LOCATION_LEN);
         lineText += this.colors["border"] + VERTICAL_SINGLE;
         lineText += this.colors["nodeNumber"] + centerText(this.callerInfoArray[i].nodeNumber, this.NODE_NUM_LEN)
         lineText += this.colors["border"] + VERTICAL_SINGLE;
         lineText += this.colors["connectionType"];
         lineText += centerText(this.callerInfoArray[i].connectionType.substring(0, this.CONNECTION_TYPE_LEN), this.CONNECTION_TYPE_LEN);
         lineText += this.colors["border"] + VERTICAL_SINGLE;
         lineText += this.colors["numCalls"];
         lineText += centerText(format("%" + this.NUM_CALLS_LEN + "d", this.callerInfoArray[i].numCallsToday), this.NUM_CALLS_LEN);
         lineText += this.colors["border"] + VERTICAL_SINGLE;
         if (returnInString)
            returnStr += lineText + "\r\n";
         else
            console.center(lineText);
      }

      // Write the end border for the table
      returnStr += this.writeEndBorder(returnInString);
   }
   else
   {
      // There are no elements in this.callerInfoArray, so tell the user that they
      // are the first caller.
      if (returnInString)
         returnStr = this.firstCallerMessage;
      else
         console.center(this.firstCallerMessage);
   }
   
   // We're done.  If pauseAtEnd is true, then display the system's pause/"Hit a key" prompt.
   if (this.pauseAtEnd && !returnInString)
      console.print("np");

   return returnStr;
}
// For the DDTodaysCallerLister class: Writes the header for the user listing.
//
// Parameters:
//  pReturnInString: Boolean - Whether or not to return the caller listing
//                   as a string rather than outputting it to the scren.
//                   This is options.
function DDTodaysCallerLister_WriteHeader(pReturnInString)
{
   var returnInString = false;
   if (pReturnInString != null)
      returnInString = pReturnInString;

   var returnStr = "";

	// Write the header
	if (returnInString)
	{
      var tableWidth = this.TIME_ON_LEN + this.USER_NAME_LEN + this.LOCATION_LEN
                      + this.NODE_NUM_LEN + this.CONNECTION_TYPE_LEN + this.NUM_CALLS_LEN
                      + 7;
      returnStr = this.colors["veryTopHeader"] + centerText("Last several callers", tableWidth) + "\r\n";
	}
	else
	{
      console.print(this.colors["veryTopHeader"]);
      console.center("Last several callers");
   }
	// Upper border lines
	// "Time on" section
	lineText = this.colors["border"] + UPPER_LEFT_SINGLE;
	for (var i = 0; i < this.TIME_ON_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// User name section
	for (var i = 0; i < this.USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Location section
	for (var i = 0; i < this.LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Node number section
	for (var i = 0; i < this.NODE_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// Connection type section
	for (var i = 0; i < this.CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += T_SINGLE;
	// # of calls section
	for (var i = 0; i < this.NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += UPPER_RIGHT_SINGLE;
	if (returnInString)
      returnStr += lineText + "\r\n";
	else
      console.center(lineText + "\r\n");

	// Column labels
	lineText = this.colors["border"] + VERTICAL_SINGLE + this.colors["colLabelText"]
			 + centerText("Time On", this.TIME_ON_LEN) + this.colors["border"] + VERTICAL_SINGLE
			 + this.colors["colLabelText"] + centerText("User Name", this.USER_NAME_LEN) + this.colors["border"]
			 + VERTICAL_SINGLE + this.colors["colLabelText"]
			 + centerText("Location", this.LOCATION_LEN) + this.colors["border"] + VERTICAL_SINGLE
			 + this.colors["colLabelText"]
			 + centerText("Node#", this.NODE_NUM_LEN)  + this.colors["border"] + VERTICAL_SINGLE
			 + this.colors["colLabelText"] + centerText("Connection", this.CONNECTION_TYPE_LEN)
			 + this.colors["border"] + VERTICAL_SINGLE + this.colors["colLabelText"]
			 + centerText("#Calls", this.NUM_CALLS_LEN) + this.colors["border"]
			 + VERTICAL_SINGLE;
	if (returnInString)
      returnStr += lineText + "\r\n";
	else
      console.center(lineText + "\r\n");

	// Lower border lines for header
	// "Time on" section
	lineText = this.colors["border"] + LEFT_T_SINGLE;
	for (var i = 0; i < this.TIME_ON_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// User name section
	for (var i = 0; i < this.USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Location section
	for (var i = 0; i < this.LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Node number section
	for (var i = 0; i < this.NODE_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// Connection type section
	for (var i = 0; i < this.CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += CROSS_SINGLE;
	// # of calls section
	for (var i = 0; i < this.NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += RIGHT_T_SINGLE;
	if (returnInString)
      returnStr += lineText + "\r\n";
   else
      console.center(lineText);

   return returnStr;
}
// For the DDTodaysCallerLister class: Writes the end border.
//
// Parameters:
//  pReturnInString: Boolean - Whether or not to return the caller listing
//                   as a string rather than outputting it to the scren.
//                   This is options.
function DDTodaysCallerLister_WriteEndBorder(pReturnInString)
{
   var returnInString = false;
   if (pReturnInString != null)
      returnInString = pReturnInString;

   var returnStr = "";

	// "Time on" section
	var lineText = this.colors["border"] + LOWER_LEFT_SINGLE;
	for (var i = 0; i < this.TIME_ON_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// User name section
	for (var i = 0; i < this.USER_NAME_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Location section
	for (var i = 0; i < this.LOCATION_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Node number section
	for (var i = 0; i < this.NODE_NUM_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// Connection type section
	for (var i = 0; i < this.CONNECTION_TYPE_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += BOTTOM_T_SINGLE;
	// # of calls section
	for (var i = 0; i < this.NUM_CALLS_LEN; ++i)
		lineText += HORIZONTAL_SINGLE;
	lineText += LOWER_RIGHT_SINGLE;
	if (returnInString)
      returnStr = lineText;
	else
      console.center(lineText);

   return returnStr;
}


// The following variable is used for a line of text throughout this script.
var lineText = "";




///////////////////////////////////////////////////////////////////////////////////
// Functions

// Removes multiple, leading, and/or trailing spaces
// The search & replace regular expressions used in this
// function came from the following URL:
//  http://qodo.co.uk/blog/javascript-trim-leading-and-trailing-spaces
//
// Parameters:
//  pString: The string to trim
//  pLeading: Whether or not to trim leading spaces (optional, defaults to true)
//  pMultiple: Whether or not to trim multiple spaces (optional, defaults to true)
//  pTrailing: Whether or not to trim trailing spaces (optional, defaults to true)
function trimSpaces(pString, pLeading, pMultiple, pTrailing)
{
	var leading = true;
	var multiple = true;
	var trailing = true;
	if(typeof(pLeading) != "undefined")
		leading = pLeading;
	if(typeof(pMultiple) != "undefined")
		multiple = pMultiple;
	if(typeof(pTrailing) != "undefined")
		trailing = pTrailing;
		
	// To remove both leading & trailing spaces:
	//pString = pString.replace(/(^\s*)|(\s*$)/gi,"");

	if (leading)
		pString = pString.replace(/(^\s*)/gi,"");
	if (multiple)
		pString = pString.replace(/[ ]{2,}/gi," ");
	if (trailing)
		pString = pString.replace(/(\s*$)/gi,"");

	return pString;
}

// Centers some text within a specified field length.
//
// Parameters:
//  pText: The text to center
//  pFieldLen: The field length
function centerText(pText, pFieldLen)
{
	var centeredText = "";
	
	// If pFieldLen is less than the text length, then truncate the string.
	if (pFieldLen < pText.length)
	{
		centeredText = pText.substring(0, pFieldLen);
	}
	else
	{
		// pFieldLen is at least equal to the text length, so we can
		// center the text.
		// Calculate the number of spaces needed to center the text.
		var numSpaces = pFieldLen - pText.length;
		if (numSpaces > 0)
		{
			var rightSpaces = (numSpaces/2).toFixed(0);
			var leftSpaces = numSpaces - rightSpaces;
			// Build centeredText
			for (var i = 0; i < leftSpaces; ++i)
				centeredText += " ";
			centeredText += pText;
			for (var i = 0; i < rightSpaces; ++i)
				centeredText += " ";
		}
		else
		{
			// pFieldLength is the same length as the text.
			centeredText = pText;
		}
	}

	return centeredText;
}