PROGRAM CheckOut;

{ this program checks the output for a certain test case by  }
{ comparing the files listed in output.txt pointed to on the }
{ command line.                                              }
{ returns errorlevel>0 on error.                             }

USES Dos,
     Ramon,
     BinTxt;

VAR LogFile   : TEXT;
    LogPath   : STRING;
    WorkDrive : STRING[2];
    CheckOK   : BOOLEAN;

FUNCTION TimeStamp : STRING;

VAR Hour,Min,Sec,SecH  : WordLong;

BEGIN
     GetTime (Hour,Min,Sec,SecH);

     TimeStamp:=AddUpWithPre0s (2,Word2String (Hour))+':'+
                AddUpWithPre0s (2,Word2String (Min))+':'+
                AddUpWithPre0s (2,Word2String (Sec));
END;


PROCEDURE Log (Msg : STRING);
BEGIN
     Msg:=TimeStamp+Spaces (5)+Msg;
     WriteLn (LogFile,Msg);
     WriteLn (Msg);
END;


PROCEDURE LogErr (IORes : BYTE; Msg : STRING);
BEGIN
     Log (Msg+' (error '+Byte2String (IORes)+')');
END;


FUNCTION RemoveUnprintable (Regel : STRING) : STRING;

VAR Result : STRING;
    Lp     : BYTE;

BEGIN
     Result:='';
     FOR Lp:=1 TO Length (Regel) DO
         IF (Regel[Lp] IN [' '..'~']) THEN
            Result:=Result+Regel[Lp]
         ELSE
             Result:=Result+'['+Byte2HexString (Byte (Regel[Lp]))+']';

     RemoveUnprintable:=Result;
END;


VAR Diff      : ARRAY[1..1024] OF LONGINT;
    DiffPos   : WORD;
    DiffCount : WORD;

FUNCTION IsNextDiff (P : LONGINT) : BOOLEAN;
BEGIN
     IsNextDiff:=FALSE;

     WHILE (DiffPos < DiffCount) DO
     BEGIN
          Inc (DiffPos);
          IF (Diff[DiffPos] = P) THEN
          BEGIN
               IsNextDiff:=TRUE;
               Exit;
          END;
     END;
END;

PROCEDURE StoreDiff (Regel : STRING);

VAR Part : STRING[30];
    P    : BYTE;
    Nr1,
    Nr2  : LONGINT;
    Nop  : ValNop;

BEGIN
     DiffCount:=0;
     DiffPos:=0;

     WHILE (Regel <> '') DO
     BEGIN
          P:=Pos (' ',Regel);
          IF (P = 0) THEN
          BEGIN
               Part:=Regel;
               Regel:='';
          END ELSE
          BEGIN
               Part:=Copy (Regel,1,P-1);
               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);
          END;

          P:=Pos ('-',Part);
          IF (P <> 0) THEN
          BEGIN
               Nr1:=HexString2Long (Copy (Part,1,P-1));
               Nr2:=HexString2Long (Copy (Part,P+1,255));
          END ELSE
          BEGIN
               P:=Pos ('+',Part);
               IF (P <> 0) THEN
               BEGIN
                    Nr1:=HexString2Long (Copy (Part,1,P-1));
                    Val (Copy (Part,P+1,255),Nr2,Nop); { aantal, including Nr1 }
                    IF (Nop <> 0) THEN
                       Log ('Syntax error in offset: '+Part);
                    Nr2:=Nr1+Nr2-1;
               END ELSE
               BEGIN
                    Nr1:=HexString2Long (Part);
                    Nr2:=Nr1;
               END;
          END;

          WHILE (Nr1 <= Nr2) DO
          BEGIN
               Inc (DiffCount);
               Diff[DiffCount]:=Nr1;
               Inc (Nr1);
          END;
     END; { while }
END;


PROCEDURE CompareBinFiles (Path1,Path2 : STRING);

CONST BLOCK_SIZE = 1024;

VAR In1,In2 : FILE;
    P,
    IORes   : BYTE;
    Temp    : STRING;
    Blok1,
    Blok2   : ARRAY[1..BLOCK_SIZE] OF BYTE;
    Lp,
    BytesRead1,
    BytesRead2 : WordLong;
    FP         : LONGINT;

LABEL CloseEm,Abort;

BEGIN
     Path2:=DeleteFrontSpaces (Path2);

     { Path1 must be the file to check }
     IF (Pos ('$:',Path1) = 0) AND (Pos ('$:',Path2) > 0) THEN
     BEGIN
          Temp:=Path1;
          Path1:=Path2;
          Path2:=Temp;
     END;

     P:=Pos ('$:',Path1);
     IF (P > 0) THEN
        Path1:=Copy (Path1,1,P-1)+WorkDrive+Copy (Path1,P+2,255);

     Assign (In1,Path1);
     {$I-} Reset (In1,1); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          LogErr (IORes,'CheckOut: Error opening '+Path1);
          CheckOK:=FALSE;
          Exit;
     END;

     Assign (In2,Path2);
     {$I-} Reset (In2,1); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          LogErr (IORes,'CheckOut: Error opening '+Path2);
          CheckOK:=FALSE;
          Close (In1);
          Exit;
     END;

     IF (FileSize (In1) <> FileSize (In2)) THEN
     BEGIN
          Log ('CheckOut: Wrong size for '+Path1+
               ' ('+Longint2String (FileSize (In1))+
               ' instead of '+Longint2String (FileSize (In2))+')');
          GOTO Abort;
     END;

     REPEAT
           FP:=FilePos (In1);

           {$I-} BlockRead (In1,Blok1[1],BLOCK_SIZE,BytesRead1); {$I+} IORes:=IOResult;
           IF (IORes <> 0) THEN
           BEGIN
                LogErr (IORes,'CheckOut: Error reading from '+Path1);
                GOTO Abort;
           END;

           {$I-} BlockRead (In2,Blok2[1],BLOCK_SIZE,BytesRead2); {$I+} IORes:=IOResult;
           IF (IORes <> 0) THEN
           BEGIN
                LogErr (IORes,'CheckOut: Error reading from '+Path2);
                GOTO Abort;
           END;

           IF (BytesRead1 <> BytesRead2) THEN
           BEGIN
                Log ('CheckOut: BlockRead failed ('+Word2String (BytesRead1)+
                     '/'+Word2String (BytesRead2)+')');
                GOTO Abort;
           END;

           FOR Lp:=1 TO BytesRead1 DO
               IF (Blok1[Lp] <> Blok2[Lp]) THEN
                  IF (NOT IsNextDiff (FP+Lp-1)) THEN
                  BEGIN
                       Log ('CheckOut ERROR: File '+Path1+' and '+Path2+' differ');
                       GOTO Abort;
                  END;

     UNTIL (BytesRead1 = 0);

     GOTO CloseEm;

Abort:
     CheckOK:=FALSE;

CloseEm:
     Close (In1);
     Close (In2);
END;


PROCEDURE CompareTextFiles (Path1,Path2 : STRING);

CONST BLOCK_SIZE = 1024;

TYPE BlockBuf = ARRAY[1..BLOCK_SIZE] OF CHAR;

    FUNCTION ReadLine (VAR F : FILE;
                       VAR Buf : BlockBuf;
                       VAR Count : WORD;
                       VAR GetPos : WORD) : STRING;

    VAR BytesRead : WordLong;
        IORes     : BYTE;
        Regel     : STRING;
        Ch        : CHAR;

    BEGIN
         Regel:='';

         REPEAT
               IF (GetPos = Count) THEN
               BEGIN
                    {$I-} BlockRead (F,Buf[1],BLOCK_SIZE,BytesRead); {$I+} IORes:=IOResult;
                    Count:=BytesRead;
                    GetPos:=0;
               END;

               IF (Count = 0) THEN
               BEGIN
                    Ch:=#13; { exits repeat/until loop }
               END ELSE
               BEGIN
                    Inc (GetPos);
                    Ch:=Buf[GetPos];

                    Regel:=Regel+Ch;
               END;

         UNTIL (Ch IN [#13,#10]) OR (Length (Regel) = 255);

         ReadLine:=Regel;
    END;

    FUNCTION DiffIsAstrixes (Line1,Line2 : STRING) : BOOLEAN;

    VAR Lp : BYTE;

    BEGIN
         DiffIsAstrixes:=FALSE;

         IF (Length (Line1) <> Length (Line2)) THEN
            Exit;

         FOR Lp:=1 TO Length (Line1) DO
             IF (Line1[Lp] <> Line2[Lp]) AND (Line2[Lp] <> '*') THEN
                Exit;

         DiffIsAstrixes:=TRUE;
    END;

{CompareTextFiles}

VAR P       : BYTE;
    IORes   : BYTE;
    In1,
    In2     : FILE;
    Buf1,
    Buf2    : BlockBuf;
    Count1,
    Count2  : WORD;
    GetPos1,
    GetPos2 : WORD;
    Regel1,
    Regel2  : STRING;
    Test1,
    Test2   : STRING;
    ThisOK  : BOOLEAN;
    LineNr  : LONGINT;

LABEL CloseEm;

BEGIN
     Path2:=DeleteFrontSpaces (Path2);

     P:=Pos ('$:',Path1);
     IF (P > 0) THEN
        Path1:=Copy (Path1,1,P-1)+WorkDrive+Copy (Path1,P+2,255);

     Assign (In1,Path1);
     {$I-} Reset (In1,1); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          LogErr (IORes,'CheckOut: Error opening '+Path1);
          CheckOK:=FALSE;
          Exit;
     END;

     Assign (In2,Path2);
     {$I-} Reset (In2,1); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          LogErr (IORes,'CheckOut: Error opening '+Path2);
          CheckOK:=FALSE;
          Close (In1);
          Exit;
     END;

     Count1:=0;
     Count2:=0;

     GetPos1:=0;
     GetPos2:=0;

     LineNr:=0;

     ThisOK:=TRUE;

     REPEAT
           Regel1:=ReadLine (In1,Buf1,Count1,GetPos1);
           Regel2:=ReadLine (In2,Buf2,Count2,GetPos2);
           Inc (LineNr);

           IF (Regel1 = Regel2) THEN
              Continue;

           Test1:=Regel1;
           Test2:=Regel2;
           WHILE (Test1[Length (Test1)] = Test2[Length (Test2)]) DO
           BEGIN
                Delete (Test1,Length (Test1),1);
                Delete (Test2,Length (Test2),1);
           END;

           WHILE (Test1[1] = Test2[1]) DO
           BEGIN
                Delete (Test1,1,1);
                Delete (Test2,1,1);
           END;

           IF (Test2 = '%WGVER%') THEN
              Continue;

           IF (Test2 = '%WGDATE%') THEN
              Continue;

           IF (Test2 = '%WGVIA%') THEN
              Continue;

           IF DiffIsAstrixes (Test1,Test2) THEN
              Continue;

           Log ('CheckOut ERROR: File '+Path1+' and '+Path2+' differ at line '+Longint2String (LineNr));
           Log ('         Line1: '+RemoveUnprintable (Regel1));
           Log ('         Line2: '+RemoveUnprintable (Regel2));
           ThisOK:=FALSE;

     UNTIL (NOT ThisOK) OR (Count1+Count2 = 0);

     IF (NOT ThisOK) THEN
        CheckOK:=FALSE;

CloseEm:
     Close (In1);
     Close (In2);
END;


PROCEDURE CheckExist (Path : STRING);

VAR P      : BYTE;
    IORes  : BYTE;
    InFile : FILE;

BEGIN
     P:=Pos ('$:',Path);
     IF (P > 0) THEN
        Path:=Copy (Path,1,P-1)+WorkDrive+Copy (Path,P+2,255);

     Assign (InFile,Path);
     {$I-} Reset (InFile,1); {$I+} IORes:=IOResult;
     IF (IORes = 0) THEN
     BEGIN
          Close (InFile);
          Exit;
     END;

     CheckOK:=FALSE;

     IF (IORes = 2) OR (IORes = 3) THEN
        Log ('CheckOut: Missing output file '+Path)
     ELSE
         LogErr (IORes,'CheckOut: Error opening output '+Path);
END;


PROCEDURE CheckRemoved (Path : STRING);

VAR P      : BYTE;
    IORes  : BYTE;
    InFile : FILE;

BEGIN
     P:=Pos ('$:',Path);
     IF (P > 0) THEN
        Path:=Copy (Path,1,P-1)+WorkDrive+Copy (Path,P+2,255);

     Assign (InFile,Path);
     {$I-} Reset (InFile,1); {$I+} IORes:=IOResult;
     IF (IORes = 0) THEN
     BEGIN
          Close (InFile);
          Log ('CheckOut: Unexpectedly present: '+Path);
          CheckOK:=FALSE;
     END;
END;


PROCEDURE CompareBinTxtFiles (Path1,Path2 : STRING);

VAR Out      : STRING;
    TempFile : FILE;
    IORes    : BYTE;
    P        : BYTE;
    PrevCheckOK : BOOLEAN;

BEGIN
     P:=Pos ('$:',Path1);
     IF (P > 0) THEN
        Path1:=Copy (Path1,1,P-1)+WorkDrive+Copy (Path1,P+2,255);

     Out:=Convert_BinTxt (Path1,'temp.txt');
     IF (Out <> '') THEN
     BEGIN
          Log ('CheckOut: '+Out);
          CheckOK:=FALSE;
          Exit;
     END;

     PrevCheckOK:=CheckOK; { report only if changed }

     CompareTextFiles ('temp.txt',Path2);

     Assign (TempFile,'temp.txt');
     {$I-} Erase (TempFile); {$I+} IORes:=IOResult;

     IF (CheckOK <> PrevCheckOK) THEN
     BEGIN
          Log ('CheckOut: temp.txt was '+Path1);
          Exit;
     END;
END;



PROCEDURE ReadOutputTxt (Path : STRING);

VAR InFile : TEXT;
    IORes  : BYTE;
    LineNr : WORD;
    Regel  : STRING;
    P      : BYTE;
    Path1,
    Path2  : STRING;
    FilTyp : CHAR;

BEGIN
     Assign (InFile,Path);
     {$I-} Reset (InFile); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          LogErr (IORes,'Error opening '+Path);
          Exit;
     END;

     LineNr:=0;

     WHILE (NOT Eof (InFile)) DO
     BEGIN
          ReadLn (InFile,Regel);
          Inc (LineNr);

          IF (Regel = '') THEN
             Continue;

          IF (Regel[1] = ';') THEN
             Continue;

          Regel:=CleanTabs (Regel,1);

          IF (Regel[2] <> ' ') THEN
          BEGIN
               Log ('CheckOut: Error in "'+Regel+'"');
               Continue;
          END;

          FilTyp:=UpCase (Regel[1]);

          IF (NOT (FilTyp IN ['T','C','B','-','R'])) THEN
          BEGIN
               Log ('CheckOut: Unsupported file type in "'+Regel+'"');
               Continue;
          END;

          P:=Pos (' ',Copy (Regel,3,255));
          IF (P = 0) AND (NOT (FilTyp IN ['-','R'])) THEN
          BEGIN
               Log ('CheckOut: Error in "'+Regel+'"');
               Continue;
          END;

          Delete (Regel,1,2);
          P:=Pos (' ',Regel);

          IF (P = 0) THEN
          BEGIN
               Path1:=Regel;
               Regel:='';
               Path2:='';
          END ELSE
          BEGIN
               Path1:=Copy (Regel,1,P-1);
               Delete (Regel,1,P);
               Regel:=DeleteFrontSpaces (Regel);

               P:=Pos (' ',Regel);
               IF (P = 0) THEN
               BEGIN
                    Path2:=Regel;
                    Regel:='';
               END ELSE
               BEGIN
                    Path2:=Copy (Regel,1,P-1);
                    Delete (Regel,1,P);
                    Regel:=DeleteFrontSpaces (Regel);
               END;
          END;

          CASE FilTyp OF
               'B' :
                   BEGIN
                        StoreDiff (Regel);
                        CompareBinFiles (Path1,Path2);
                   END;

               'T':
                   CompareTextFiles (Path1,Path2);

               'C':
                   CompareBinTxtFiles (Path1,Path2);

               '-':
                   CheckExist (Path1);

               'R':
                   CheckRemoved (Path1);

          END; { case }
     END; { while }

     Close (InFile);
END;

{ main }

VAR IORes : BYTE;

BEGIN
     CheckOK:=TRUE;

     WorkDrive:=GetEnv ('WORKDRIVE');
     IF (WorkDrive = '') THEN
     BEGIN
          WriteLn (#7,'WORKDRIVE is not set');
          Halt (10);
     END;

     LogPath:=GetEnv ('WGTESTLOG');
     IF (LogPath = '') THEN
     BEGIN
          WriteLn (#7,'WGTESTLOG is not set');
          Halt (10);
     END;

     Assign (LogFile,LogPath);
     {$I-} Append (LogFile); {$I+} IORes:=IOResult;
     IF (IORes <> 0) THEN
     BEGIN
          WriteLn (#7,'Error appending to '+LogPath,' (error ',IORes,')');
          Halt (10);
     END;

     IF (ParamCount <> 1) THEN
     BEGIN
          Log ('Usage: CheckOut <path to output.txt>');
          Close (LogFile);
          Halt (10);
     END;

     ReadOutputTxt (FExpand (ParamStr (1)));

     IF CheckOK THEN
        Log ('CheckOut: Output verified OK');

     Close (LogFile);

     IF (NOT CheckOK) THEN
        Halt (10);
END.
