/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdSelfTest.h"
#include "lcmd/LCmdCopy.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdInternalViewer.h"
#include "lcmd/LCmdMainWindow.h"

#include "glib/gc/create.h"
#include "glib/gc/pointer.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/io/GFileOutputStream.h"
#include "glib/io/GFileInputStream.h"
#include "glib/vfs/GFile.h"
#include "glib/vfs/GExtendedAttributes.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GProgressBar.h"
#include "glib/util/GMath.h"
#include "glib/util/GRandom.h"
#include "glib/primitives/GAtomicCounter.h"
#include "glib/GProgram.h"

LCmdSelfTest::LCmdSelfTest ( TestID tid )
             :prg(GProgram::GetProgram()),
              tid(tid)
{
}

LCmdSelfTest::~LCmdSelfTest ()
{
}

LCmdSelfTest::AutoFileDeleter::AutoFileDeleter ( LCmdSelfTest& selfTest, const GString& path, bool isdir )
                              :selfTest(selfTest),
                               path(path), 
                               isdir(isdir)
{
}

LCmdSelfTest::AutoFileDeleter::~AutoFileDeleter () 
{
   try {
      if (isdir)
         selfTest.removeDirectory(path);
      else
         selfTest.removeFile(path);
   } catch (...) {
      // Just ignore this, since we might be called in some
      // error condition/context already.
   }
}

void LCmdSelfTest::assertTrue ( bool b, const GString& errMsg )
{
   if (!b)
   {
      GString msg;
      if (errMsg == "")
         msg = "Assert error!";
      else
         msg = errMsg;
      gthrow_(GIllegalStateException(msg));
   }
}

void LCmdSelfTest::runTheWorkerThread ( GWorkerThread& worker )
{
   GLog::FilterId oldFid = GLog::GetFilterLevel();
   GLog::SetFilterLevel(GLog::TEST);
   try {
      prg.printF("----------------------------------------------------------");
      doTestImpl(worker);
      if (worker.isStopRequested())
      {
         prg.printF("CANCEL");
      }
      else
      {
         GString msg = "OK - Everything that was tested seems to work just fine.";
         prg.printF(msg);
         currentTestDescr = "";
         worker.updateMonitor();
         worker.showWorkerMessageBox(msg, GMessageBox::TYPE_INFO);
      }
      GLog::SetFilterLevel(oldFid);
   } catch (GException& e) {
      GLog::SetFilterLevel(oldFid);
      GString estr = e.toString();
      GString stackTrace = e.getStackTrace(estr);
      GString logPath = GLog::GetLogFilePath();
      GString msg("ERROR - Self Test Failed:\n%s\n\nSee the log-file for details:\n'%s'.", GVArgs(estr).add(logPath));
      GString u2("Log-file");
      prg.printF(msg);
      GLog::PrintLn(msg);
      GLog::PrintLn(stackTrace);
      GMessageBox::Answer answ = worker.showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Od(u2)", GString::Empty, false, stackTrace, u2);
      if (answ == GMessageBox::IDUSER2)
      {
         LCmdMainWindow& mainWin = dynamic_cast<LCmdMainWindow&>(prg.getMainWindow());
         GAbstractCommand* cmd = mainWin.getCommand("cmdDebugShowLogFile");
         if (cmd != null)
            mainWin.postCommand(cmd);
      }
   } catch (...) {
      GLog::SetFilterLevel(oldFid);
      prg.printF("UNKNOWN ERROR");
   }
}

void LCmdSelfTest::setCurrentTestDescr ( GWorkerThread& worker, 
                                         const GString& descr )
{
   prg.printF("--- Test '%s'", GVArgs(descr));
   currentTestDescr = GString("Testing now: %s ...", GVArgs(descr));
   worker.updateMonitor();
}

void LCmdSelfTest::onWorkerThreadUserEvent ( GWorkerThread& worker, 
                                             GDialogPanel& monitor, 
                                             const GString& msgID, 
                                             GObject* userParam )
{
   if (msgID == "TestWorkerThread")
   {
      if (!testWorkerThread(worker, "Step 1 of 3", 0))
         return;
      if (!testWorkerThread(worker, "Step 2 of 3", 1000))
         return;
      if (!testWorkerThread(worker, "Step 3 of 3", 4000))
         return;
   }
}

void LCmdSelfTest::onWorkerThreadUpdateMonitor ( GWorkerThread& worker, 
                                                 GDialogPanel& monitor )
{
   GString txt = currentTestDescr;
   if (txt == "")
   {
      txt = "Finished!";
      monitor.setComponentDisabled("DLG_CANCEL");
   }
   monitor.setComponentText("Descr", txt);
}

void LCmdSelfTest::onWorkerThreadCommand ( GWorkerThread& worker, 
                                           GDialogPanel& monitor, 
                                           const GString& cmdID )
{
   if (cmdID == "DLG_CANCEL")
   {
      monitor.setComponentDisabled(cmdID);
      worker.requestStop();
   }
}

void LCmdSelfTest::doTestImpl ( GWorkerThread& worker )
{
   bool needPrepareAndCleanup = true;

   // ---
   // Test "Low Level APIs".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_LowLevelAPIs))
   {
      setCurrentTestDescr(worker, "Low Level APIs");
      if (tid != TID_All)
         needPrepareAndCleanup = false;

      // Integer parsing and string creation.
      GProgram::GetProgram().printF("Integer parsing and string creation...");
      GInteger maxInt = GInteger::MAX_VALUE;
      GString maxIntStr("%s", GVArgs(maxInt));
      assertTrue(maxIntStr == "2147483647");
      assertTrue(maxIntStr == maxInt.toString());
      GInteger minInt = GInteger::MIN_VALUE;
      GString minIntStr("%s", GVArgs(minInt));
      assertTrue(minIntStr == minInt.toString());
      assertTrue(minIntStr == "-2147483648");

      // String formatting.
      GProgram::GetProgram().printF("String formatting...");
      GString str1 = "Test1\b\n\r\t\\\"\001\200Test2";
      GString str2 = GTextResource::MakeTextLiteral(str1, '"');
      assertTrue(str2 == "\"Test1\\b\\n\\r\\t\\\\\\\"\\001\\200Test2\"");
      GString str3 = GTextResource::ParseTextLiteral(str2, true);
      assertTrue(str3 == str1);

      // Filename filter matching.
      GProgram::GetProgram().printF("Filename filter matching...");
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "*.txt"));
      assertTrue(!GFile::IsFileNameInFilter("File1.txt", "*.txt1"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt.log", "*.log"));
      assertTrue(!GFile::IsFileNameInFilter("File1.txt.log", "*.txt"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt.log", "*.txt*"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "*i*.?x?"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "F*e*.tx?"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "F*e*tx?"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "File1.txt"));
      assertTrue(!GFile::IsFileNameInFilter("File1.txt", "File1.txt?"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "File1.txt*"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "*File1.txt"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "*File1.txt*"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "?????.???"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "?????????"));
      assertTrue(!GFile::IsFileNameInFilter("File1.txt", "????????"));
      assertTrue(!GFile::IsFileNameInFilter("File1.txt", "?"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "*"));
      assertTrue(GFile::IsFileNameInFilter("File1.txt", "*.*"));
   }

   // ---
   // Test "Garbage Collector"
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_GarbageCollector))
   {
      setCurrentTestDescr(worker, "Garbage Collector");
      if (tid != TID_All)
         needPrepareAndCleanup = false;
      testGarbageCollector(worker);
   }

   // ---
   // Test "Multithreaded Object Synchronization"
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_ObjectSynch))
   {
      setCurrentTestDescr(worker, "Multithreaded Object Synchronization");
      if (tid != TID_All)
         needPrepareAndCleanup = false;

      class MyThread : public GThread
      {
      public:

         GAtomicCounter<int> error;
         MyThread* theOtherThread;
         GObject synchObj;
         bool finishedLoop;
         bool finishedWait;
         bool descruct;

         MyThread ( const GString name )
            :GThread(name), 
             theOtherThread(null), 
             finishedLoop(false),
             finishedWait(false),
             descruct(false)
         {
            GProgram::GetProgram().printF("Constructing thread '%s'", GVArgs(name));
         }

         virtual ~MyThread ()
         {
            const GString& name = getName();
            GProgram::GetProgram().printF("Destructing thread '%s'", GVArgs(name));
            descruct = true;
            synchObj.notifyAll();
         }

         virtual void run ()
         {
            const GString& name = getName();
            ulonglong timeStart = GSystem::CurrentTimeMillis();
            while (!descruct && GSystem::CurrentTimeMillis() - timeStart < 1500)
            {
               static GObject Obj;
               static bool IsHere = false;
               GProgram::GetProgram().printF("Thread '%s' waiting for monitor", GVArgs(name));
               GObject::Synchronizer synch(Obj);
               GProgram::GetProgram().printF("Thread '%s' entered monitor", GVArgs(name));
               if (IsHere)
               {
                  error++;
                  break;
               }
               IsHere = true;
               GThread::Sleep(500);
               IsHere = false;
            }

            finishedLoop = true;
            if (descruct)
               return;
            GProgram::GetProgram().printF("Thread '%s' waiting for notification", GVArgs(name));
            synchObj.wait();

            GProgram::GetProgram().printF("Thread '%s' has been notified", GVArgs(name));
            finishedWait = true;
            if (descruct)
               return;
         }
      };

      MyThread thread1("MyThread1");
      MyThread thread2("MyThread2");
      thread1.theOtherThread = &thread2;
      thread2.theOtherThread = &thread1;
      thread1.start();
      thread2.start();
      while (!thread1.finishedLoop || !thread2.finishedLoop)
         GThread::Sleep(50);
      assertTrue(thread1.error == 0);
      assertTrue(thread2.error == 0);
      GThread::Sleep(1500);
      assertTrue(!thread1.finishedWait);
      assertTrue(!thread2.finishedWait);
      prg.printF("Notifying thread '%s'", GVArgs(thread1.getName()));
      thread1.synchObj.notifyAll();
      prg.printF("Notifying thread '%s'", GVArgs(thread2.getName()));
      thread2.synchObj.notifyAll();
      while (thread1.isRunning() || thread2.isRunning())
         GThread::Sleep(50);
      assertTrue(thread1.finishedWait);
      assertTrue(thread2.finishedWait);
   }

   // ---
   // Test "Container Classes".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_ContainerClasses))
   {
      setCurrentTestDescr(worker, "Container Classes");
      if (tid != TID_All)
         needPrepareAndCleanup = false;
      testContainerClasses(worker);
   }

   // ---
   // Test "Worker Thread".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_WorkerThread))
   {
      setCurrentTestDescr(worker, "Worker Thread");
      if (tid != TID_All)
         needPrepareAndCleanup = false;
      // We must send a message to the GUI-thread and let the GUI-thread 
      // call our testing method. This is needed because the testing method 
      // will do some GUI-operations which is not supported by this 
      // current non-GUI thread.
      worker.sendUserMessageToMonitor("TestWorkerThread");
   }

   if (worker.isStopRequested())
      return;

   if (!needPrepareAndCleanup)
      return;

   // ---
   // Create a temporary working directory, usually in "c:/temp".
   setCurrentTestDescr(worker, "Preparing");
   tempDir = localVfs.createTemporaryFile("lcmd");
   assertTrue(localVfs.removeFile(0, tempDir, false) == GError::Ok);
   assertTrue(localVfs.createDirectory(tempDir) == GError::Ok);
   AutoFileDeleter autoDeleterForTempDir(*this, tempDir, true);
   localVfs.slash(tempDir);
   prg.printF("Created directory '%s'", GVArgs(tempDir));

   // ---
   // Test "Low Level File I/O".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_LowLevelFileIO))
   {
      setCurrentTestDescr(worker, "Low Level File I/O");
      GString dir = tempDir;
      localVfs.slash(dir);
      prg.printF("Make sure directory exist (with trailing slash): '%s'", GVArgs(dir));
      assertTrue(localVfs.existDirectory(dir));
      dir.removeLastChar();
      prg.printF("Make sure directory exist (without trailing slash): '%s'", GVArgs(dir));
      assertTrue(localVfs.existDirectory(dir));
      localVfs.slash(dir);
      GString fakeDir = dir + "fakeDir";
      prg.printF("Make sure directory does not exist: '%s'", GVArgs(fakeDir));
      assertTrue(!localVfs.existDirectory(fakeDir));
      prg.printF("Make sure directory is empty: '%s'", GVArgs(dir));
      GVfs::List filenameList;
      localVfs.fillList(filenameList, dir + "*", true, true, true, true);
      assertTrue(filenameList.size() == 0);
      prg.printF("Create a dummy temporary file in directory: '%s'", GVArgs(dir));
      GString filePath1 = dir + "dummyfile1.tmp";
      createFile(filePath1);
      AutoFileDeleter autoDeleter1(*this, filePath1, false);
      prg.printF("Make sure directory contains one single file: '%s'", GVArgs(dir));
      localVfs.fillList(filenameList, dir + "*", true, true, true, true);
      assertTrue(filenameList.size() == 1);
      localVfs.fillList(filenameList, dir + "*.tmp", true, true, true, true);
      assertTrue(filenameList.size() == 1);
      localVfs.fillList(filenameList, dir + "*.txt", true, true, true, true);
      assertTrue(filenameList.size() == 0);

      // Test file seek.
      prg.printF("Test file seek-operations on file: '%s'", GVArgs(filePath1));
      GError err;
      prg.printF("   GVfs::openFile()");
      GVfs::FileHandle hfile = localVfs.openFile(filePath1, &err, GVfs::Mode_ReadOnly, GVfs::Create_Never, GVfs::Share_DenyWrite, GVfs::OF_FLAG_RANDOM_ACCESS);
      GVfs::AutoFileHandleCloser fileCloser(localVfs, hfile);
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      assertTrue(hfile != null, "localVfs.openFile() returned null.");
      prg.printF("   GVfs::getFileSize()");
      longlong fsize = localVfs.getFileSize(hfile, &err);
      prg.printF("   GVfs::getFileSize() returned: %d", GVArgs(fsize));
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      assertTrue(hfile != null, "localVfs.getFileSize() returned 0.");
      prg.printF("   GVfs::setFileSeekPosFromEnd(hfile, 0)");
      err = localVfs.setFileSeekPosFromEnd(hfile, 0);
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      longlong currentFPos = localVfs.getFileSeekPos(hfile, &err);
      prg.printF("   GVfs::getFileSeekPos() returned: %d", GVArgs(currentFPos));
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      assertTrue(currentFPos == fsize, GString("localVfs.getFileSeekPos() returned %d, expected %d.", GVArgs(currentFPos).add(fsize)));
      prg.printF("   GVfs::setFileSeekPosFromStart(hfile, 0)");
      err = localVfs.setFileSeekPosFromStart(hfile, 0);
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      currentFPos = localVfs.getFileSeekPos(hfile, &err);
      prg.printF("   GVfs::getFileSeekPos() returned: %d", GVArgs(currentFPos));
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      assertTrue(currentFPos == 0, GString("localVfs.getFileSeekPos() returned %d, expected %d.", GVArgs(currentFPos).add(0)));
      prg.printF("   GVfs::setFileSeekPosFromEnd(hfile, -100)");
      err = localVfs.setFileSeekPosFromEnd(hfile, -100);
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      currentFPos = localVfs.getFileSeekPos(hfile, &err);
      prg.printF("   GVfs::getFileSeekPos() returned: %d", GVArgs(currentFPos));
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      assertTrue(currentFPos == fsize-100, GString("localVfs.getFileSeekPos() returned %d, expected %d.", GVArgs(currentFPos).add(fsize-100)));
      prg.printF("   GVfs::setFileSeekPosFromCurrent(hfile, 50)");
      err = localVfs.setFileSeekPosFromCurrent(hfile, 50);
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      currentFPos = localVfs.getFileSeekPos(hfile, &err);
      prg.printF("   GVfs::getFileSeekPos() returned: %d", GVArgs(currentFPos));
      assertTrue(err == GError::Ok, GString(GSystem::GetApiRetString(err)));
      assertTrue(currentFPos == fsize-50, GString("localVfs.getFileSeekPos() returned %d, expected %d.", GVArgs(currentFPos).add(fsize-50)));

      // Test various walking-paths.
      // #1
      GString srcPath1;
      assertTrue(localVfs.walkPath(srcPath1, ""));
      assertTrue(srcPath1 == "");
      // #2
      srcPath1 = "";
      assertTrue(localVfs.walkPath(srcPath1, "."));
      assertTrue(srcPath1 == "");
      // #3
      srcPath1 = "";
      assertTrue(!localVfs.walkPath(srcPath1, ".."));
      assertTrue(srcPath1 == "");
      // #4
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, "./././."));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TOP/SUB");
      // #5
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, "C:/TEST/SUB2"));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TEST/SUB2");
      // #6
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, "/TEST/SUB2"));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TEST/SUB2");
      // #7
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, "../SUB2"));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TOP/SUB2");
      // #8
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, ".."));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TOP");
      // #9
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, "../.."));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/");
      // #10
      srcPath1 = "C:/TOP/SUB";
      assertTrue(!localVfs.walkPath(srcPath1, "../../.."));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/");
      // #11
      srcPath1 = "C:/TOP/SUB";
      assertTrue(localVfs.walkPath(srcPath1, "./../SUB/../../TOP/SUB/DEEP/.."));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TOP/SUB");
      // #12
      srcPath1 = "C:/TOP";
      assertTrue(localVfs.walkPath(srcPath1, "SUB/"));
      srcPath1.replaceAll('\\', '/'); // Make us see foreward slashes only.
      assertTrue(srcPath1 == "C:/TOP/SUB/"); // Should be slashed since input was slahed.
   }

   // ---
   // Test "Text Viewer"
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_TextViewer))
   {
      setCurrentTestDescr(worker, "Text Viewer");
      GString textFilePath = tempDir + "textfile.txt";
      GFileOutputStream fout(localVfs, textFilePath, true, false, true);
      AutoFileDeleter autoDeleterForTextFilePath(*this, textFilePath, false);
      fout.printf("This is a text file used to test the Text Viewer class of Larsen Commander.\n"
         "Next is a line to clearly show each tab position:\n"
         "|-------|-------|-------|-------|-------|-------|-------|-------|-------|-------|-------|\n"
         "And\there\tis\tone\tword\tfor\teach\ttab\tposition,\tfor\ttest.\n"
         "Word1 WORD1 Word2 WORD2 LongWord1 LongWord2 LONGWORD1 LONGWORD2\n");
      fout.close();
      GFile f(textFilePath);
      LCmdInternalViewer lcmdViewer(f);
      GTextViewer& viewer = lcmdViewer.getTextViewer();
      viewer.setWordWrap(false);
      assertTrue(viewer.getLinesCount() == 5);
      GTextViewer::SearchParams searchParams;
      LCmdDlgTextSearch& searchParamsEditor = lcmdViewer.getSearchParamsEditor();
      searchParams.searchString = "clearly";
      searchParams.searchForward = true;
      searchParams.searchCaseSen = true;
      searchParams.searchWord = true;
      searchParamsEditor.setFixedAutoParams(searchParams);
      viewer.cmdSearch();
      assertTrue(viewer.getCurrentPosY() == 1);
      assertTrue(viewer.getCurrentPosX() == 25);
      searchParams.searchString = "is is";
      searchParams.searchForward = false;
      searchParams.searchWord = false;
      searchParamsEditor.setFixedAutoParams(searchParams);
      viewer.unselectAll();
      viewer.cmdSearch();
      assertTrue(viewer.getCurrentPosY() == 0);
      assertTrue(viewer.getCurrentPosX() == 2);
      viewer.cmdCopy();
      GString pastedText = GSystem::GetClipboardText();
      assertTrue(pastedText == "is is");
      viewer.unselectAll();
      assertTrue(!viewer.isAnySelectedText());
      viewer.gotoPos(3, 5);
      assertTrue(viewer.getCurrentPosY() == 3);
      assertTrue(viewer.getCurrentPosX() == 5);

      // Test "search forward".
      GString selectedText;
      searchParams.searchForward = true;
      do {
         // Test "case sensitive".
         searchParams.searchCaseSen = true;
         do {
            // Test "not whole words only".
            searchParams.searchString = "Word1";
            searchParams.searchWord = false;
            do {
               viewer.gotoPos(0, 0);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 5);
               assertTrue(pos1.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos2 = viewer.getCurrentPos();
               assertTrue(pos2.x == 33);
               assertTrue(pos2.y == 4);
            } while (false);
            // Test "whole words only", #1.
            searchParams.searchString = "Word1";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(0, 0);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 5);
               assertTrue(pos1.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
            // Test "whole words only", #2.
            searchParams.searchString = "Word2 WORD2";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(0, 0);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 23);
               assertTrue(pos1.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
         } while (false);
         // Test "case insensitive".
         searchParams.searchCaseSen = false;
         do {
            // Test "not whole words only".
            searchParams.searchString = "Word1";
            searchParams.searchWord = false;
            do {
               viewer.gotoPos(0, 0);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 5);
               assertTrue(pos1.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos2 = viewer.getCurrentPos();
               assertTrue(pos2.x == 11);
               assertTrue(pos2.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos3 = viewer.getCurrentPos();
               assertTrue(pos3.x == 33);
               assertTrue(pos3.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos4 = viewer.getCurrentPos();
               assertTrue(pos4.x == 53);
               assertTrue(pos4.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
            // Test "whole words only", #1.
            searchParams.searchString = "Word1";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(0, 0);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 5);
               assertTrue(pos1.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos2 = viewer.getCurrentPos();
               assertTrue(pos2.x == 11);
               assertTrue(pos2.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
            // Test "whole words only", #2.
            searchParams.searchString = "word2 word2";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(0, 0);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 23);
               assertTrue(pos1.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
         } while (false);
      } while (false);

      // Test "search backward".
      searchParams.searchForward = false;
      do {
         // Test "case sensitive".
         searchParams.searchCaseSen = true;
         do {
            // Test "not whole words only".
            searchParams.searchString = "Word1";
            searchParams.searchWord = false;
            do {
               viewer.gotoPos(GInteger::MAX_VALUE, GInteger::MAX_VALUE);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 28);
               assertTrue(pos1.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos2 = viewer.getCurrentPos();
               assertTrue(pos2.x == 0);
               assertTrue(pos2.y == 4);
            } while (false);
            // Test "whole words only", #1.
            searchParams.searchString = "Word1";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(GInteger::MAX_VALUE, GInteger::MAX_VALUE);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 0);
               assertTrue(pos1.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
            // Test "whole words only", #2.
            searchParams.searchString = "Word2 WORD2";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(GInteger::MAX_VALUE, GInteger::MAX_VALUE);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 12);
               assertTrue(pos1.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
         } while (false);
         // Test "case insensitive".
         searchParams.searchCaseSen = false;
         do {
            // Test "not whole words only".
            searchParams.searchString = "Word1";
            searchParams.searchWord = false;
            do {
               viewer.gotoPos(GInteger::MAX_VALUE, GInteger::MAX_VALUE);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 48);
               assertTrue(pos1.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos2 = viewer.getCurrentPos();
               assertTrue(pos2.x == 28);
               assertTrue(pos2.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos3 = viewer.getCurrentPos();
               assertTrue(pos3.x == 6);
               assertTrue(pos3.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos4 = viewer.getCurrentPos();
               assertTrue(pos4.x == 0);
               assertTrue(pos4.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
            // Test "whole words only", #1.
            searchParams.searchString = "Word1";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(GInteger::MAX_VALUE, GInteger::MAX_VALUE);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 6);
               assertTrue(pos1.y == 4);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos2 = viewer.getCurrentPos();
               assertTrue(pos2.x == 0);
               assertTrue(pos2.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
            // Test "whole words only", #2.
            searchParams.searchString = "word2 word2";
            searchParams.searchWord = true;
            do {
               viewer.gotoPos(GInteger::MAX_VALUE, GInteger::MAX_VALUE);
               searchParamsEditor.setFixedAutoParams(searchParams);
               assertTrue(viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
               GTextViewer::Pos pos1 = viewer.getCurrentPos();
               assertTrue(pos1.x == 12);
               assertTrue(pos1.y == 4);
               assertTrue(!viewer.searchNextImpl());
               assertTrue(viewer.getSelectedText(selectedText).equalsString(searchParams.searchString, !searchParams.searchCaseSen));
            } while (false);
         } while (false);
      } while (false);
   }

   // ---
   // Create a dummy file containing some dummy/testable data.
   GString path1 = tempDir + "file1.txt";
   createFile(path1);
   AutoFileDeleter autoDeleterForPath1(*this, path1, false);
   addExtendedAttributes(path1);
   verifyExtendedAttributes(path1);
   verifyFile(path1);

   // ---
   // Test "Copy Single File".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_CopySingleFile))
   {
      setCurrentTestDescr(worker, "Copy Single File");
      GString path2 = tempDir + "file2.txt";
      copyFile(path1, path2);
      compareContentOfFiles(path1, path2);
      compareAttributesOfFiles(path1, path2);
      removeFile(path2);
   }

   // ---
   // Test "Move Single File".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_MoveSingleFile))
   {
      setCurrentTestDescr(worker, "Move Single File");
      GString temp = tempDir + "SomeOtherFileName.txt";
      moveFile(path1, temp);
      assertTrue(!localVfs.existFile(path1));
      assertTrue(localVfs.existFile(temp));
      moveFile(temp, path1); // Move back to original filename "path2".
      verifyFile(path1);
   }

   // ---
   // Test "Change Filename Case".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_ChangeFilenameCase))
   {
      setCurrentTestDescr(worker, "Change Filename Case");
      GFile f(path1);

      // Change to UPPERCASE.
      GString fnameUpper = f.getFileName();
      fnameUpper.toUpperCase();
      f.setFileName(fnameUpper);
      localVfs.moveOrRenameFile(path1, f.getFullPath(), false);
      GFileItem fiUpper(localVfs, path1);
      assertTrue(fiUpper.getFileName() == fnameUpper);

      // Change to MixedCase.
      GString fnameMixed = fnameUpper;
      fnameMixed.toLowerCase();
      fnameMixed.setCharAt(0, GCharacter::ToUpperCase(fnameMixed[0]));
      f.setFileName(fnameMixed);
      localVfs.moveOrRenameFile(path1, f.getFullPath(), false);
      GFileItem fiMixed(localVfs, path1);
      assertTrue(fiMixed.getFileName() == fnameMixed);

      // Change to lowercase.
      GString fnameLower = fnameUpper;
      fnameMixed.toLowerCase();
      f.setFileName(fnameLower);
      localVfs.moveOrRenameFile(path1, f.getFullPath(), false);
      GFileItem fiLower(localVfs, path1);
      assertTrue(fiLower.getFileName() == fnameLower);
   }

   // ---
   // Test "Change File Attributes".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_ChangeFileAttributes))
   {
      setCurrentTestDescr(worker, "Change File Attributes");
      GFileItem fi(localVfs, path1);
      fi.setArchiveFlag(false);
      fi.setHiddenFlag(false);
      fi.setSystemFlag(false);
      fi.setReadOnlyFlag(false);
      localVfs.writeAttrAndTimes(null, fi, path1);

      // Check that all flags are off.
      GError err;
      int attr = localVfs.getFileAttributes(path1, &err);
      assertTrue(err == GError::Ok);
      assertTrue((attr & GVfs::FAttrArchive) == 0);
      assertTrue((attr & GVfs::FAttrHidden) == 0);
      assertTrue((attr & GVfs::FAttrSystem) == 0);
      assertTrue((attr & GVfs::FAttrReadOnly) == 0);

      // Check setting flag "read only".
      fi.setReadOnlyFlag(true);
      localVfs.writeAttrAndTimes(null, fi, path1);
      attr = localVfs.getFileAttributes(path1, &err);
      assertTrue(err == GError::Ok);
      assertTrue((attr & GVfs::FAttrArchive) == 0);
      assertTrue((attr & GVfs::FAttrHidden) == 0);
      assertTrue((attr & GVfs::FAttrSystem) == 0);
      assertTrue((attr & GVfs::FAttrReadOnly) != 0);

      // Check clearing flag "read only".
      fi.setReadOnlyFlag(false);
      localVfs.writeAttrAndTimes(null, fi, path1);
      attr = localVfs.getFileAttributes(path1, &err);
      assertTrue(err == GError::Ok);
      assertTrue((attr & GVfs::FAttrArchive) == 0);
      assertTrue((attr & GVfs::FAttrHidden) == 0);
      assertTrue((attr & GVfs::FAttrSystem) == 0);
      assertTrue((attr & GVfs::FAttrReadOnly) == 0);
   }


   // ---
   // Test "Change File Time".
   if (!worker.isStopRequested() &&
      (tid == TID_All || tid == TID_ChangeFileTime))
   {
      setCurrentTestDescr(worker, "Change File Time");
      const GDate newDate(1980, 6, 1);
      const GTime newTime(18, 30, 00);
      const GTimeStamp newTimeStamp(newDate, newTime);
      GFileItem fi1(localVfs, path1);
      fi1.timeCreate = newTimeStamp;
      localVfs.writeAttrAndTimes(null, fi1, path1);
      GFileItem fi2(localVfs, path1);
      prg.printF("Time stamp file1=%s, file2=%s", GVArgs(fi2.timeCreate.toString()).add(fi1.timeCreate.toString()));
      assertTrue(fi2.timeCreate == fi1.timeCreate);
   }

   // ---
   setCurrentTestDescr(worker, "Cleanup");
}

void LCmdSelfTest::verifyExtendedAttributes ( const GString& path )
{
   GExtendedAttributes eas1(path);
   assertTrue(eas1.getEACount() == 3);
   GString val1 = eas1.getEAString("EA1");
   assertTrue(val1 == "Test value 1");
   GString val2 = eas1.getEAString("EA2");
   assertTrue(val2 == "Test value 2");
   GString val3 = eas1.getEAString("EA3");
   assertTrue(val3 == "Test value 3");
   prg.printF("Verifyed Extended Attributes in file '%s'", GVArgs(path));
}

void LCmdSelfTest::addExtendedAttributes ( const GString& path )
{
   GExtendedAttributes eas1(path);
   eas1.setEAString("EA1", "Test value 1");
   eas1.setEAString("EA2", "Test value 2");
   eas1.setEAString("EA3", "Test value 3");
   eas1.setEAString("EA4", "Test value 4");
   eas1.removeEA("EA4");
   assertTrue(eas1.writeEAs(path) == NO_ERROR);
   prg.printF("Added some Extended Attributes to file '%s'", GVArgs(path));
}

void LCmdSelfTest::compareAttributesOfFiles ( const GString& path1, const GString& path2 )
{
   GError err = NO_ERROR;
   int attr1 = localVfs.getFileAttributes(path1, &err);
   assertTrue(err == GError::Ok);
   int attr2 = localVfs.getFileAttributes(path2, &err);
   assertTrue(err == GError::Ok);
   assertTrue(attr1 == attr2);
   GFileItem fi1(localVfs, path1);
   GFileItem fi2(localVfs, path2);
   assertTrue(fi1.timeCreate == fi2.timeCreate);
   GExtendedAttributes ea1(path1);
   GExtendedAttributes ea2(path2);
   assertTrue(ea1 == ea2);
   prg.printF("Compared file time and attributes of file '%s' and '%s'", GVArgs(path1).add(path2));
}

void LCmdSelfTest::compareContentOfFiles ( const GString& path1, const GString& path2 )
{
   assertTrue(GFile::FilesEquals(localVfs, path1, localVfs, path2));
   prg.printF("Compared content of file '%s' and '%s'", GVArgs(path1).add(path2));
}


void LCmdSelfTest::moveFile ( const GString& path1, const GString& path2 )
{
   bool move = true;
   copyOrMoveFile(path1, path2, move);
   prg.printF("Moved file '%s' to '%s'", GVArgs(path1).add(path2));
}

void LCmdSelfTest::copyFile ( const GString& path1, const GString& path2 )
{
   bool move = false;
   copyOrMoveFile(path1, path2, move);
   prg.printF("Copied file '%s' to '%s'", GVArgs(path1).add(path2));
}

void LCmdSelfTest::copyOrMoveFile ( const GString& path1, const GString& path2, bool move )
{
   GArray<LCmdCopyFileItem> items;
   items.add(new LCmdCopyFileItem(path1, &path2));
   LCmdFilePanel& currentPanel = LCmdFilePanel::GetCurrentPanel();
   GVfs& vfsSrc = localVfs;
   GVfs& vfsDst = localVfs;
   int countFilesSkipped = 0;
   assertTrue(LCmdCopy::CopyOrMoveFiles(vfsSrc, vfsDst,
                                    items, move, currentPanel,
                                    false, // fpanelIsSrc,
                                    false, // renameOnly
                                    false, // removeEAName
                                    GString::Empty, // autoSelect1
                                    GString::Empty, // autoSelect2
                                    &countFilesSkipped,
                                    false)); // refillPanel
   assertTrue(countFilesSkipped == 0);
}

void LCmdSelfTest::removeFile ( const GString& path )
{
   assertTrue(localVfs.removeFile(0, path, false) == GError::Ok);
   prg.printF("Removed file '%s'", GVArgs(path));
}

void LCmdSelfTest::removeDirectory ( const GString& dir )
{
   GError rc = localVfs.removeDirectory(0, dir, false);
   if (rc != GError::Ok)
      prg.printF("Error removing directory '%s': %s", GVArgs(dir).add(rc.getErrorMessage()));
   assertTrue(rc == GError::Ok);
   prg.printF("Removed directory '%s'", GVArgs(dir));
}

void LCmdSelfTest::createFile ( const GString& path )
{
   const bool textMode = true;
   GFileOutputStream os(localVfs, path, true, false, textMode);
   for (int i=1; i<100; i++)
   {
      for (int ii=0; ii<i; ii++)
      {
         os.printf("%02d", GVArgs(i));
         if (ii < i - 1) // If not the last "ii" in loop.
            os.printf(" ", GVArgs(i));
         else
            os.printf("\n");
      }
   }
   prg.printF("Created file '%s'", GVArgs(path));
}

void LCmdSelfTest::verifyFile ( const GString& path )
{
   GString line;
   const bool textMode = true;
   GFileInputStream is(localVfs, path, textMode);
   for (int i=1; i<100; i++)
   {
      is.readString(line);
      assertTrue(!line.endsWith("\r\n"));
      assertTrue(line.endsWith('\n'));
      line.stripTrailingEol();
      assertTrue(line.length() == i*3-1);
      GString str = line.substring(0, 2);
      int val = GInteger::ParseInt(str);
      assertTrue(val == i);
   }
   prg.printF("Veryfied content of file '%s'", GVArgs(path));
}

/**
 * Code to test the Garbage Collector.
 *
 * @author  Leif Erik Larsen
 * @since   2005.01.26
 */
class LCmdSelfTestGC : public GThread
{
public:

   GProgram& prg;
   GWorkerThread& worker;
   LCmdSelfTest& tester;
   const int iters;
   const int maxMillis;
   GAtomicCounter<int> countA;

   class A : public GObject
   {
   public:

      LCmdSelfTestGC& gcTest;
      bool alive;

      A ( LCmdSelfTestGC& gcTest ) : gcTest(gcTest)
      {
         alive = true;
         gcTest.countA++;
      }

      virtual ~A () 
      {
         alive = true;
         gcTest.countA--;
      }
   };

   class B : public A
   {
   public:
      p<GObject> obj;

      B ( LCmdSelfTestGC& gcTest ) : A(gcTest)
      {
         obj = CREATE GObject();
      }

      virtual ~B () 
      {
      }
   };

   class C: public B
   {
   public:

      C ( LCmdSelfTestGC& gcTest ) : B(gcTest)
      {
      }

      virtual ~C () 
      {
      }
   };

   /** A class with circular reference to class E. */
   class D : public C
   {
   public:
      p<GObject> e;

      D ( LCmdSelfTestGC& gcTest ) : C(gcTest)
      {
      }

      virtual ~D () 
      {
      }
   };

   /** A class with circular reference to class E. */
   class E : public C
   {
   public:
      p<GObject> d;

      E ( LCmdSelfTestGC& gcTest ) : C(gcTest)
      {
      }

      virtual ~E () 
      {
      }
   };

   LCmdSelfTestGC ( const GString& name, 
                    GWorkerThread& worker,
                    LCmdSelfTest& tester,
                    GProgram& prg, 
                    int iters, 
                    int maxMillis )
      :GThread(name), 
       prg(prg), 
       worker(worker),
       tester(tester),
       iters(iters), 
       maxMillis(maxMillis), 
       countA(0)
   {
      start();
   }

   virtual ~LCmdSelfTestGC () 
   {
      if (!worker.isStopRequested())
         tester.assertTrue(countA == 0);
   }
   
   virtual void run () 
   {
      for (int i=1; i<=iters; i++)
      {
         prg.printF("Thread '%s' starting iteration %d of %d", GVArgs(getName()).add(i).add(iters));
         ulonglong millisStart = GSystem::CurrentTimeMillis();
         for (;;)
         {
            ulonglong millis = GSystem::CurrentTimeMillis() - millisStart;
            if (millis >= maxMillis || worker.isStopRequested())
               break;
            B bb(*this);
            p<A> a = CREATE A(*this);
            p<B> b = CREATE B(*this);
            p<C> c = CREATE C(*this);
            p<D> d = CREATE D(*this);
            p<E> e = CREATE E(*this);
            d->e = e;
            e->d = d;
         }
      }
   }
};

void LCmdSelfTest::testGarbageCollector ( GWorkerThread& worker )
{
   GC::CollectObjects(true);
   int objCount1 = GC::GetNoOfObjects();
   prg.printF("Number of objects in the GC-pool before GC-test starts: %d", GVArgs(objCount1));

   LCmdSelfTestGC thr1("THR1", worker, *this, prg, 5, 1500);
   LCmdSelfTestGC thr2("THR2", worker, *this, prg, 5, 1600);
   LCmdSelfTestGC thr3("THR3", worker, *this, prg, 5, 1700);
   LCmdSelfTestGC thr4("THR4", worker, *this, prg, 5, 1800);
   LCmdSelfTestGC thr5("THR5", worker, *this, prg, 5, 1900);
   for (;;)
   {
      GThread::Sleep(1000);
      int objCountNow = GC::GetNoOfObjects();
      prg.printF("Number of objects in the GC-pool is now %d.", GVArgs(objCountNow));
      if (thr1.isRunning())
         continue;
      if (thr2.isRunning())
         continue;
      if (thr3.isRunning())
         continue;
      if (thr4.isRunning())
         continue;
      if (thr5.isRunning())
         continue;
      break;
   }

   GC::CollectObjects(true);
   int objCount2 = GC::GetNoOfObjects();
   if (worker.isStopRequested())
      return;

   prg.printF("Number of objects in the GC-pool after collecting objects: %d", GVArgs(objCount2));
   assertTrue(objCount1 == objCount2);
   prg.printF("The test appears to have succeeded.");
}

class MyWorkerThreadTester : public GWorkerThreadAdapter
{
public:

   LCmdSelfTest& selfTest;
   GString titleText;
   int stayInvisibleMillis;
   GAtomicCounter<int> stepCount;
   int stepMax;
   int stepSleepMillis;
   bool cancelledByUser;
   ulonglong timeStart;

   MyWorkerThreadTester ( LCmdSelfTest& selfTest, const GString& titleText, int stayInvisibleMillis ) 
      : selfTest(selfTest), titleText(titleText), stayInvisibleMillis(stayInvisibleMillis)
   {
      stepCount = 0;
      stepMax = 100;
      stepSleepMillis = 50;
      cancelledByUser = false;
      timeStart = GSystem::CurrentTimeMillis();
   }

   virtual ~MyWorkerThreadTester ()
   {
   }

   virtual void runTheWorkerThread ( GWorkerThread& worker )
   {
      bool messageBoxHasBeenShown = false;
      while (!worker.isStopRequested())
      {
         bool monitorIsVisible = worker.getMonitorDialog()->getOwnerFrame().isVisible();
         ulonglong timeDiff = GSystem::CurrentTimeMillis() - timeStart;
         if (timeDiff < GMath::Max(0, stayInvisibleMillis - 100)) // Subtract 100 ms, since PC timer might be inaccurate.
            selfTest.assertTrue(!monitorIsVisible, "Monitor expected to be invisible!");
         // ---
         ulonglong timeUsedForMessageBoxWaiting = worker.getTimeMillisUsedWaitingForUserAnswers();
         if (messageBoxHasBeenShown)
            selfTest.assertTrue(timeUsedForMessageBoxWaiting > 0);
         else
            selfTest.assertTrue(timeUsedForMessageBoxWaiting == 0);
         // Do some work, but periodically check and respect if we 
         // are requested to cancel.
         GThread::Sleep(stepSleepMillis);  
         if (worker.isStopRequested())
            break; // We are requested by user to cancel!
         if (stepCount++ >= stepMax)
            break; // Ok, we are finished!
         if (stepCount == stepMax/2) // Show message after about 1/2 of the progress time.
         {
            // The message box will automatically answer "Yes" after 5 seconds.
            messageBoxHasBeenShown = true;
            GString txt("Do you want to continue %s?", GVArgs(titleText));
            ulonglong timeStartMsgBox = GSystem::CurrentTimeMillis();
            const int timeoutSec = 2; // Timeout message-box after two seconds.
            GString flags("Yn2T%d", GVArgs(timeoutSec));
            GMessageBox::Answer answ = worker.showWorkerMessageBox(txt, GMessageBox::TYPE_QUESTION, flags);
            ulonglong timeDiffStartMsgBox = GSystem::CurrentTimeMillis() - timeStartMsgBox;
            ulonglong timeDiffReported = worker.getTimeMillisUsedWaitingForUserAnswers() - timeUsedForMessageBoxWaiting;
            if (answ == GMessageBox::IDTIMEOUT)
            {
               selfTest.assertTrue(timeDiffStartMsgBox >= timeoutSec*1000);
               selfTest.assertTrue(timeDiffReported >= timeoutSec*1000);
               answ = GMessageBox::IDYES; // Use default answer=Yes if timeout occured.
            }
            else
            {
               selfTest.assertTrue(timeDiffStartMsgBox < timeoutSec*1000);
               selfTest.assertTrue(timeDiffReported < timeoutSec*1000);
            }
            if (answ != GMessageBox::IDYES)
               break;
         }
      }
   }

   virtual void onWorkerThreadInitDialog ( GWorkerThread& worker, GDialogPanel& monitor )
   {
      monitor.setTitleText(titleText);
      GWindow& comp = monitor.getComponentByID("PROGRESSBAR");
      GProgressBar& p = dynamic_cast<GProgressBar&>(comp);
      p.setMinValue(0);
      p.setMaxValue(stepMax);
      p.setCurrentValue(stepCount);
   }

   virtual void onWorkerThreadUpdateMonitor ( GWorkerThread& worker, GDialogPanel& monitor )
   {
      monitor.setComponentValue("PROGRESSBAR", stepCount);
   }

   virtual void onWorkerThreadCommand ( GWorkerThread& worker, 
                                        GDialogPanel& monitor, 
                                        const GString& cmdID )
   {
      if (cmdID == "DLG_CANCEL")
      {
         monitor.setComponentEnabled(cmdID, false);
         worker.requestStop(false);
         cancelledByUser = true;
      }
   }
};

bool LCmdSelfTest::testWorkerThread ( GWorkerThread& mainWorker, const GString& titleText, int stayInvisibleMillis )
{
   prg.printF(titleText);
   MyWorkerThreadTester adapter(*this, titleText, stayInvisibleMillis);
   GWorkerThread wt("DlgWorkerThreadTester", 50, &adapter, false);
   GWindow& parentWin = mainWorker.getMonitorDialog()->getOwnerFrame();
   if (wt.workModal(parentWin, adapter.stayInvisibleMillis))
   {
      assertTrue(!adapter.cancelledByUser);
      return true;
   }
   if (adapter.cancelledByUser)
      return false;
   assertTrue(!wt.isSuccess(), "Worker should report it self to be unsuccessful!");
   assertTrue(false, wt.getWorkerThreadExceptionMessage());
   return true;
}

class MyHashtableValue : public GObject
{
public:
   GString val;
   int& instanceCounter;
   MyHashtableValue ( const GString& val, int& instanceCounter ) 
      :val(val), instanceCounter(instanceCounter) 
   {
      instanceCounter++;
   }
   virtual ~MyHashtableValue ()
   {
      instanceCounter--;
   }
};

void LCmdSelfTest::testContainerClasses ( GWorkerThread& worker )
{
   const int seedCount = 200;
   for (int seed=1; seed<=seedCount; seed++)
   {
      prg.printF("Testing GHashtable with seed %d/%d...", GVArgs(seed).add(seedCount));
      GRandom random(seed);
      int myValueInstanceCounter = 0;
      const int hashtableCount = (seed+1) * 40;
      int uniqueKeysCounter = 0;
      GArray<GInteger> uniqueKeys;
      GHashtable<GInteger, MyHashtableValue> hashTable(250, 0.75F);
      for (int i=0; i<hashtableCount; i++)
      {
         // Find another random integer that is not already contained in 
         // the hashtable. Use a small key-range in order to stress the 
         // bucket-algorithm used by the hashtable. 255 means a rather 
         // small key-range, so hashtable buckets might be large.
         GInteger key = random.nextInt(255);
         if (!hashTable.containsKey(key))
         {
            uniqueKeysCounter++;
            uniqueKeys.add(new GInteger(key));
         }
         MyHashtableValue* value = new MyHashtableValue(key.toString(), myValueInstanceCounter);
         hashTable.put(key, value, true);
         assertTrue(hashTable.size() == uniqueKeysCounter);
         assertTrue(hashTable.get(key) == value);
      }
      assertTrue(hashTable.size() == uniqueKeysCounter);
      assertTrue(myValueInstanceCounter == uniqueKeysCounter);
      while (hashTable.size() > 0)
      {
         GHashtable<GInteger, MyHashtableValue>::KeyEnumerator enkey = hashTable.getKeyEnumerator();
         GHashtable<GInteger, MyHashtableValue>::ValueEnumerator enval = hashTable.getValueEnumerator();
         int enumCounter = 0;
         while (enval.hasMoreElements())
         {
            MyHashtableValue* nextValue = enval.nextElement();
            assertTrue(nextValue != null);
            GInteger* nextKey = enkey.nextElement();
            MyHashtableValue* val = hashTable.get(*nextKey);
            assertTrue(val != null);
            assertTrue(val == nextValue);
            enumCounter++;
         }
         assertTrue(!enval.hasMoreElements());
         assertTrue(enumCounter == hashTable.size());
         int startSize = hashTable.size();
         int removeCount = startSize / 2;
         if (removeCount == 0)
            removeCount = 1;
         for (int i=0; i<removeCount; i++)
         {
            // Remove at random position. Get the next position to remove.
            int removeIdx = random.nextInt(uniqueKeys.getCount());
            GInteger key = uniqueKeys.get(removeIdx);
            uniqueKeys.remove(removeIdx);
            assertTrue(hashTable.remove(key));
            assertTrue(hashTable.size() == uniqueKeys.getCount());
         }
         assertTrue(uniqueKeys.getCount() == startSize - removeCount);
         assertTrue(hashTable.size() == startSize - removeCount);
         assertTrue(myValueInstanceCounter == startSize - removeCount);
      }
      assertTrue(hashTable.isEmpty());
      assertTrue(uniqueKeys.isEmpty());
      assertTrue(myValueInstanceCounter == 0);
   }
   prg.printF("OK.");
}
