/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_PROCESSLAUNCHER
#define __LCMD_PROCESSLAUNCHER

#include "glib/util/GProcessLauncher.h"

/**
 * This class is used by Larsen Commander to launch any process, regardless 
 * if it is a child process or not. The process is always launched by a 
 * secondary thread, which is implemented by the {@link #run} method if 
 * this class. This class is also used to launch system dependent shell 
 * objects, such as e.g. OS/2 Workplace Shell Objects and 
 * Windows Explorer Icons.
 *
 * It is nice to have a separate thread doing this. Then we can always have 
 * the thread wait synchronously for the child process to finish and the 
 * main thread can just test if the child process has finished by testing
 * the status of the secondary thread.
 *
 * When launching a child process the method {@link #run} will not 
 * return until the child process has finished. All standard 
 * output (stdout & stderr) from the child process is redirected to 
 * a temporary pipe that lives only while the child process is running.
 * The data that is written to the pipe (by the child process) are 
 * constantly read by the thread that launched the child process, while 
 * waiting for the child process to finish and exit. The data that is 
 * read from that pipe will be appended to the console monitor window 
 * of Larsen Commander on the fly. All this pipe redirection performance 
 * is automatically done by this class.
 *
 * @author  Leif Erik Larsen
 * @since   1998.10.07
 */
class LCmdProcessLauncher : public GProcessLauncher
{
   private:

      class LCmdCmdLineEntry& owner;

      bool finished;

      bool isInternalCmd;

      /** True if the command is to be launched by WinOpenObject() if not a program file. */
      bool runAsObject;

      /** True if we must execute command via COMSPEC. */
      bool forceCmd;

      /** True if we shall print PID-information to the console when child process has started and ended. */
      bool printPIDInfo;

      /** The original command string, as of before it was parsed. */
      GString origCommand;

      /** True when {@link #detachChildProg} has been called. */
      volatile bool detachRequested;

   public:

      LCmdProcessLauncher ( class LCmdCmdLineEntry& owner, 
                            const GString& origCommand, 
                            const GString& workingDir, 
                            const GString& prgName, 
                            const GString& paramStr, 
                            bool forceNewSession, 
                            bool isInternalCmd, 
                            bool runAsObject, 
                            bool closeOnExit, 
                            bool forceRunViaShell );

      virtual ~LCmdProcessLauncher ();

   private:

      /** Disable the copy constructor. */
      LCmdProcessLauncher ( const LCmdProcessLauncher& src ) 
         : GProcessLauncher(GString::Empty, GString::Empty), owner(src.owner) {}

      /** Disable the assignment operator. */
      LCmdProcessLauncher& operator= ( const LCmdProcessLauncher& ) { return *this; }

   public:

      const GString& getOriginalCommand () const;

      bool hasFinished () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2004.09.16
       */
      void detachChildProg ();

   protected:

      /**
       * Adjust the specified directory with respect to any aliases that can
       * possibly affect the directory. If there are no aliases that affects
       * the directory then we will simply return a copy of the specified
       * directory.
       */
      virtual GString getWorkingDir ( const GString& defaultDir ) const;

      /**
       * The run() method of our super class has found the program to be an
       * existing program path on the system.
       *
       * Now it is up to us to decide if the program should be forced to run as
       * a new session or not. We will requeste it to run as a new session if
       * the program name is listed in the set of Separate Session Apps (SSA).
       */
      virtual void programPathHasBeenValidated ( const GString& prgName, 
                                                 const GString& paramStr );

      /**
       * This method is called by our super class when it has successfully
       * launched a new program as a child process.
       */
      virtual void processHasBeenLaunched ( bool asChild );

      /**
       * This method is called by our super class when a child process
       * has finished.
       */
      virtual void processHasFinished ( bool asChild, 
                                        int result, 
                                        bool normalExit );

      /**
       * This method is automatically called by {@link #run} if and when the 
       * process has been launched successfully and it was launched as a 
       * child process. 
       *
       * We will loop to read the stdout & stderr of the child process
       * and append it to our console monitor window while waiting for 
       * the child process to finish and exit.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      virtual void waitForTheChildProcessToFinish ();

   private:

      /**
       * Will start the process and (if it is a child process of ours-) 
       * redirect its stdout/stderr to the console monitor while waiting 
       * for the child process to finish and exit.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      virtual void run ();

      /**
       * To be called by run() only.
       */
      void runAsInternalCommand ();

      /**
       * To be called by run() only.
       */
      void runAsExternalCommand ();

      /**
       * Method used to print a sequence of repeating characters.
       */
      void printLineOfCharacters ( char chr, int num, bool lineFeed = true );

      /**
       * Print command line help to the console monitor.
       */
      void printHelp ();

      /*
       * Implementation of the internal command "alias", which will 
       * print the currently defined aliases to the console monitor.
       */
      void internalCmd_Alias ();

      /*
       * Implementation of the internal command "finddup", which will 
       * lookup and print duplicates files in some %path% variable.
       */
      void internalCmd_FindDup ();

      /**
       * Implementation of the internal command "?", which prints 
       * some short help about the various internal commands to 
       * the console monitor.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      void internalCmd_Help ();

      /**
       * Implementation of the internal command "help", which prints 
       * some general help text to the console monitor.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      void internalCmd_HelpGeneral ();

      /**
       * Implementation of the internal command "hist" or "h", which will 
       * print a list of the N most recent commands to the console monitor.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      void internalCmd_Hist ();

      /**
       * Implementation of the internal command "info", which will 
       * print some system information to the console monitor.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.04.14
       */
      void internalCmd_Info ();

      /*
       * Implementation of the internal command "popd", which will 
       * print the content of the Directory Stack to the console monitor.
       */
      void internalCmd_PopD ();

      /*
       * Implementation of the internal command "which", which will 
       * print the location of some module in some %path% variable.
       */
      void internalCmd_Which ();
};

#endif // #ifndef __LCMD_PROCESSLAUNCHER
