/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdOptions.h"
#include "lcmd/LCmdCmdContainer.h"
#include "lcmd/LCmdCmdLinePrompt.h"
#include "lcmd/LCmdCmdLineEntry.h"
#include "lcmd/LCmdSubClient.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmd.h"

#include "glib/gui/GMenu.h"
#include "glib/gui/GDropList.h"
#include "glib/gui/GListBox.h"
#include "glib/gui/GKeyBar.h"
#include "glib/gui/GStatusbar.h"
#include "glib/gui/GTabbedPanel.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/util/GTokenizer.h"
#include "glib/exceptions/GIllegalStateException.h"

LCmdOptions::LCmdOptions ( LCmd& lcmd )
            :DefaultFontFilePanelHeader("9.WarpSans"),
             DefaultFontFrameTitlebar("9.WarpSans Bold"),
             DefaultFontFrameMenubar("9.WarpSans Bold"),
             DefaultFontConMon("11.System VIO"),
             DefaultFontPrompt("9.WarpSans Bold"),
             DefaultFontCmdLineEntry("9.WarpSans"),
             editor("e.exe", "!p", "!h", false, true),
             viewer("e.exe", "!p", "!h", true, false),
             ffind("pmseek.exe", "", "!h", false, true),
             fdiff("pmdiff.exe", "!l !r", "!h", false, true),
             cclean("clonecln.exe", "", "!h", false, true),
             undel("phoenix2.exe", "", "!h", true, false),
             eas("eabrowse.exe", "!p", "!h", true, false)
{
   ::lcmd = &lcmd;

   panel1RelWidth              = 500; // 50%
   consoleRelHeight            = 360; // 36%
   panelSizesStart5050         = false;
   panelsFullScreen            = false;
   panelsAreVisible            = true;
   fancyMenuBars               = true;
   fancyPopupMenues            = true;

   fileFilter.inclFiles        = true;
   fileFilter.inclDirs         = true;
   fileFilter.flagArchive      = 2;
   fileFilter.flagReadOnly     = 2;
   fileFilter.flagSystem       = 2;
   fileFilter.flagHidden       = 2;
   fileFilter.useInclFilterStr = false;
   fileFilter.useExclFilterStr = false;
   fileFilter.inclFilterStr    = "";
   fileFilter.exclFilterStr    = "";

   cmpDirs.timeStampOpt        = LCmdDlgCompareDirs::TAGTMST_DIFF;
   cmpDirs.sizeOpt             = LCmdDlgCompareDirs::TAGSIZE_DIFF;
   cmpDirs.tagUnmatched        = true;
   cmpDirs.compareContents     = false;
}

LCmdOptions::~LCmdOptions ()
{
}

LCmdOptions& LCmdOptions::GetOptions ()
{
   return lcmd->options;
}

/**
 * List of some commands that are known not to be compatible with the
 * Console Monitor of Larsen Commander.
 */
KnownSSA knownSSA[] =
{
   { true,  false, "4os2.exe" },
   { true,  false, "cmd.exe" },
   { true,  true,  "cmdshl.cmd" },
   { true,  true,  "diskcopy" },
   { true,  true,  "fc.exe" },
   { true,  true,  "fdisk" },
   { true,  true,  "fte.exe" },
   { true,  true,  "ftp.exe" },
   { true,  true,  "more" },
   { true,  true,  "pkzip.exe" },
   { true,  true,  "rar.exe" },
   { true,  true,  "tedit.exe" },
   { true,  true,  "telnet.exe" },
   { true,  true,  "yaos.exe" },
   { true,  true,  "xaos.exe" },
   { true,  true,  "checkini.exe" },
   { true,  true,  "os2scan.exe" },
   { true,  true,  "watchcat.exe" },
   { 0, 0, 0 }
};

LCmdOptions::ProgOption::ProgOption ()
{
   internalAvail = false;
   useExternal = false;
   init_def();
}

LCmdOptions::ProgOption::ProgOption ( const ProgOption& prog )
{
   copyFrom(prog);
}

LCmdOptions::ProgOption::ProgOption ( const char* progName, 
                                      const char* params, 
                                      const char* workDir, 
                                      bool internalAvail, 
                                      bool useExternal )
{
   this->internalAvail = internalAvail;
   this->useExternal = useExternal;
   this->progName = progName;
   this->params = params;
   this->workDir = workDir;
   init_def();
}

LCmdOptions::ProgOption& LCmdOptions::ProgOption::copyFrom ( const LCmdOptions::ProgOption& prog )
{
   this->internalAvail = prog.internalAvail;
   this->useExternal = prog.useExternal;
   this->progName = prog.progName;
   this->params = prog.params;
   this->workDir = prog.workDir;
   init_def();
   return *this;
}

LCmdOptions::ProgOption& LCmdOptions::ProgOption::operator= ( const LCmdOptions::ProgOption& prog )
{
   return copyFrom(prog);
}

bool LCmdOptions::ProgOption::operator== ( const LCmdOptions::ProgOption& prog )
{
   if (&prog == this)
   {
      return true;
   }
   else
   if (internalAvail == prog.internalAvail &&
       useExternal == prog.useExternal &&
       progName == prog.progName &&
       params == prog.params &&
       workDir == prog.workDir)
   {
      return true;
   }
   else
   {
      return false;
   }
}

void LCmdOptions::ProgOption::init_def ()
{
   def_internalAvail = internalAvail;
   def_useExternal = useExternal;
   def_progName = progName;
   def_params = params;
   def_workDir = workDir;
}

void LCmdOptions::ProgOption::restoreToDefault ()
{
   internalAvail = def_internalAvail;
   useExternal = def_useExternal;
   progName = def_progName;
   params = def_params;
   workDir = def_workDir;
}

bool LCmdOptions::ProgOption::launchIt ( bool forceNewSession, bool noEcho ) const
{
   int flags = LCmdCmdLineEntry::ECF_CLOSEON_EXIT;
   if (forceNewSession)
      flags |= LCmdCmdLineEntry::ECF_FORCE_NEW_SESSION;
   if (!noEcho)
      flags |= LCmdCmdLineEntry::ECF_DO_ECHO;
   GString cmd = progName;
   if (cmd.containsWhiteSpace())
      cmd.insert('"').append('"', 1);
   cmd += GString::Blank;
   cmd += params;
   LCmdCmdLineEntry& entry = lcmd->cmdCont.cmdline.entry;
   return entry.executeCommand(cmd, workDir, flags);
}


LCmdOptions::SaveOptOnExit::SaveOptOnExit ()
{
   restoreToDefault();
}

void LCmdOptions::SaveOptOnExit::restoreToDefault ()
{
   everything = true;

   // Set all other "save options" to false by default, since 
   // the "everything" defaults to true anyway. This is important in
   // order for the "Save everytghing on exit" to operate as expected 
   // to the first time user of Larsen Commander. Because that 
   // command switches only the "everything" option and leaves the
   // below options unswitched. If the first time user toggles the 
   // "Save everything on exit" command to OFF he probably wants to 
   // prevent Larsen Commander from saving anything at all. So keep
   // these options false by default.
   currentDirs = false;
   storedDirs = false;
   dirHist = false;
   cmdHist = false;
   dirCache = false;
   curSelFiles = false;
   visibleState = false;
   panelModeSort = false;
   frameWinPos = false;
   colors = false;
   fonts = false;
   otherOptions = false;
}

LCmdOptions::DirCacheOpts::DirCacheOpts ()
{
   restoreToDefault();
}

void LCmdOptions::DirCacheOpts::restoreToDefault ()
{
   addDirsDuringWalk = true;
   autoStripDirsAfterDirDeletions = true;
   includeVfsDirs = false;
}

LCmdOptions::ConfirmOpts::ConfirmOpts ()
{
   restoreToDefault();
}

void LCmdOptions::ConfirmOpts::restoreToDefault ()
{
   prgExit = false;
   delFile = true;
   delFileInSubDirs = true;
   delEmptyDir = true;
   delDirContainingFiles = true;
   delHidSysReadFile = true;
   overwriteFileOrDir = true;
}

LCmdOptions::FileCopyOpts::FileCopyOpts ()
{
   restoreToDefault();
}

void LCmdOptions::FileCopyOpts::restoreToDefault ()
{
   showStatistics = true;
   waitOnFinish = false;
   warnIfNotEnoughTargetSpace = true;
   preallocSpace = Never;
   preallocSpaceOnDrives = "";
   maxPreallocateSize = 1024L * 1024L * 256L;
   maxBuffSize = 64;
}

void LCmdOptions::FileCopyOpts::setPreallocSpaceFromInt ( int value )
{
   PreallocID id = PreallocID(value);
   switch (id)
   {
      case Never: 
      case Always: 
      case SpecifiedDrives: 
         preallocSpace = id; 
         break;
      default: 
         gthrow_(GIllegalArgumentException(GString("Unknown PreallocID: %d", GVArgs(value))));
   }
}

bool LCmdOptions::FileCopyOpts::isPreallocSpaceForDrive ( const GString& driveStr )
{
   switch (preallocSpace)
   {
      case Never:
         return false;
      case Always:
         return true;
      case SpecifiedDrives:
      {
         GTokenizer toknzr(preallocSpaceOnDrives, ";,", GString::Empty, true);
         for (;;)
         {
            const GToken* token = toknzr.getNextToken();
            if (token->isEmpty())
               break;
            GString tokenStr = token->toString();
            if (tokenStr.equalsIgnoreCase(driveStr))
               return true;
         }
         return false;
      }
   }
   gthrow_(GIllegalStateException(GString("Unknown preallocSpace: %d", GVArgs(int(preallocSpace)))));
}

LCmdOptions::FileDelOpts::FileDelOpts ()
{
   restoreToDefault();
}

void LCmdOptions::FileDelOpts::restoreToDefault ()
{
   defaultDeleteToTrashCan = false;
   useDelKey = false;
   showDirsProgressOnly = false;
}

LCmdOptions::ConVariousOpts::ConVariousOpts ()
{
   restoreToDefault();
}

void LCmdOptions::ConVariousOpts::restoreToDefault ()
{
   defaultCloseConOnExit = false;
   startConFullScreen = false;
   startDosFullScreen = false;
}

LCmdOptions::VfsVariousOpts::VfsVariousOpts ()
{
   restoreToDefault();
}

void LCmdOptions::VfsVariousOpts::restoreToDefault ()
{
}

LCmdOptions::VfsZipOpts::VfsZipOpts ()
{
   restoreToDefault();
}

void LCmdOptions::VfsZipOpts::restoreToDefault ()
{
   sysLaunchZipFiles = false;
   toolPath = "zip.exe";
   argsDelete = "-d \"%1\" * -i@\"%2\"";
}

LCmdOptions::VariousOpts::VariousOpts ()
{
   restoreToDefault();
}

void LCmdOptions::VariousOpts::restoreToDefault ()
{
   addToItemHeight = 0;
   addToItemWidth = 0;
   delayShowTooltipMillis = 750;
   autoHideTooltipMillis = 6000;
}

LCmdDlgOptions::Startup::Startup ( FilePanelID fpid )
                        :fpid(fpid),
                         current(null)
{
}

void LCmdDlgOptions::Startup::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("ToggleStartInCurDir", current->useCurrentDir, false);
   dlg.setComponentValue("ToggleUseSpecDir", current->useDir, false);
   dlg.setComponentValue("EntrySpecDir", current->dir, false);
}

void LCmdDlgOptions::Startup::getControlValues ( GDialogPanel& dlg )
{
   current->useCurrentDir = dlg.getComponentBoolValue("ToggleStartInCurDir");
   current->useDir = dlg.getComponentBoolValue("ToggleUseSpecDir");
   current->dir = dlg.getComponentValue("EntrySpecDir");
}

bool LCmdDlgOptions::Startup::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         current = (fpid == LEFT ? &lcmd->fp1->startup : &lcmd->fp2->startup);
         backup = *current;
         setControlValues(dlg);
         return true;
      }

      case GM_DISMISSDIALOG: 
      {
         getControlValues(dlg);
         current = null;
         return true; 
      }

      case GM_CTRLCHANGED: 
      {
         getControlValues(dlg);
         GString id = msg.getParam1String();
         if (id == "ToggleStartInCurDir")
         {
            // Toggle button: Startup in "current" directory.
            dlg.setComponentEnabled("ToggleUseSpecDir", !current->useCurrentDir);
            dlg.setComponentEnabled("EntrySpecDir", !current->useCurrentDir);
            return true;
         }
         return true; 
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            *current = backup;
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current->resetToDefault();
            setControlValues(dlg);
            return true;
         }
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::Filter::Filter ( FilePanelID fpid )
                       :fpid(fpid),
                        fpanel(null),
                        current(null)
{
}

void LCmdDlgOptions::Filter::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("103", current->include, false);
   dlg.setComponentValue("101", current->exclude, false);
   dlg.setComponentValue("106", current->hideArchive, false);
   dlg.setComponentValue("107", current->hideHidden, false);
   dlg.setComponentValue("108", current->hideSystem, false);
   dlg.setComponentValue("109", current->hideReadOnly, false);
   dlg.setComponentValue("110", current->showDirs, false);
   dlg.setComponentValue("105", current->showFiles, false);
}

void LCmdDlgOptions::Filter::getControlValues ( GDialogPanel& dlg )
{
   current->include = dlg.getComponentValue("103");
   current->exclude = dlg.getComponentValue("101");
   current->hideArchive = dlg.getComponentBoolValue("106");
   current->hideHidden = dlg.getComponentBoolValue("107");
   current->hideSystem = dlg.getComponentBoolValue("108");
   current->hideReadOnly = dlg.getComponentBoolValue("109");
   current->showDirs = dlg.getComponentBoolValue("110");
   current->showFiles = dlg.getComponentBoolValue("105");
}

bool LCmdDlgOptions::Filter::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         fpanel = (fpid == LEFT ? lcmd->fp1 : lcmd->fp2);
         current = (fpid == LEFT ? &fpanel->filter : &fpanel->filter);
         backup = *current;
         setControlValues(dlg);
         return true;
      }

      case GM_DISMISSDIALOG:
      {
         getControlValues(dlg);
         if (!current->equals(backup))
         {
            // Content(s) of this dialog has been changed, so we must
            // reread filename items.
            fpanel->reRead();
         }
         current = null;
         fpanel = null;
         return true; 
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            *current = backup;
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current->resetToDefault();
            setControlValues(dlg);
            return true;
         }
         return true; 
      }

      default:
         return false;
   }
}

LCmdDlgOptions::Sort::Sort ( FilePanelID fpid )
                     :fpid(fpid),
                      fpanel(null),
                      current(null)
{
   whatID[0] = 102;
   whatID[1] = 105;
   whatID[2] = 108;
   whatID[3] = 118;

   howID[0] = 104;
   howID[1] = 107;
   howID[2] = 110;
   howID[3] = 120;
}

void LCmdDlgOptions::Sort::setControlValues ( GDialogPanel& dlg )
{
   for (int i=0; i<4; i++)
   {
      dlg.setComponentValue(GInteger::ToString(whatID[i]), current->what[i], false);
      dlg.setComponentValue(GInteger::ToString(howID[i]), current->how[i], false);
   }
}

void LCmdDlgOptions::Sort::getControlValues ( GDialogPanel& dlg )
{
   for (int i=0; i<4; i++)
   {
      current->what[i] = LCmdFilePanelSortOptions::SORT_WHAT(dlg.getComponentIntValue(GInteger::ToString(whatID[i]), LCmdFilePanelSortOptions::PSW_UNSORTED));
      current->how[i] = LCmdFilePanelSortOptions::SORT_HOW(dlg.getComponentIntValue(GInteger::ToString(howID[i]), LCmdFilePanelSortOptions::PSH_ASCENDING));
   }
}

bool LCmdDlgOptions::Sort::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           dlg.setNoInitCtrlChanged(true);
           fpanel = (fpid == LEFT ? lcmd->fp1 : lcmd->fp2);
           current = (fpid == LEFT ? &fpanel->sortOpt : &fpanel->sortOpt);
           backup = *current;
           setControlValues(dlg);
           return true;

      case GM_DISMISSDIALOG:
           current = null;
           fpanel = null;
           return true;

      case GM_CTRLCHANGED:
           getControlValues(dlg);
           fpanel->sortList(true, true);
           return true;

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              *current = backup;
              setControlValues(dlg);
              fpanel->sortList(true, true);
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current->resetToDefault();
              setControlValues(dlg);
              fpanel->sortList(true, true);
              return true;
           }
           return true; }

      default:
           return true;
   }
}

void LCmdDlgOptions::ViewCmn::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("102", current.showToolbar, false);
   dlg.setComponentValue("103", current.showKeybar, false);
   dlg.setComponentValue("104", current.showStatbar, false);
   dlg.setComponentValue("112", current.panelsFullScreen, false);
   dlg.setComponentValue("113", current.showCmdLine, false);
   dlg.setComponentValue("114", current.panelSizesStart5050, false);
   dlg.setComponentValue("107", current.panel1RelWidth / 10, false);
   dlg.setComponentValue("115", current.consoleRelHeight / 10, false);
}

void LCmdDlgOptions::ViewCmn::getControlValues ( GDialogPanel& dlg )
{
   current.showToolbar = dlg.getComponentBoolValue("102");
   current.showKeybar = dlg.getComponentBoolValue("103");
   current.showStatbar = dlg.getComponentBoolValue("104");
   current.panelsFullScreen = dlg.getComponentBoolValue("112");
   current.showCmdLine = dlg.getComponentBoolValue("113");
   current.panelSizesStart5050 = dlg.getComponentBoolValue("114");
   current.panel1RelWidth = dlg.getComponentIntValue("107") * 10;
   current.consoleRelHeight = dlg.getComponentIntValue("115") * 10;
}

void LCmdDlgOptions::ViewCmn::activateCurrentSettings ()
{
   lcmd->mainWin.setToolbarVisible(current.showToolbar);
   lcmd->mainWin.setKeybarVisible(current.showKeybar);
   lcmd->mainWin.setStatusbarVisible(current.showStatbar);
   lcmd->mainWin.subClient->setFullScreenMode(current.panelsFullScreen);
   lcmd->mainWin.subClient->setCommandLineVisible(current.showCmdLine);
   lcmd->options.panelSizesStart5050 = current.panelSizesStart5050;
   lcmd->options.panel1RelWidth = current.panel1RelWidth;
   lcmd->options.consoleRelHeight = current.consoleRelHeight;
   lcmd->mainWin.subClient->panelsParent.layout();
   lcmd->mainWin.subClient->recalcAllPartsOfLayout();
}

bool LCmdDlgOptions::ViewCmn::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           current.showToolbar = lcmd->mainWin.isToolbarVisible();
           current.showStatbar = lcmd->mainWin.isStatusbarVisible();
           current.showKeybar = lcmd->mainWin.isKeybarVisible();
           current.showCmdLine = lcmd->mainWin.subClient->cmdcont.cmdline.isVisible();
           current.panelsFullScreen = lcmd->options.panelsFullScreen;
           current.panel1RelWidth = lcmd->options.panel1RelWidth;
           current.consoleRelHeight = lcmd->options.consoleRelHeight;
           current.panelSizesStart5050 = lcmd->options.panelSizesStart5050;

           backup = current;

           dlg.setComponentEnabled("112", current.showCmdLine);
           if (!current.showCmdLine)
              current.panelsFullScreen = true;

           setControlValues(dlg);
           dlg.setNoInitCtrlChanged(true);
           return true;

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           if (id == "102") // Toggle button toolbar on/off
           {
              getControlValues(dlg);
              lcmd->mainWin.setToolbarVisible(current.showToolbar);
           }
           else
           if (id == "103") // Toggle button keybar on/off
           {
              getControlValues(dlg);
              lcmd->mainWin.setKeybarVisible(current.showKeybar);
           }
           else
           if (id == "104") // Toggle button statusbar on/off
           {
              getControlValues(dlg);
              lcmd->mainWin.setStatusbarVisible(current.showStatbar);
           }
           else
           if (id == "112") // Toggle "Full Screen" mode on/off
           {
              getControlValues(dlg);
              lcmd->mainWin.subClient->setFullScreenMode(current.panelsFullScreen);
           }
           else
           if (id == "113") // Toggle Command Line on/off
           {
              getControlValues(dlg);
              dlg.setComponentEnabled("112", current.showCmdLine);
              if (!current.showCmdLine)
              {
                 current.panelsFullScreen = true;
                 setControlValues(dlg);
              }
              lcmd->mainWin.subClient->setCommandLineVisible(current.showCmdLine);
           }
           else
           if (id == "114") // 50/50 panel sizes upon startup
           {
              getControlValues(dlg);
              lcmd->options.panelSizesStart5050 = current.panelSizesStart5050;
           }
           else
           if (id == "107") // Left Panel Size Spin Button
           {
              getControlValues(dlg);
              lcmd->options.panel1RelWidth = current.panel1RelWidth;
              lcmd->mainWin.subClient->panelsParent.layout();
           }
           else
           if (id == "115") // Console Size Spin Button
           {
              getControlValues(dlg);
              lcmd->options.consoleRelHeight = current.consoleRelHeight;
              lcmd->mainWin.subClient->recalcAllPartsOfLayout();
           }
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              current = backup;
              setControlValues(dlg);
              activateCurrentSettings();
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.showToolbar = true;
              current.showStatbar = true;
              current.showKeybar = true;
              current.panelsFullScreen = false;
              current.showCmdLine = true;
              current.panel1RelWidth = 500; // 50%
              current.consoleRelHeight = 360; // 36%
              current.panelSizesStart5050 = false;
              setControlValues(dlg);
              activateCurrentSettings();
              return true;
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::View::View ( FilePanelID fpid )
                     :fpid(fpid),
                      fpanel(null),
                      current(null)
{
}

void LCmdDlgOptions::View::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("105", current->showHeaderbar, false);
   dlg.setComponentValue("107", current->showInfobar, false);
   dlg.setComponentValue("102", current->showItemIcon, false);
   dlg.setComponentValue("121", current->showColumnsBar, false);

   dlg.setComponentValue("118", current->showHeaderDrivButt, false);
   dlg.setComponentValue("119", current->showHeaderUpDirButt, false);
   dlg.setComponentValue("120", current->showHeaderRootButt, false);

   dlg.setComponentValue("103", current->showIcon, false);

   switch (current->viewMode)
   {
      case LCmdFilePanelViewOptions::VIEWMODE_BRIEF:
           dlg.setComponentValue("112", true, false);
           break;

      case LCmdFilePanelViewOptions::VIEWMODE_WIDE:
           dlg.setComponentValue("113", true, false);
           break;

      case LCmdFilePanelViewOptions::VIEWMODE_FULL:
           dlg.setComponentValue("114", true, false);
           break;

      case LCmdFilePanelViewOptions::VIEWMODE_INFO:
           dlg.setComponentValue("115", true, false);
           break;

#if __ENABLE_TREE_VIEW
      case LCmdFilePanelViewOptions::VIEWMODE_TREE:
           dlg.setComponentValue("116", true, false);
           break;
#endif
   }
}

LCmdFilePanelViewOptions::VIEWMODE LCmdDlgOptions::View::getControlValues ( GDialogPanel& dlg )
{
   current->showHeaderbar  = dlg.getComponentBoolValue("105");
   current->showInfobar    = dlg.getComponentBoolValue("107");
   current->showItemIcon   = dlg.getComponentBoolValue("102");
   current->showColumnsBar = dlg.getComponentBoolValue("121");

   current->showHeaderDrivButt  = dlg.getComponentBoolValue("118");
   current->showHeaderUpDirButt = dlg.getComponentBoolValue("119");
   current->showHeaderRootButt  = dlg.getComponentBoolValue("120");

   current->showIcon = LCmdFilePanelViewOptions::SHOWICON(dlg.getComponentIntValue("103"));

   if (dlg.getComponentBoolValue("112"))
      return LCmdFilePanelViewOptions::VIEWMODE_BRIEF;
   else
   if (dlg.getComponentBoolValue("113"))
      return LCmdFilePanelViewOptions::VIEWMODE_WIDE;
   else
   if (dlg.getComponentBoolValue("114"))
      return LCmdFilePanelViewOptions::VIEWMODE_FULL;
   else
   if (dlg.getComponentBoolValue("115"))
      return LCmdFilePanelViewOptions::VIEWMODE_INFO;
   else
#if __ENABLE_TREE_VIEW
   if (dlg.getComponentBoolValue("116"))
      return LCmdFilePanelViewOptions::VIEWMODE_TREE;
   else
#endif
      return LCmdFilePanelViewOptions::VIEWMODE_BRIEF; // Will never happen, but in case.
}

bool LCmdDlgOptions::View::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           fpanel = (fpid == LEFT ? lcmd->fp1 : lcmd->fp2);
           current = (fpid == LEFT ? &fpanel->view : &fpanel->view);
           backup = *current;
           setControlValues(dlg);
           dlg.setNoInitCtrlChanged(true);

           // Disable the icon-style combo box if icon visibility is
           // initially toggled off
           dlg.setComponentEnabled("103", current->showItemIcon);

           // Disable "Columns Bar" if panel mode is not equal to "FULL"
           dlg.setComponentEnabled("121", current->viewMode == LCmdFilePanelViewOptions::VIEWMODE_FULL);

           // Disable the headerbar options if the headerbar is
           // initially toggled off.
           dlg.setComponentEnabled("118", current->showHeaderbar);
           dlg.setComponentEnabled("119", current->showHeaderbar);
           dlg.setComponentEnabled("120", current->showHeaderbar);
           return true;

      case GM_DISMISSDIALOG:
           current = null;
           fpanel = null;
           return true;

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           if (id == "102") // Toggle button, icon on/off
           {
              bool show = dlg.getComponentBoolValue("102");
              fpanel->cmdToggleShowIcons(show);
              dlg.setComponentEnabled("103", show);
           }
           else
           if (id == "121") // Toggle button, Columns Bar on/off
           {
              getControlValues(dlg);
              LCmdFilePanelModeAbstract& curView = fpanel->getCurrentView();
              curView.layout();
           }
           else
           if (id == "103") // Drop list of icon styles
           {
              getControlValues(dlg);
              fpanel->cmdSetIconMode(current->showIcon, current->showItemIcon, true);
           }
           else
           if (id == "105" || // Toggle button, headerbar on/off
               id == "107")   // Toggle button, infobar on/off
           {
              getControlValues(dlg);
              dlg.setComponentEnabled("118", current->showHeaderbar);
              dlg.setComponentEnabled("119", current->showHeaderbar);
              dlg.setComponentEnabled("120", current->showHeaderbar);
              fpanel->frameWin.layout();
           }
           else
           if (id == "118" || // Toggle button, Headerbar Drives Button on/off
               id == "119" || // Toggle button, Headerbar Walk Up Button on/off
               id == "120")   // Toggle button, Headerbar Root Directory Up Button on/off
           {
              getControlValues(dlg);
              fpanel->headerWin.layout();
              fpanel->headerWin.invalidateAll(true);
           }
           else
           if (id == "112" || // Radio button, brief view
               id == "113" || // Radio button, wide view
               id == "114" || // Radio button, full view
               id == "115" || // Radio button, info view
               id == "116")   // Radio button, tree view
           {
              LCmdFilePanelViewOptions::VIEWMODE newMode;
              newMode = getControlValues(dlg);
              dlg.setComponentEnabled("121", newMode == LCmdFilePanelViewOptions::VIEWMODE_FULL);
              fpanel->setViewMode(newMode);
           }
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              if (current->viewMode != backup.viewMode)
                 fpanel->setViewMode(backup.viewMode);
              *current = backup;
              fpanel->cmdSetIconMode(current->showIcon, current->showItemIcon, true);
              setControlValues(dlg);
              fpanel->frameWin.layout();
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current->showHeaderbar = true;
              current->showInfobar = true;
              current->showHeaderDrivButt = true;
              current->showHeaderUpDirButt = true;
              current->showHeaderRootButt = true;
              current->showColumnsBar = true;
              fpanel->cmdSetIconMode(LCmdFilePanelViewOptions::SHOWICON_INTERNAL, true);
              if (current->viewMode != LCmdFilePanelViewOptions::VIEWMODE_BRIEF)
                 fpanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
              setControlValues(dlg);
              fpanel->frameWin.layout();
              return true;
           }
           return true; }

      default:
           return true;
   }
}

void LCmdDlgOptions::ViewStyle::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("103", current.toolbarFlatButtons, false);
   dlg.setComponentValue("105", current.toolbarTooltip, false);
   dlg.setComponentValue("102", current.cmdlineFlatButtons, false);
   dlg.setComponentValue("113", current.cmdlineTooltip, false);
   dlg.setComponentValue("104", current.keybarFlatButtons, false);
   dlg.setComponentValue("115", current.keybarTooltip, false);
   dlg.setComponentValue("118", current.headbarsTooltip, false);
   dlg.setComponentValue("121", current.columnbarsTooltip, false);
   dlg.setComponentValue("InfobarsTooltip", current.infobarsTooltip, false);
   dlg.setComponentValue("StatusbarTooltip", current.statusbarTooltip, false);
   dlg.setComponentValue("107", current.fancyMenuBars, false);
   dlg.setComponentValue("111", current.fancyPopupMenues, false);
}

void LCmdDlgOptions::ViewStyle::getControlValues ( GDialogPanel& dlg )
{
   current.toolbarFlatButtons = dlg.getComponentBoolValue("103");
   current.toolbarTooltip = dlg.getComponentBoolValue("105");
   current.cmdlineFlatButtons = dlg.getComponentBoolValue("102");
   current.cmdlineTooltip = dlg.getComponentBoolValue("113");
   current.keybarFlatButtons = dlg.getComponentBoolValue("104");
   current.keybarTooltip = dlg.getComponentBoolValue("115");
   current.headbarsTooltip = dlg.getComponentBoolValue("118");
   current.columnbarsTooltip = dlg.getComponentBoolValue("121");
   current.infobarsTooltip = dlg.getComponentBoolValue("InfobarsTooltip");
   current.statusbarTooltip = dlg.getComponentBoolValue("StatusbarTooltip");
   current.fancyMenuBars = dlg.getComponentBoolValue("107");
   current.fancyPopupMenues = dlg.getComponentBoolValue("111");
}

void LCmdDlgOptions::ViewStyle::activateCurrentSettings ()
{
   GToolbar* toolbar = lcmd->mainWin.getToolbar();
   if (toolbar != null)
   {
      toolbar->setFlatButtonBorders(current.toolbarFlatButtons);
      toolbar->setShowTooltip(current.toolbarTooltip);
   }
   lcmd->mainWin.subClient->cmdcont.cmdline.setFlatButtonBorders(current.cmdlineFlatButtons);
   lcmd->mainWin.subClient->cmdcont.cmdline.setShowTooltip(current.cmdlineTooltip);

   GKeyBar* keybar = lcmd->mainWin.getKeybar();
   if (keybar != null)
   {
      keybar->setFlatButtonBorders(current.keybarFlatButtons);
      keybar->setShowTooltip(current.keybarTooltip);
   }

   LCmdPanelsParent& pp = lcmd->mainWin.subClient->panelsParent;
   pp.frame1.headerWin.setShowTooltip(current.headbarsTooltip);
   pp.frame2.headerWin.setShowTooltip(current.headbarsTooltip);
   pp.frame1.fpanel.viewFull.columnsBar.setShowTooltip(current.columnbarsTooltip);
   pp.frame1.fpanel.viewWide.columnsBar.setShowTooltip(current.columnbarsTooltip);
   pp.frame2.fpanel.viewFull.columnsBar.setShowTooltip(current.columnbarsTooltip);
   pp.frame2.fpanel.viewWide.columnsBar.setShowTooltip(current.columnbarsTooltip);
   pp.frame1.infoBar.setShowTooltip(current.infobarsTooltip);
   pp.frame2.infoBar.setShowTooltip(current.infobarsTooltip);
   GStatusbar* statusbar = lcmd->mainWin.getStatusbar();
   if (statusbar != null) // Will always be true, but in case.
      statusbar->setShowTooltip(current.statusbarTooltip);
   lcmd->options.fancyMenuBars = current.fancyMenuBars;
   lcmd->options.fancyPopupMenues = current.fancyPopupMenues;
}

bool LCmdDlgOptions::ViewStyle::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: {
           GToolbar* toolbar = lcmd->mainWin.getToolbar();
           if (toolbar != null)
           {
              current.toolbarFlatButtons = toolbar->isFlatButtonBorders();
              current.toolbarTooltip = toolbar->isShowTooltip();
           }

           current.cmdlineFlatButtons = lcmd->cmdLine.isFlatButtonBorders();
           current.cmdlineTooltip = lcmd->cmdLine.isShowTooltip();

           GKeyBar* keybar = lcmd->mainWin.getKeybar();
           if (keybar != null)
           {
              current.keybarFlatButtons = keybar->isFlatButtonBorders();
              current.keybarTooltip = keybar->isShowTooltip();
           }
           current.headbarsTooltip = lcmd->mainWin.subClient->panelsParent.frame1.headerWin.isShowTooltip();
           current.columnbarsTooltip = lcmd->mainWin.subClient->panelsParent.frame1.fpanel.viewFull.columnsBar.isShowTooltip();
           current.infobarsTooltip = lcmd->mainWin.subClient->panelsParent.frame1.infoBar.isShowTooltip();
           GStatusbar* statusbar = lcmd->mainWin.getStatusbar();
           if (statusbar != null) // Will always be true, but in case.
              current.statusbarTooltip = statusbar->isShowTooltip();
           current.fancyMenuBars = lcmd->options.fancyMenuBars;
           current.fancyPopupMenues = lcmd->options.fancyPopupMenues;

           backup = current;
           setControlValues(dlg);
           dlg.setNoInitCtrlChanged(true);
           return true; }

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           getControlValues(dlg);
           if (id == "102") // Toggle flat command line buttons on/off
              lcmd->cmdCont.cmdline.setFlatButtonBorders(current.cmdlineFlatButtons);
           else
           if (id == "113") // Toggle command line tooltip on/off
              lcmd->cmdLine.setShowTooltip(current.cmdlineTooltip);
           else
           if (id == "103") // Toggle flat toolbar buttons
           {
              GToolbar* toolbar = lcmd->mainWin.getToolbar();
              if (toolbar != null)
                 toolbar->setFlatButtonBorders(current.toolbarFlatButtons);
           }
           else
           if (id == "104") // Toggle flat keybar buttons
           {
              GKeyBar* keybar = lcmd->mainWin.getKeybar();
              if (keybar != null)
                 keybar->setFlatButtonBorders(current.keybarFlatButtons);
           }
           else
           if (id == "115") // Toggle keybar button tooltip
           {
              GKeyBar* keybar = lcmd->mainWin.getKeybar();
              if (keybar != null)
                 keybar->setShowTooltip(current.keybarTooltip);
           }
           else
           if (id == "118") // Toggle headerbars buttons tooltip.
           {
              lcmd->mainWin.subClient->panelsParent.frame1.headerWin.setShowTooltip(current.headbarsTooltip);
              lcmd->mainWin.subClient->panelsParent.frame2.headerWin.setShowTooltip(current.headbarsTooltip);
           }
           else
           if (id == "121") // Toggle columnbars buttons tooltip.
           {
              lcmd->mainWin.subClient->panelsParent.frame1.fpanel.viewFull.columnsBar.setShowTooltip(current.columnbarsTooltip);
              lcmd->mainWin.subClient->panelsParent.frame1.fpanel.viewWide.columnsBar.setShowTooltip(current.columnbarsTooltip);
              lcmd->mainWin.subClient->panelsParent.frame2.fpanel.viewFull.columnsBar.setShowTooltip(current.columnbarsTooltip);
              lcmd->mainWin.subClient->panelsParent.frame2.fpanel.viewWide.columnsBar.setShowTooltip(current.columnbarsTooltip);
           }
           else
           if (id == "InfobarsTooltip") // Toggle infobars buttons tooltip.
           {
              lcmd->mainWin.subClient->panelsParent.frame1.infoBar.setShowTooltip(current.infobarsTooltip);
              lcmd->mainWin.subClient->panelsParent.frame2.infoBar.setShowTooltip(current.infobarsTooltip);
           }
           else
           if (id == "StatusbarTooltip") // Toggle statusbar cells tooltip.
           {
              GStatusbar* statusbar = lcmd->mainWin.getStatusbar();
              if (statusbar != null) // Will always be true, but in case.
                 statusbar->setShowTooltip(current.statusbarTooltip);
           }
           if (id == "105") // Toggle show toolbar tooltip.
           {
              GToolbar* toolbar = lcmd->mainWin.getToolbar();
              if (toolbar != null)
                 toolbar->setShowTooltip(current.toolbarTooltip);
           }
           else
           if (id == "107") // Toggle fancy menubars
           {
              lcmd->options.fancyMenuBars = current.fancyMenuBars;
              GMenu* mbar = lcmd->mainWin.getMenubar();
              mbar->setUseFancyMenues(lcmd->options.fancyMenuBars);
           }
           else
           if (id == "111") // Toggle fancy popup menues
           {
              lcmd->options.fancyPopupMenues = current.fancyPopupMenues;
              lcmd->mainWin.getPopupMenu()->setUseFancyMenues(lcmd->options.fancyPopupMenues); // Update the user menu (F2)
              lcmd->fp1->getPopupMenu()->setUseFancyMenues(lcmd->options.fancyPopupMenues);
              lcmd->fp2->getPopupMenu()->setUseFancyMenues(lcmd->options.fancyPopupMenues);
              lcmd->cmdLine.entry.getPopupMenu()->setUseFancyMenues(lcmd->options.fancyPopupMenues);
              lcmd->cmdCont.conmon.getPopupMenu()->setUseFancyMenues(lcmd->options.fancyPopupMenues);
           }
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              current = backup;
              setControlValues(dlg);
              activateCurrentSettings();
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.toolbarFlatButtons = true;
              current.toolbarTooltip = true;
              current.cmdlineFlatButtons = true;
              current.cmdlineTooltip = true;
              current.keybarFlatButtons = true;
              current.keybarTooltip = true;
              current.headbarsTooltip = true;
              current.columnbarsTooltip = true;
              current.infobarsTooltip = true;
              current.statusbarTooltip = true;
              current.fancyMenuBars = true;
              current.fancyPopupMenues = true;
              setControlValues(dlg);
              activateCurrentSettings();
              return true;
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::Confirm::Confirm ( LCmdOptions::ConfirmOpts& current )
                        :current(current),
                         backup(current)
{
}

void LCmdDlgOptions::Confirm::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("Exit", current.prgExit, false);
   dlg.setComponentValue("DelFile", current.delFile, false);
   dlg.setComponentValue("DelFileInSubDirs", current.delFileInSubDirs, false);
   dlg.setComponentValue("DelDir", current.delEmptyDir, false);
   dlg.setComponentValue("DelNonEmptyDir", current.delDirContainingFiles, false);
   dlg.setComponentValue("DelHidSysReadOnly", current.delHidSysReadFile, false);
   dlg.setComponentValue("Overwrite", current.overwriteFileOrDir, false);
}

void LCmdDlgOptions::Confirm::getControlValues ( GDialogPanel& dlg )
{
   current.prgExit = dlg.getComponentBoolValue("Exit");
   current.delFile = dlg.getComponentBoolValue("DelFile");
   current.delFileInSubDirs = dlg.getComponentBoolValue("DelFileInSubDirs");
   current.delEmptyDir = dlg.getComponentBoolValue("DelDir");
   current.delDirContainingFiles = dlg.getComponentBoolValue("DelNonEmptyDir");
   current.delHidSysReadFile = dlg.getComponentBoolValue("DelHidSysReadOnly");
   current.overwriteFileOrDir = dlg.getComponentBoolValue("Overwrite");
}

bool LCmdDlgOptions::Confirm::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         backup = current;
         setControlValues(dlg);
         return true;
      }

      case GM_CTRLCHANGED: 
      {
         GString id = msg.getParam1String();
         if (id == "Exit" ||
             id == "DelFile" ||
             id == "DelFileInSubDirs" ||
             id == "DelDir" ||
             id == "DelNonEmptyDir" ||
             id == "DelHidSysReadOnly" ||
             id == "Overwrite")
         {
            getControlValues(dlg);
         }
         return true; 
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            current = backup;
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current.restoreToDefault();
            setControlValues(dlg);
            return true;
         }
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::SaveOpt::SaveOpt ( LCmdOptions::SaveOptOnExit& current )
                        :current(current),
                         backup(current)
{
}

void LCmdDlgOptions::SaveOpt::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("101", current.everything, false);
   dlg.setComponentValue("102", current.currentDirs, false);
   dlg.setComponentValue("103", current.storedDirs, false);
   dlg.setComponentValue("104", current.curSelFiles, false);
   dlg.setComponentValue("105", current.visibleState, false);
   dlg.setComponentValue("106", current.panelModeSort, false);
   dlg.setComponentValue("107", current.frameWinPos, false);
   dlg.setComponentValue("108", current.otherOptions, false);
   dlg.setComponentValue("109", current.colors, false);
   dlg.setComponentValue("110", current.fonts, false);
}

void LCmdDlgOptions::SaveOpt::getControlValues ( GDialogPanel& dlg )
{
   current.everything = dlg.getComponentBoolValue("101");
   current.currentDirs = dlg.getComponentBoolValue("102");
   current.storedDirs = dlg.getComponentBoolValue("103");
   current.curSelFiles = dlg.getComponentBoolValue("104");
   current.visibleState = dlg.getComponentBoolValue("105");
   current.panelModeSort = dlg.getComponentBoolValue("106");
   current.frameWinPos = dlg.getComponentBoolValue("107");
   current.otherOptions = dlg.getComponentBoolValue("108");
   current.colors = dlg.getComponentBoolValue("109");
   current.fonts = dlg.getComponentBoolValue("110");
}

bool LCmdDlgOptions::SaveOpt::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           backup = current;
           setControlValues(dlg);
           // Force GM_CTRLCHANGED in order to set initial enable/disable state.
           dlg.getComponentByID("101").sendCtrlChanged();
           return true;

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           getControlValues(dlg);
           if (id == "101") // everything
           {
              dlg.setComponentEnabled("102", !current.everything);
              dlg.setComponentEnabled("103", !current.everything);
              dlg.setComponentEnabled("104", !current.everything);
              dlg.setComponentEnabled("105", !current.everything);
              dlg.setComponentEnabled("106", !current.everything);
              dlg.setComponentEnabled("107", !current.everything);
              dlg.setComponentEnabled("108", !current.everything);
              dlg.setComponentEnabled("109", !current.everything);
              dlg.setComponentEnabled("110", !current.everything);
           }
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              current = backup;
              setControlValues(dlg);
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.restoreToDefault();
              setControlValues(dlg);
              return true;
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::FileCopy::FileCopy ()
                         :current(lcmd->options.fileCopy),
                          backup(current)
{
}

void LCmdDlgOptions::FileCopy::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("ShowStatistics", current.showStatistics, false);
   dlg.setComponentValue("WaitWhenFinished", current.waitOnFinish, false);
   dlg.setComponentValue("WarnIfNotEnoughTargetSpace", current.warnIfNotEnoughTargetSpace, false);
   dlg.setComponentValue("PreallocSpace", current.preallocSpace, false);
   dlg.setComponentValue("PreallocSpaceOnDrives", current.preallocSpaceOnDrives, false);
   dlg.setComponentValue("PreallocIfNotLargerThan", current.maxPreallocateSize/(1024L*1024L), false);
   dlg.setComponentValue("MaxBufferSize", current.maxBuffSize, false);
}

void LCmdDlgOptions::FileCopy::getControlValues ( GDialogPanel& dlg )
{
   current.showStatistics = dlg.getComponentBoolValue("ShowStatistics");
   current.waitOnFinish = dlg.getComponentBoolValue("WaitWhenFinished");
   current.warnIfNotEnoughTargetSpace = dlg.getComponentBoolValue("WarnIfNotEnoughTargetSpace");
   current.setPreallocSpaceFromInt(dlg.getComponentIntValue("PreallocSpace"));
   current.preallocSpaceOnDrives = dlg.getComponentValue("PreallocSpaceOnDrives");
   current.maxPreallocateSize = dlg.getComponentLongValue("PreallocIfNotLargerThan") * 1024L * 1024L;
   if (current.maxPreallocateSize < 0)
      current.maxPreallocateSize = 0;
   current.maxBuffSize = dlg.getComponentIntValue("MaxBufferSize");
   if (current.maxBuffSize < 1)
      current.maxBuffSize = 1;
   else
   if (current.maxBuffSize > 65536) // 64 MBytes
      current.maxBuffSize = 65536;
}

void LCmdDlgOptions::FileCopy::updateEnabledStateOfDrivesEntry ( GDialogPanel& dlg )
{
   bool enableDrivesEntry = (current.preallocSpace == LCmdOptions::FileCopyOpts::SpecifiedDrives);
   bool enableMaxSizeEntry = (current.preallocSpace != LCmdOptions::FileCopyOpts::Never);
   dlg.setComponentEnabled("PreallocSpaceOnDrives", enableDrivesEntry);
   dlg.setComponentVisible("PreallocSpaceOnDrivesLeader", enableDrivesEntry);
   dlg.setComponentVisible("PreallocSpaceOnDrives", enableDrivesEntry);
   dlg.setComponentVisible("PreallocSpaceOnDrivesEg", enableDrivesEntry);
   dlg.setComponentVisible("PreallocIfNotLargerThan", enableMaxSizeEntry);
   dlg.setComponentVisible("PreallocIfNotLargerThanLeader", enableMaxSizeEntry);
   dlg.setComponentVisible("PreallocIfNotLargerThanMB", enableMaxSizeEntry);
}

bool LCmdDlgOptions::FileCopy::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         backup = current;
         setControlValues(dlg);
         updateEnabledStateOfDrivesEntry(dlg);
         return true;
      }

      case GM_CTRLCHANGED: 
      {
         GString compID = msg.getParam1String();
         getControlValues(dlg);
         if (compID == "PreallocSpace" || compID == "PreallocIfNotLargerThan")
            updateEnabledStateOfDrivesEntry(dlg);
         return true; 
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            current = backup;
            setControlValues(dlg);
            updateEnabledStateOfDrivesEntry(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current.restoreToDefault();
            setControlValues(dlg);
            updateEnabledStateOfDrivesEntry(dlg);
            return true;
         }
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::ConVarious::ConVarious ( LCmdOptions::ConVariousOpts& current )
                           :current(current),
                            backup(current)
{
}

void LCmdDlgOptions::ConVarious::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("CloseOnExit", current.defaultCloseConOnExit, false);
   dlg.setComponentValue("RunConAs", current.startConFullScreen ? 1 : 0, false);
   dlg.setComponentValue("RunDosAs", current.startDosFullScreen ? 1 : 0, false);
}

void LCmdDlgOptions::ConVarious::getControlValues ( GDialogPanel& dlg )
{
   current.defaultCloseConOnExit = dlg.getComponentBoolValue("CloseOnExit");
   current.startConFullScreen = (dlg.getComponentIntValue("RunConAs") == 1);
   current.startDosFullScreen = (dlg.getComponentIntValue("RunDosAs") == 1);
}

bool LCmdDlgOptions::ConVarious::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
         backup = current;
         setControlValues(dlg);
         return true;

      case GM_CTRLCHANGED: 
         getControlValues(dlg);
         return true;

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            current = backup;
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current.restoreToDefault();
            setControlValues(dlg);
            return true;
         }
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::VfsVarious::VfsVarious ( LCmdOptions::VfsVariousOpts& current )
                           :current(current),
                            backup(current)
{
}

void LCmdDlgOptions::VfsVarious::setControlValues ( GDialogPanel& dlg )
{
}

void LCmdDlgOptions::VfsVarious::getControlValues ( GDialogPanel& dlg )
{
}

bool LCmdDlgOptions::VfsVarious::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
         backup = current;
         setControlValues(dlg);
         return true;

      case GM_CTRLCHANGED: 
         getControlValues(dlg);
         return true;

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            current = backup;
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current.restoreToDefault();
            setControlValues(dlg);
            return true;
         }
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::VfsZip::VfsZip ( LCmdOptions::VfsZipOpts& current )
                       :current(current),
                        backup(current)
{
}

void LCmdDlgOptions::VfsZip::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("SysLaunch", current.sysLaunchZipFiles, false);
   dlg.setComponentValue("ToolPath", current.toolPath, false);
   dlg.setComponentValue("ArgsDelete", current.argsDelete, false);
}

void LCmdDlgOptions::VfsZip::getControlValues ( GDialogPanel& dlg )
{
   current.sysLaunchZipFiles = dlg.getComponentBoolValue("SysLaunch");
   current.toolPath = dlg.getComponentValue("ToolPath");
   current.argsDelete = dlg.getComponentValue("ArgsDelete");
}

bool LCmdDlgOptions::VfsZip::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
         backup = current;
         setControlValues(dlg);
         return true;

      case GM_CTRLCHANGED: 
         getControlValues(dlg);
         return true;

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "DLG_UNDO")
         {
            current = backup;
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            current.restoreToDefault();
            setControlValues(dlg);
            return true;
         }
         else
         if (cmdID == "BrowseToolPath")
         {
            GString titleText = "%DlgOptVfsZip_TitleChooseTool";
            GString path = dlg.userChooseFile(titleText, current.toolPath);
            if (path != "")
               dlg.setComponentValue("ToolPath", path);
         }
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::Various::Various ( LCmdOptions::VariousOpts& current )
                        :current(current),
                         backup(current)
{
}

void LCmdDlgOptions::Various::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("AddToItemHeight", current.addToItemHeight, false);
   dlg.setComponentValue("AddToItemWidth", current.addToItemWidth, false);
   dlg.setComponentValue("DelayShowTooltipMillis", current.delayShowTooltipMillis, false);
   dlg.setComponentValue("AutoHideTooltipMillis", current.autoHideTooltipMillis, false);
}

void LCmdDlgOptions::Various::getControlValues ( GDialogPanel& dlg )
{
   current.addToItemHeight = dlg.getComponentIntValue("AddToItemHeight");
   current.addToItemWidth = dlg.getComponentIntValue("AddToItemWidth");
   current.delayShowTooltipMillis = dlg.getComponentIntValue("DelayShowTooltipMillis");
   current.autoHideTooltipMillis = dlg.getComponentIntValue("AutoHideTooltipMillis");

   // Activate the tooltip settings.
   GTooltip::SetDefaultDelayShowMillis(current.delayShowTooltipMillis);
   GTooltip::SetDefaultAutoHideMillis(current.autoHideTooltipMillis);

   // Recaclculate all items simply by emulating that the
   // font has been changed.
   LCmdFilePanel& curPanel = LCmdFilePanel::GetCurrentPanel();
   LCmdFilePanelModeAbstract& curView = curPanel.getCurrentView();
   if (curPanel.items.getCount() > 0)
      curPanel.items[0].briefWidth = 0; // Force layout() to recalculate item widths.
   curView.layout();

   // Recalculate all items of the opposite file panel too.
   LCmdFilePanel& oppositePanel = curPanel.getOppositePanel();
   LCmdFilePanelModeAbstract& oppositeCurView = oppositePanel.getCurrentView();
   if (oppositePanel.items.getCount() > 0)
      oppositePanel.items[0].briefWidth = 0; // Force layout() to recalculate item widths.
   oppositeCurView.layout();
}

bool LCmdDlgOptions::Various::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           backup = current;
           setControlValues(dlg);
           dlg.setNoInitCtrlChanged(true);
           return true;

      case GM_CTRLCHANGED: {
           getControlValues(dlg);
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              current = backup;
              setControlValues(dlg);
              getControlValues(dlg); // Recalculate display.
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.restoreToDefault();
              setControlValues(dlg);
              getControlValues(dlg); // Recalculate display.
              return true;
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::FileDel::FileDel ( LCmdOptions::FileDelOpts& current )
                        :current(current),
                         backup(current)
{
}

void LCmdDlgOptions::FileDel::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("101", current.defaultDeleteToTrashCan, false);
   dlg.setComponentValue("102", current.useDelKey, false);
   dlg.setComponentValue("103", current.showDirsProgressOnly, false);
}

void LCmdDlgOptions::FileDel::getControlValues ( GDialogPanel& dlg )
{
   current.defaultDeleteToTrashCan = dlg.getComponentBoolValue("101");
   current.useDelKey = dlg.getComponentBoolValue("102");
   current.showDirsProgressOnly = dlg.getComponentBoolValue("103");
}

bool LCmdDlgOptions::FileDel::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           backup = current;
           setControlValues(dlg);
           return true;

      case GM_CTRLCHANGED: {
           getControlValues(dlg);
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              current = backup;
              setControlValues(dlg);
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.restoreToDefault();
              setControlValues(dlg);
              return true;
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::DirCache::DirCache ( LCmdOptions::DirCacheOpts& current )
                         :current(current),
                          backup(current)
{
}

void LCmdDlgOptions::DirCache::setControlValues ( GDialogPanel& dlg )
{
   dlg.setComponentValue("AddDirsDuringWalk", current.addDirsDuringWalk, false);
   dlg.setComponentValue("AutoStripDirsAfterDirDeletions", current.autoStripDirsAfterDirDeletions, false);
   dlg.setComponentValue("IncludeVfsDirs", current.includeVfsDirs, false);
}

void LCmdDlgOptions::DirCache::getControlValues ( GDialogPanel& dlg )
{
   current.addDirsDuringWalk = dlg.getComponentBoolValue("AddDirsDuringWalk");
   current.autoStripDirsAfterDirDeletions = dlg.getComponentBoolValue("AutoStripDirsAfterDirDeletions");
   current.includeVfsDirs = dlg.getComponentBoolValue("IncludeVfsDirs");
}

bool LCmdDlgOptions::DirCache::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           backup = current;
           setControlValues(dlg);
           return true;

      case GM_CTRLCHANGED: {
           getControlValues(dlg);
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              current = backup;
              setControlValues(dlg);
              return true;
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.restoreToDefault();
              setControlValues(dlg);
              return true;
           }
           return true; }

      default:
           return true;
   }
}

void LCmdDlgOptions::FontCmn::setControlValues ( GDialogPanel& dlg )
{
   GWindow& winTxtToolbar = dlg.getComponentByID("TxtToolbar");
   GWindow& winTxtConMon = dlg.getComponentByID("TxtConMon");
   GWindow& winTxtPrompt = dlg.getComponentByID("TxtPrompt");
   GWindow& winTxtCmdLine = dlg.getComponentByID("TxtCmdLine");
   GWindow& winTxtKeybar = dlg.getComponentByID("TxtKeybar");
   GWindow& winTxtStatusbar = dlg.getComponentByID("TxtStatusbar");

   GToolbar* toolbar = lcmd->mainWin.getToolbar();
   GKeyBar* keybar = lcmd->mainWin.getKeybar();
   GStatusbar* statbar = lcmd->mainWin.getStatusbar();

   GString fontToolbar = (toolbar == null ? GString::Empty : toolbar->getFontNameSize());
   GString fontConMon = lcmd->cmdCont.conmon.getFontNameSize();
   GString fontPrompt = lcmd->cmdLine.prompt.getFontNameSize();
   GString fontCmdLine = lcmd->cmdLine.entry.getFontNameSize();
   GString fontKeybar = (keybar == null ? GString::Empty : keybar->getElement(0).getFontNameSize());
   GString fontStatusbar = (statbar == null ? GString::Empty : statbar->getFontNameSize());

   winTxtToolbar.setFontNameSize(fontToolbar);
   winTxtConMon.setFontNameSize(fontConMon);
   winTxtPrompt.setFontNameSize(fontPrompt);
   winTxtCmdLine.setFontNameSize(fontCmdLine);
   winTxtKeybar.setFontNameSize(fontKeybar);
   winTxtStatusbar.setFontNameSize(fontStatusbar);

   winTxtToolbar.setText(fontToolbar);
   winTxtConMon.setText(fontConMon);
   winTxtPrompt.setText(fontPrompt);
   winTxtCmdLine.setText(fontCmdLine);
   winTxtKeybar.setText(fontKeybar);
   winTxtStatusbar.setText(fontStatusbar);
}

bool LCmdDlgOptions::FontCmn::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: 
      {
         GToolbar* toolbar = lcmd->mainWin.getToolbar();
         GKeyBar* keybar = lcmd->mainWin.getKeybar();
         GStatusbar* statbar = lcmd->mainWin.getStatusbar();

         backup.fontToolBar = (toolbar == null ? GString::Empty : toolbar->getFontNameSize());
         backup.fontConMon = lcmd->cmdCont.conmon.getFontNameSize();
         backup.fontPrompt = lcmd->cmdLine.prompt.getFontNameSize();
         backup.fontCmdLine = lcmd->cmdLine.entry.getFontNameSize();
         backup.fontStatusBar = (statbar == null ? GString::Empty: statbar->getFontNameSize());

         // ---
         backup.fontKeyBarButtons.removeAll();
         if (keybar != null)
         {
            const int num = keybar->getElementCount();
            for (int i=0; i<num; i++)
            {
               GString fontName = keybar->getElement(i).getFontNameSize();
               GString* obj = new GString(fontName);
               backup.fontKeyBarButtons.add(obj);
            }
         }

         setControlValues(dlg);
         return true; 
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "CmdToolbar") // Toolbar
         {
            GToolbar* toolbar = lcmd->mainWin.getToolbar();
            if (toolbar != null)
               toolbar->userChooseFont("%DlgOptFontCmn_TitleChooseToolbar", true, &dlg.getOwnerFrame());
         }
         else
         if (cmdID == "CmdConMon") // Console Monitor
         {
            lcmd->cmdCont.conmon.userChooseFont("%DlgOptFontCmn_TitleChooseConMon", true, &dlg.getOwnerFrame(), GString::Empty, true);
         }
         else
         if (cmdID == "CmdPrompt") // Prompt
         {
            lcmd->cmdLine.prompt.userChooseFont("%DlgOptFontCmn_TitleChoosePrompt", true, &dlg.getOwnerFrame());
         }
         else
         if (cmdID == "CmdCmdLine") // Command Line
         {
            lcmd->cmdLine.entry.userChooseFont("%DlgOptFontCmn_TitleChooseCmdLine", true, &dlg.getOwnerFrame());
         }
         else
         if (cmdID == "CmdKeybar") // Keybar
         {
            GKeyBar* keybar = lcmd->mainWin.getKeybar();
            if (keybar != null)
            {
               GString font = keybar->userChooseFont("%DlgOptFontCmn_TitleChooseKeybar", true, &dlg.getOwnerFrame());
               if (font != "")
                  keybar->setFontNameSizeOnAllElements(font);
            }
         }
         else
         if (cmdID == "CmdStatusbar") // Statusbar
         {
            GStatusbar* statbar = lcmd->mainWin.getStatusbar();
            if (statbar != null)
               statbar->userChooseFont("%DlgOptFontCmn_TitleChooseStatusbar", true, &dlg.getOwnerFrame());
         }
         else
         if (cmdID == "DLG_UNDO")
         {
            GToolbar* toolbar = lcmd->mainWin.getToolbar();
            GKeyBar* keybar = lcmd->mainWin.getKeybar();
            GStatusbar* statbar = lcmd->mainWin.getStatusbar();

            // ---
            if (toolbar != null)
               toolbar->setFontNameSize(backup.fontToolBar);
            lcmd->cmdCont.conmon.setFontNameSize(backup.fontConMon);
            lcmd->cmdLine.prompt.setFontNameSize(backup.fontPrompt);
            lcmd->cmdLine.entry.setFontNameSize(backup.fontCmdLine);
            if (statbar != null)
               statbar->setFontNameSize(backup.fontStatusBar);

            // ---
            if (keybar != null)
            {
               const int num = keybar->getElementCount();
               for (int i=0; i<num; i++)
               {
                  GString fontName = backup.fontKeyBarButtons[i];
                  keybar->getElement(i).setFontNameSize(fontName);
               }
            }
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            GWindow* titlebar = lcmd->mainWin.getTitlebar();
            GMenu* menubar = lcmd->mainWin.getMenubar();
            GToolbar* toolbar = lcmd->mainWin.getToolbar();
            GKeyBar* keybar = lcmd->mainWin.getKeybar();
            GStatusbar* statbar = lcmd->mainWin.getStatusbar();

            if (titlebar != null)
               titlebar->setFontNameSize(lcmd->options.DefaultFontFrameTitlebar);
            if (menubar != null)
               menubar->setFontNameSize(lcmd->options.DefaultFontFrameMenubar);
            if (toolbar != null)
               toolbar->setFontNameSize(GAbstractToolbarWindow::DefaultFont);
            lcmd->cmdCont.conmon.setFontNameSize(lcmd->options.DefaultFontConMon);
            lcmd->cmdLine.prompt.setFontNameSize(lcmd->options.DefaultFontPrompt);
            lcmd->cmdLine.entry.setFontNameSize(lcmd->options.DefaultFontCmdLineEntry);
            if (keybar != null)
            {
               keybar->setFontNameSize(GKeyBar::DefaultFont);
               keybar->setFontNameSizeOnAllElements(GKeyBar::DefaultFont);
            }
            if (statbar != null)
               statbar->setFontNameSize(GWindow::SysDefFontNameSize);
         }
         setControlValues(dlg);
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::Font::Font ( FilePanelID fpid )
                     :fpid(fpid),
                      fpanel(null)
{
}

void LCmdDlgOptions::Font::setControlValues ( GDialogPanel& dlg )
{
   GWindow& winTxtHeaderbar = dlg.getComponentByID("TxtHeaderbar");
   GWindow& winTxtBrief = dlg.getComponentByID("TxtBrief");
   GWindow& winTxtWide = dlg.getComponentByID("TxtWide");
   GWindow& winTxtFull = dlg.getComponentByID("TxtFull");
#if __ENABLE_TREE_VIEW
   GWindow& winTxtTree = dlg.getComponentByID("TxtTree");
#endif
   GWindow& winTxtInfo = dlg.getComponentByID("TxtInfo");
   GWindow& winTxtInfobar = dlg.getComponentByID("TxtInfobar");

   GString fontHeaderbar(fpanel->headerWin.getFontNameSize());
   GString fontBrief(fpanel->viewBrief.getFontNameSize());
   GString fontWide(fpanel->viewWide.getFontNameSize());
   GString fontFull(fpanel->viewFull.getFontNameSize());
#if __ENABLE_TREE_VIEW
   GString fontTree(fpanel->viewTree.getFontNameSize());
#endif
   GString fontInfo(fpanel->viewInfo.getFontNameSize());
   GString fontInfobar(fpanel->infoBar.getFontNameSize());

   winTxtHeaderbar.setFontNameSize(fontHeaderbar);
   winTxtBrief.setFontNameSize(fontBrief);
   winTxtWide.setFontNameSize(fontWide);
   winTxtFull.setFontNameSize(fontFull);
#if __ENABLE_TREE_VIEW
   winTxtTree.setFontNameSize(fontTree);
#endif
   winTxtInfo.setFontNameSize(fontInfo);
   winTxtInfobar.setFontNameSize(fontInfobar);

   winTxtHeaderbar.setText(fontHeaderbar);
   winTxtBrief.setText(fontBrief);
   winTxtWide.setText(fontWide);
   winTxtFull.setText(fontFull);
#if __ENABLE_TREE_VIEW
   winTxtTree.setText(fontTree);
#endif
   winTxtInfo.setText(fontInfo);
   winTxtInfobar.setText(fontInfobar);
}

bool LCmdDlgOptions::Font::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: 
      {
         fpanel = (fpid == LEFT ? lcmd->fp1 : lcmd->fp2);
         backup.fontHeaderBar = fpanel->headerWin.getFontNameSize();
         backup.fontBrief = fpanel->viewBrief.getFontNameSize();
         backup.fontWide = fpanel->viewWide.getFontNameSize();
         backup.fontFull = fpanel->viewFull.getFontNameSize();
#if __ENABLE_TREE_VIEW
         backup.fontTree = fpanel->viewTree.getFontNameSize();
#endif
         backup.fontInfo = fpanel->viewInfo.getFontNameSize();
         backup.fontInfoBar = fpanel->infoBar.getFontNameSize();
         setControlValues(dlg);
         return true; 
      }

      case GM_DISMISSDIALOG:
      {
         fpanel = null;
         return true;
      }

      case GM_COMMAND: 
      {
         GString cmdID = msg.getParam1String();
         if (cmdID == "CmdHeaderbar") // HeaderBar
         {
            fpanel->headerWin.userChooseFont("%DlgOptFont_TitleChooseHeaderbar", true, &dlg.getOwnerFrame());
         }
         else
         if (cmdID == "CmdBrief") // Brief
         {
            LCmdFilePanelViewOptions::VIEWMODE viewMode = fpanel->view.viewMode;
            fpanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_BRIEF);
            fpanel->viewBrief.userChooseFont("%DlgOptFont_TitleChooseBrief", true, &dlg.getOwnerFrame());
            fpanel->setViewMode(viewMode);
         }
         if (cmdID == "CmdWide") // Wide
         {
            LCmdFilePanelViewOptions::VIEWMODE viewMode = fpanel->view.viewMode;
            fpanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_WIDE);
            fpanel->viewWide.userChooseFont("%DlgOptFont_TitleChooseWide", true, &dlg.getOwnerFrame());
            fpanel->setViewMode(viewMode);
         }
         else
         if (cmdID == "CmdFull") // Full
         {
            LCmdFilePanelViewOptions::VIEWMODE viewMode = fpanel->view.viewMode;
            fpanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_FULL);
            fpanel->viewFull.userChooseFont("%DlgOptFont_TitleChooseFull", true, &dlg.getOwnerFrame());
            fpanel->setViewMode(viewMode);
         }
         else
#if __ENABLE_TREE_VIEW
         if (cmdID == "CmdTree") // Tree
         {
            LCmdFilePanelViewOptions::VIEWMODE viewMode = fpanel->view.viewMode;
            fpanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_TREE);
            fpanel->viewTree.userChooseFont("%DlgOptFont_TitleChooseTree", true, &dlg.getOwnerFrame());
            fpanel->setViewMode(viewMode);
         }
         else
#endif
         if (cmdID == "CmdInfo") // Info
         {
            LCmdFilePanelViewOptions::VIEWMODE viewMode = fpanel->view.viewMode;
            fpanel->setViewMode(LCmdFilePanelViewOptions::VIEWMODE_INFO);
            fpanel->viewInfo.userChooseFont("%DlgOptFont_TitleChooseInfo", true, &dlg.getOwnerFrame());
            fpanel->setViewMode(viewMode);
         }
         else
         if (cmdID == "CmdInfobar") // InfoBar
         {
            fpanel->infoBar.userChooseFont("%DlgOptFont_TitleChooseInfobar", true, &dlg.getOwnerFrame());
         }
         else
         if (cmdID == "DLG_UNDO")
         {
            fpanel->headerWin.setFontNameSize(backup.fontHeaderBar);
            fpanel->viewBrief.setFontNameSize(backup.fontBrief);
            fpanel->viewWide.setFontNameSize(backup.fontWide);
            fpanel->viewFull.setFontNameSize(backup.fontFull);
#if __ENABLE_TREE_VIEW
            fpanel->viewTree.setFontNameSize(backup.fontTree);
#endif
            fpanel->viewInfo.setFontNameSize(backup.fontInfo);
            fpanel->infoBar.setFontNameSize(backup.fontInfoBar);
         }
         else
         if (cmdID == "DLG_DEFAULT")
         {
            fpanel->headerWin.setFontNameSize(lcmd->options.DefaultFontFilePanelHeader);
            fpanel->viewBrief.setFontNameSize(GWindow::SysDefFontNameSize);
            fpanel->viewWide.setFontNameSize(GWindow::SysDefFontNameSize);
            fpanel->viewFull.setFontNameSize(GWindow::SysDefFontNameSize);
#if __ENABLE_TREE_VIEW
            fpanel->viewTree.setFontNameSize(GWindow::SysDefFontNameSize);
#endif
            fpanel->viewInfo.setFontNameSize(GWindow::SysDefFontNameSize);
            fpanel->infoBar.setFontNameSize(GWindow::SysDefFontNameSize);
         }
         setControlValues(dlg);
         return true; 
      }

      default:
         return true;
   }
}

LCmdDlgOptions::Programs::Programs ( LCmdOptions::ProgOption& current )
                         :current(current),
                          backup(current)
{
}

void LCmdDlgOptions::Programs::setControlValues ( GDialogPanel& dlg )
{
   if (!current.internalAvail)
   {
      current.useExternal = true;
      dlg.setComponentEnabled("103", false);
   }
   dlg.setComponentValue("103", current.useExternal, false);
   dlg.setComponentValue("101", current.progName, false);
   dlg.setComponentValue("113", current.params, false);
   dlg.setComponentValue("111", current.workDir, false);
}

bool LCmdDlgOptions::Programs::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
           backup = current;
           setControlValues(dlg);
           return true;

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           if (id == "101") // The program module name entry field
              current.progName = dlg.getComponentValue("101");
           else
           if (id == "113") // The program parameter string
              current.params = dlg.getComponentValue("113");
           else
           if (id == "111") // The working directory string
              current.workDir = dlg.getComponentValue("111");
           else
           if (id == "103") // Toggle Button: Use external ...
              current.useExternal = dlg.getComponentBoolValue("103");
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "102") // Browse for the program file module ...
           {
              GString titleText = "%DlgOptProg_TitleChooseProgram";
              GString path = dlg.userChooseFile(titleText, current.progName);
              if (path != "")
                 dlg.setComponentValue("101", path);
           }
           else
           if (cmdID == "DLG_UNDO")
           {
              current.internalAvail = backup.internalAvail;
              current.useExternal = backup.useExternal;
              current.progName = backup.progName;
              current.params = backup.params;
              current.workDir = backup.workDir;
              setControlValues(dlg);
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              current.restoreToDefault();
              setControlValues(dlg);
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::Commands::Commands ()
                         :bckEnabledInternal(50),
                          bckEnabledCmdExe(50)
{
}

bool LCmdDlgOptions::Commands::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: {
           bckEnabledInternal.removeAll();
           bckEnabledCmdExe.removeAll();

           // Fill in with internal commands into the two list boxes.
           GListBox& l101 = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
           GListBox& l103 = dynamic_cast<GListBox&>(dlg.getComponentByID("103"));
           l101.removeAllItems();
           l103.removeAllItems();
           l101.enterCtrlChangedFilter();
           l103.enterCtrlChangedFilter();
           GKeyBag<InternalCommand>& bag = lcmd->cmdLine.entry.internalCommands;
           const int num = bag.getCount();
           for (int i=0; i<num; i++)
           {
              InternalCommand& cmd = bag.getIndexedItem(i);
              bool shell = cmd.isTransferToShell();
              GListBox& list = shell ? l103 : l101;
              GString text = cmd.getCommandStr();
              list.addItem(text/*, iconName, null, false, null*/);
              const int count = list.getItemCount();
              bool on = !cmd.isDisabled();
              list.setItemSelected(count-1, on);
              if (shell)
                 bckEnabledCmdExe.add(new GBoolean(on));
              else
                 bckEnabledInternal.add(new GBoolean(on));
           }
           l101.exitCtrlChangedFilter();
           l103.exitCtrlChangedFilter();
           dlg.setNoInitCtrlChanged(true);
           return true; }

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           if (id != "101" && id != "103")
              return false;
           // Loop through all items in the list, in case CTRL+/ or CTRL+\ was
           // used to select/deselect all.
           GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID(id));
           const int num = list.getItemCount();
           GArray<GBoolean> sel(num);
           for (int i=0; i<num; i++)
           {
              bool on = list.isItemSelected(i);
              sel.add(new GBoolean(on));
           }

           // Toggle the commands acordingly.
           GKeyBag<InternalCommand>& bag = lcmd->cmdLine.entry.internalCommands;
           for (int i=0; i<num; i++)
           {
              GString cmdName = list.getItemText(i);
              int bagIndex = bag.findEntry(cmdName, null);
              InternalCommand& cmd = bag.getIndexedItem(bagIndex);
              bool on = sel[i].boolValue();
              cmd.setDisabled(!on);
           }
           return true; }

      case GM_COMMAND: {
           GString cmdID = msg.getParam1String();
           if (cmdID == "DLG_UNDO")
           {
              GListBox& l101 = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
              for (int i=0; i<bckEnabledInternal.getCount(); i++)
                 l101.setItemSelected(i, bckEnabledInternal[i].boolValue());

              GListBox& l103 = dynamic_cast<GListBox&>(dlg.getComponentByID("103"));
              for (int i=0; i<bckEnabledCmdExe.getCount(); i++)
                 l103.setItemSelected(i, bckEnabledCmdExe[i].boolValue());
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              GListBox& l101 = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
              for (int i=0; i<bckEnabledInternal.getCount(); i++)
                 l101.setItemSelected(i, true);

              GListBox& l103 = dynamic_cast<GListBox&>(dlg.getComponentByID("103"));
              for (int i=0; i<bckEnabledCmdExe.getCount(); i++)
                 l103.setItemSelected(i, true);
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::StrBagEd::StrBagEd ( GKeyBag<GString>& current )
                         :current(current),
                          backup(current.getCount(), 
                          -3, 
                          current.isIgnoreCase())
{
}

void LCmdDlgOptions::StrBagEd::setControlValues ( GDialogPanel& dlg )
{
   GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
   const int num = current.getCount();
   for (int i=0; i<num; i++)
   {
      GString item = current.getKey(i);
      item += "=" + current.getIndexedItem(i);
      list.addItem(item);
   }
   list.setSelectedIndex(num > 0 ? 0 : -1);
}

bool LCmdDlgOptions::StrBagEd::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: {
           // Initialize the backup.
           backup.removeAll();
           const int num = current.getCount();
           for (int i=0; i<num; i++)
           {
              GString* val = new GString(current.getIndexedItem(i));
              const GString& key = current.getKey(i);
              backup.put(key, val);
           }

           // ---
           dlg.setNoInitCtrlChanged(true);
           setControlValues(dlg);
           dlg.setComponentEnabled("DLG_DEFAULT", false);
           dlg.setComponentEnabled("106", num >= 1); // Remove
           dlg.setComponentEnabled("107", num >= 1); // Update
           return true; }

      case GM_DISMISSDIALOG:
           backup.removeAll();
           return true;

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           if (id == "101") // The strings list box.
           {
              GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
              dlg.setComponentEnabled("107", false); // Update
              int itemIndex = list.getSelectedIndex();
              if (itemIndex >= 0 && itemIndex < current.getCount())
              {
                 const GString& name = current.getKey(itemIndex);
                 const GString& value = current.getIndexedItem(itemIndex);
                 dlg.setComponentValue("102", name, false);
                 dlg.setComponentValue("104", value, false);
                 dlg.setComponentEnabled("106", true); // Remove
              }
              else
              {
                 dlg.setComponentValue("102", "", false);
                 dlg.setComponentValue("104", "", false);
                 dlg.setComponentEnabled("106", false); // Remove
              }
           }
           else
           if (id == "102") // The string name entry field.
           {
              dlg.setComponentEnabled("107", true); // Update
           }
           else
           if (id == "104") // The string value entry field
           {
              dlg.setComponentEnabled("107", true); // Update
           }
           return true; }

      case GM_COMMAND: {
           GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
           GString cmdID = msg.getParam1String();
           if (cmdID == "106") // Push-button: "Remove"
           {
              int itemIndex = list.getSelectedIndex();
              if (itemIndex >= 0 && itemIndex < current.getCount())
              {
                 const GString& key = current.getKey(itemIndex);
                 current.remove(key);
                 list.removeItem(itemIndex);
                 if (itemIndex == current.getCount())
                    itemIndex -= 1;
                 if (itemIndex >= 0)
                 {
                    list.setSelectedIndex(itemIndex);
                 }
                 else
                 {
                    dlg.setComponentValue("102", "");
                    dlg.setComponentValue("104", "");
                    dlg.setComponentEnabled("106", false); // Remove
                    dlg.setComponentEnabled("107", false); // Update
                 }
              }
           }
           else
           if (cmdID == "107") // Push-button: "Update".
           {
              int itemIndex;
              GString name = dlg.getComponentValue("102");
              name.toUpperCase();
              GString value = dlg.getComponentValue("104");
              GString item(name.length() + value.length() + 1);
              item += name;
              item += "=";
              item += value;
              if (current.containsKey(name))
              {
                 GString* val = new GString(value);
                 current.update(name, val);
                 itemIndex = current.findEntry(name);
                 list.setItemText(itemIndex, item);
              }
              else
              {
                 GString* val = new GString(value);
                 current.put(name, val);
                 itemIndex = current.findEntry(name);
                 list.insertItem(itemIndex, item);
              }
              if (itemIndex >= 0)
                 list.setSelectedIndex(itemIndex);
           }
           else
           if (cmdID == "DLG_UNDO")
           {
              list.removeAllItems();
              current.removeAll();
              const int num = backup.getCount();
              for (int i=0; i<num; i++)
              {
                 GString* val = new GString(backup.getIndexedItem(i));
                 const GString& key = backup.getKey(i);
                 current.put(key, val);
              }
              setControlValues(dlg);
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::SSA::SSA ( GKeyBag<LCmdSepSessionApps>& current )
                    :current(current),
                     backup(32, -3, current.isIgnoreCase())
{
}

bool LCmdDlgOptions::SSA::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG: {
           dlg.setNoInitCtrlChanged(true);

           // Fill in with SSA-apps into the list box, and copy
           // to the backup bag.
           backup.removeAll();
           GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
           list.removeAllItems();
           const int num = current.getCount();
           for (int i=0; i<num; i++)
           {
              LCmdSepSessionApps& cmd = current.getIndexedItem(i);
              const GString& key = current.getKey(i);
              list.addItem(key);
              LCmdSepSessionApps* bck = new LCmdSepSessionApps(cmd.viaMain, cmd.viaPipe);
              backup.put(key, bck);
           }

           // Set initial enabled state of some controls.
           dlg.setComponentEnabled("104", num >= 1);
           dlg.setComponentEnabled("105", num >= 1);
           dlg.setComponentEnabled("106", num >= 1);
           dlg.setComponentEnabled("107", num >= 1);
           return true; }

      case GM_DISMISSDIALOG:
           backup.removeAll();
           return true;

      case GM_CTRLCHANGED: {
           GString id = msg.getParam1String();
           if (id == "101") // The list box.
           {
              GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
              dlg.setComponentEnabled("107", false); // Update
              int itemIndex = list.getSelectedIndex();
              if (itemIndex >= 0 && itemIndex < current.getCount())
              {
                 const GString& name = current.getKey(itemIndex);
                 dlg.setComponentValue("102", name, false);
                 dlg.setComponentValue("105", current.getIndexedItem(itemIndex).viaMain, false);
                 dlg.setComponentValue("104", current.getIndexedItem(itemIndex).viaPipe, false);
                 dlg.setComponentEnabled("106", true); // Remove
                 dlg.setComponentEnabled("104", true);
                 dlg.setComponentEnabled("105", true);
              }
              else
              {
                 dlg.setComponentValue("102", "", false);
                 dlg.setComponentEnabled("106", false); // Remove
                 dlg.setComponentEnabled("104", false);
                 dlg.setComponentEnabled("105", false);
              }
           }
           else
           if (id == "102") // The string name entry field.
           {
              GString name = dlg.getComponentValue("102");
              bool isNew = !dlg.isComponentEnabled("104");
              if (isNew)
              {
                 dlg.setComponentValue("105", true);
                 dlg.setComponentValue("104", true);
              }
              dlg.setComponentEnabled("107", name != ""); // Update
              dlg.setComponentEnabled("104", name != "");
              dlg.setComponentEnabled("105", name != "");
           }
           else
           if (id == "104" || // Toggle button: viaPipe
               id == "105")   // Toggle button: viaMain
           {
              dlg.setComponentEnabled("107", true); // Update
           }
           return true; }

      case GM_COMMAND: {
           GListBox& list = dynamic_cast<GListBox&>(dlg.getComponentByID("101"));
           GString cmdID = msg.getParam1String();
           if (cmdID == "106") // Push-button: "Remove"
           {
              int itemIndex = list.getSelectedIndex();
              if (itemIndex >= 0 && itemIndex < current.getCount())
              {
                 const GString& key = current.getKey(itemIndex);
                 current.remove(key);
                 list.removeItem(itemIndex);
                 if (itemIndex == current.getCount())
                    itemIndex -= 1;
                 if (itemIndex >= 0)
                 {
                    list.setSelectedIndex(itemIndex);
                 }
                 else
                 {
                    dlg.setComponentValue("102", "");
                    dlg.setComponentEnabled("104", false);
                    dlg.setComponentEnabled("105", false);
                    dlg.setComponentEnabled("106", false);
                    dlg.setComponentEnabled("107", false);
                    dlg.setComponentValue("105", false);
                    dlg.setComponentValue("104", false);
                 }
              }
           }
           else
           if (cmdID == "107") // Push-button: "Update"
           {
              int itemIndex;
              GString name = dlg.getComponentValue("102");
              bool viaMain = dlg.getComponentBoolValue("105");
              bool viaPipe = dlg.getComponentBoolValue("104");
              if (current.containsKey(name))
              {
                 itemIndex = current.findEntry(name);
                 list.removeItem(itemIndex);
                 current.remove(name);
              }
              LCmdSepSessionApps* cmd = new LCmdSepSessionApps(viaMain, viaPipe);
              current.put(name, cmd);
              itemIndex = current.findEntry(name);
              list.insertItem(itemIndex, name);
              if (itemIndex >= 0)
                 list.setSelectedIndex(itemIndex);
           }
           else
           if (cmdID == "DLG_UNDO")
           {
              list.removeAllItems();
              current.removeAll();
              const int num = backup.getCount();
              for (int i=0; i<num; i++)
              {
                 LCmdSepSessionApps& bck = backup.getIndexedItem(i);
                 const GString& key = backup.getKey(i);
                 LCmdSepSessionApps* cmd = new LCmdSepSessionApps(bck.viaMain, bck.viaPipe);
                 current.put(key, cmd);
                 list.addItem(key);
              }
           }
           else
           if (cmdID == "DLG_DEFAULT")
           {
              list.removeAllItems();
              current.removeAll();
              for (int i=0; knownSSA[i].name != null; i++)
              {
                 const char *name = knownSSA[i].name;
                 LCmdSepSessionApps* obj = new LCmdSepSessionApps(knownSSA[i].viaMain, knownSSA[i].viaPipe);
                 current.put(name, obj);
                 int itemIndex = current.findEntry(name);
                 list.insertItem(itemIndex, name);
              }
           }
           return true; }

      default:
           return true;
   }
}

LCmdDlgOptions::LCmdDlgOptions ()
               :hasBeenExecuted(false),
                startup1(LEFT),
                startup2(RIGHT),
                filter1(LEFT),
                filter2(RIGHT),
                sort1(LEFT),
                sort2(RIGHT),
                viewCmn(),
                view1(LEFT),
                view2(RIGHT),
                viewStyle(),
                confirm(lcmd->options.confirm),
                saveOpt(lcmd->options.saveOnExit),
                fileCopy(),
                fileDel(lcmd->options.fileDel),
                dirCache(lcmd->options.dirCache),
                conVarious(lcmd->options.conVarious),
                vfsVarious(lcmd->options.vfsVarious),
                vfsZip(lcmd->options.vfsZip),
                various(lcmd->options.various),
                fontCmn(),
                font1(LEFT),
                font2(RIGHT),
                progEdit(lcmd->options.editor),
                progView(lcmd->options.viewer),
                progFFind(lcmd->options.ffind),
                progFDiff(lcmd->options.fdiff),
                progCClean(lcmd->options.cclean),
                progUndel(lcmd->options.undel),
                progEAEdit(lcmd->options.eas),
                commands(),
                env(lcmd->getEnvironmentVars().envBag),
                aliases(lcmd->cmdLine.entry.aliases.aliases),
                ssa(lcmd->cmdLine.entry.sepSessionApps),
                dlgFrame(lcmd->makeDialog("DlgOptions", this)),
                tabs(dynamic_cast<GTabbedPanel&>(dlgFrame->getDialogPanel().getComponentByID("101"))),
                tabsStartup(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageStartup").getComponentByID("101"))),
                tabsFilter(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageFilter").getComponentByID("101"))),
                tabsSort(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageSort").getComponentByID("101"))),
                tabsView(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageView").getComponentByID("101"))),
                tabsVarious(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageVarious").getComponentByID("101"))),
                tabsFonts(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageFonts").getComponentByID("101"))),
                tabsPrograms(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPagePrograms").getComponentByID("101"))),
                tabsConsole(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageConsole").getComponentByID("101"))),
                tabsVfs(dynamic_cast<GTabbedPanel&>(tabs.getTab("DlgOptPageVfs").getComponentByID("101")))
{
   tabsStartup.getTab("LStartup").setDialogMessageHandler(&startup1);
   tabsStartup.getTab("RStartup").setDialogMessageHandler(&startup2);
   tabsFilter.getTab("LFilter").setDialogMessageHandler(&filter1);
   tabsFilter.getTab("RFilter").setDialogMessageHandler(&filter2);
   tabsSort.getTab("LSort").setDialogMessageHandler(&sort1);
   tabsSort.getTab("RSort").setDialogMessageHandler(&sort2);
   tabsView.getTab("ViewCmn").setDialogMessageHandler(&viewCmn);
   tabsView.getTab("LView").setDialogMessageHandler(&view1);
   tabsView.getTab("RView").setDialogMessageHandler(&view2);
   tabsView.getTab("ViewStyle").setDialogMessageHandler(&viewStyle);
   tabsVarious.getTab("Confirm").setDialogMessageHandler(&confirm);
   tabsVarious.getTab("SaveOpt").setDialogMessageHandler(&saveOpt);
   tabsVarious.getTab("FileCopy").setDialogMessageHandler(&fileCopy);
   tabsVarious.getTab("FileDel").setDialogMessageHandler(&fileDel);
   tabsVarious.getTab("DirCache").setDialogMessageHandler(&dirCache);
   tabsVarious.getTab("Various").setDialogMessageHandler(&various);
   tabsFonts.getTab("FontCmn").setDialogMessageHandler(&fontCmn);
   tabsFonts.getTab("LFont").setDialogMessageHandler(&font1);
   tabsFonts.getTab("RFont").setDialogMessageHandler(&font2);
   tabsPrograms.getTab("ProgEdit").setDialogMessageHandler(&progEdit);
   tabsPrograms.getTab("ProgView").setDialogMessageHandler(&progView);
   tabsPrograms.getTab("ProgFFind").setDialogMessageHandler(&progFFind);
   tabsPrograms.getTab("ProgFDiff").setDialogMessageHandler(&progFDiff);
   tabsPrograms.getTab("ProgCClean").setDialogMessageHandler(&progCClean);
   tabsPrograms.getTab("ProgUndel").setDialogMessageHandler(&progUndel);
   tabsPrograms.getTab("ProgEAEdit").setDialogMessageHandler(&progEAEdit);
   tabsConsole.getTab("Cmd").setDialogMessageHandler(&commands);
   tabsConsole.getTab("Env").setDialogMessageHandler(&env);
   tabsConsole.getTab("Aliases").setDialogMessageHandler(&aliases);
   tabsConsole.getTab("SSA").setDialogMessageHandler(&ssa);
   tabsConsole.getTab("ConVarious").setDialogMessageHandler(&conVarious);
   tabsVfs.getTab("VfsVarious").setDialogMessageHandler(&vfsVarious);
   tabsVfs.getTab("VfsZip").setDialogMessageHandler(&vfsZip);
}

void LCmdDlgOptions::executeModal ( const GString& id )
{
   initialPageID = id;
   dlgFrame->setNoInitCtrlChanged(true);
   dlgFrame->executeModal(&lcmd->mainWin);
}

bool LCmdDlgOptions::handleDialogMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GM_INITDIALOG:
      {
         if (initialPageID == "")
         {
            // Start with the recent active page.
            if (!hasBeenExecuted) // If this is the first time the dialog is executed.
            {
               hasBeenExecuted = true;
               GSectionBag& ini = lcmd->getIniProperties();
               initialPageID = ini.getString("DlgOptions", "recentTab", GString::Empty);
            }
         }

         // ---
         GString id = initialPageID;
         if (id == "LView" || id == "RView" || 
             id == "ViewCmn" || id == "ViewStyle")
         {
            tabs.activateTab("DlgOptPageView");
            tabsView.activateTab(id);
         }
         else
         if (id == "LSort" || id == "RSort")
         {
            tabs.activateTab("DlgOptPageSort");
            tabsSort.activateTab(id);
         }
         else
         if (id == "LStartup" || id == "RStartup")
         {
            tabs.activateTab("DlgOptPageStartup");
            tabsStartup.activateTab(id);
         }
         else
         if (id == "LFilter" || id == "RFilter")
         {
            tabs.activateTab("DlgOptPageFilter");
            tabsFilter.activateTab(id);
         }
         else
         if (id == "Confirm" || id == "FileCopy" ||
            id == "FileDel" || id == "SaveOpt" ||
            id == "DirCache" || id == "Various")
         {
            tabs.activateTab("DlgOptPageVarious");
            tabsVarious.activateTab(id);
         }
         else
         if (id == "FontCmn" || id == "LFont" || id == "RFont")
         {
            tabs.activateTab("DlgOptPageFonts");
            tabsFonts.activateTab(id);
         }
         else
         if (id == "ProgEdit" || id == "ProgFFind" || 
             id == "ProgFDiff" || id == "ProgCClean" || 
             id == "ProgUndel" || id == "ProgEAEdit" ||
             id == "ProgView")
         {
            tabs.activateTab("DlgOptPagePrograms");
            tabsPrograms.activateTab(id);
         }
         else
         if (id == "Cmd" || id == "Env" || 
             id == "Aliases" || id == "SSA" ||
             id == "ConVarious")
         {
            tabs.activateTab("DlgOptPageConsole");
            tabsConsole.activateTab(id);
         }
         else
         if (id == "VfsVarious" || id == "VfsZip")
         {
            tabs.activateTab("DlgOptPageVfs");
            tabsVfs.activateTab(id);
         }
         else
         {
            // Start with the default page.
         }
         return true;
      }

      case GM_DISMISSDIALOG: 
      {
         // Fetch the ID of the currently active tab, and store it in the
         // user INI-properties of the program.
         GString recentTab;
         GTabbedPanel& tabs1 = dynamic_cast<GTabbedPanel&>(dlg.getComponentByID("101"));
         int tabs1Idx = tabs1.getActiveTabIndex();
         if (tabs1Idx >= 0)
         {
            GWindow& tabs1Cur = tabs1.getIndexedTab(tabs1Idx);
            GTabbedPanel& tabs2 = dynamic_cast<GTabbedPanel&>(tabs1Cur.getComponentByID("101"));
            int tabs2Idx = tabs2.getActiveTabIndex();
            if (tabs2Idx >= 0)
            {
               const GTabbedPanel::Page& page = tabs2.getIndexedTabPage(tabs2Idx);
               recentTab = page.getUserID();
            }
         }
         GSectionBag& ini = lcmd->getIniProperties();
         ini.putString("DlgOptions", "recentTab", recentTab, true);
         return true; 
      }

      case GM_CLOSEWIN:
      {
         dlg.dismiss();
         return true;
      }

      default:
         return false;
   }
}

void LCmdDlgOptions::CmdJumpToOptionsPage ( const GString& id )
{
   if (lcmd->optionsDlg == null)
   {
      // ulonglong timeStart = GSystem::CurrentTimeMillis();
      lcmd->optionsDlg = new LCmdDlgOptions();
      // lcmd->printF("Time used to create the options-dialog: %d ms.", GVArgs(GSystem::CurrentTimeMillis() - timeStart));
   }
   lcmd->optionsDlg->executeModal(id);
}
