/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_MAINWINDOW
#define __LCMD_MAINWINDOW

#include "glib/gui/event/GCommandMap.h"
#include "lcmd/LCmdOptions.h"
#include "lcmd/LCmdSelfTest.h"

/**
 * This is the class of the main Larsen Commander window.
 *
 * This can also be seen as the client area window. It is a direct child
 * window of the main frame window.
 *
 * It is used as the parent of other Larsen Commander component windows such
 * as the toolbar, panels parent, keybar window and the statusbar window.
 *
 * @author  Leif Erik Larsen
 * @since   1999.04.08
 */
class LCmdMainWindow : public GFrameWindow
{
   friend class LCmd;

   public:

      /** A reference to the Larsen Commander program object of where we belongs. */
      class LCmd& lcmd;

      /** The Sub Client Window. */
      class LCmdSubClient* subClient;

      /** Texts used in the statusbar cells. */
      GStringl strCountChildProcesses;
      GStringl strConLineCurrent;
      GStringl strConLineCount;
      GStringl strPhysMemFree;
      GStringl strSysCountProcesses;
      GStringl strSysCountThreads;
      GStringl strDriveFree;
      GStringl strDirCacheCount;

   public:

      LCmdMainWindow ( class LCmd& lcmd );
      virtual ~LCmdMainWindow ();

   protected:

      virtual bool onActivate ();
      virtual bool onClose ();
      virtual bool onKeyDown ( const GKeyMessage& key );
      virtual bool onUserMessage ( GUserMessage& msg );
      virtual bool onTimer ( const GString& timerID, GObject* userData );

   public:

      /**
       * Close and destroy the specified text viewer frame window.
       * This method is called by {@link #onUserMessage} as well as by the
       * Larsen Commander termination code in order to close all
       * open (if any) text viewer in a clean way before the
       * rest of the application terminates.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.09
       */
      void closeAndDestroyTextViewer ( class LCmdInternalViewer* v );

      /**
       * Make sure that the keyboard focus is set on the correct component.
       *
       * This function should be called whenever Larsen Commander
       * gets activated and whenever the user has shifted which of the two
       * panels is active. It should also be called whenever any of the
       * window components (such as the Command Line, Header Bar, etc.) are
       * toggled on or off.
       */
      virtual void ensureFocusOK ();

      virtual bool executeAbstractCommand ( GAbstractCommand *cmd );

      /**
       * Get a reference to the main window of Larsen Commander.
       * 
       * @author  Leif Erik Larsen
       * @since   2004.11.16
       */
      static LCmdMainWindow& GetMainWindow ();

      /**
       * @author  Leif Erik Larsen
       * @see     2004.07.29
       */
      virtual void grabFocus ( bool force = false );

      virtual void updateCommandStates ();

      /**
       * Update the statusbar cell that shows the current number
       * of child processes.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       */
      void updateStatusbarChildProcessCount ();

      /**
       * Update the statusbar cell that shows which console 
       * monitor line is active.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       */
      void updateStatusbarConsoleCurrentLine ();

      /**
       * Update the statusbar cell that shows the number of lines 
       * in the console monitor.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       */
      void updateStatusbarConsoleLineCount ();

      /**
       * Update the statusbar cell that shows the number of items
       * in the Dynamic Directory Cache.
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.10
       */
      void updateStatusbarDirCacheCount ();

      /**
       * Update the statusbar cell that shows the free space of the 
       * current file panel drive.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.05
       */
      void updateStatusbarDriveFreeCell ();

      /**
       * Load and activate the profile variables of Larsen Commander.
       * @see #writeProfile
       */
      void queryProfile ( const GString& sectName );

      /**
       * Write the profile variables of Larsen Commander.
       *
       * @see #queryProfile
       * @see LCmd#writeProfile
       */
      void writeProfile ( const GString& sectName, bool force );

   private:

      static bool CheckSortOpt ( LCmdFilePanel& p, LCmdFilePanelSortOptions::SORT_WHAT what );

      void queryProgProfile ( const GString& sectName, LCmdOptions::ProgOption& prog );
      void writeProgProfile ( const GString& sectName, LCmdOptions::ProgOption& prog, bool force );

      void viewFile ( const LCmdOptions::ProgOption& viewer );

      /**
       * Perform the specified self test. 
       * This method will catch exceptions and assertion faults and display
       * a message box to the user. It will also show a message box in case
       * everything tests ok.
       *
       * @author  Leif Erik Larsen
       * @since   2005.01.14
       */
      void doSelfTest ( LCmdSelfTest::TestID tid );

   public:

      void cmdEditEAs ( GAbstractCommand* cmd = null );
      void cmdProgOptionsEABrowser ( GAbstractCommand* cmd = null );
      void cmdProgOptionsUndeleter ( GAbstractCommand* cmd = null );
      void cmdProgOptionsFileDiff ( GAbstractCommand* cmd = null );
      void cmdProgOptionsFileDupFinder ( GAbstractCommand* cmd = null );
      void cmdProgOptionsFileFinder ( GAbstractCommand* cmd = null );

      void cmdActivateOppositePanel ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir0 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir1 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir2 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir3 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir4 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir5 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir6 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir7 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir8 ( GAbstractCommand* cmd = null );
      void cmdActivateStoredDir9 ( GAbstractCommand* cmd = null );
      void cmdAlignPanelsHeight ( GAbstractCommand* cmd = null );
      void cmdAlignPanelsWidth ( GAbstractCommand* cmd = null );

      /**
       * Append an end-of-file character (^Z) to the child process
       * of which is (and only if it is) currently waiting for input to
       * its stdin from our command line.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.20
       */
      void cmdAppendEndOfStream ( GAbstractCommand* cmd = null );

      void cmdExtendLeftPanelWidth ( GAbstractCommand* cmd = null );
      void cmdExtendRightPanelWidth ( GAbstractCommand* cmd = null );
      void cmdExtendPanelsHeight ( GAbstractCommand* cmd = null );
      void cmdExtendConMonHeight ( GAbstractCommand* cmd = null );
      void cmdAutoFilenameBackward ( GAbstractCommand* cmd = null );
      void cmdAutoFilenameForeward ( GAbstractCommand* cmd = null );
      void cmdBreak ( GAbstractCommand* cmd = null );
      void cmdCalcDirSize ( GAbstractCommand* cmd = null );
      void cmdCalcRootDirSize ( GAbstractCommand* cmd = null );
      void cmdChooseFileAttrib ( GAbstractCommand* cmd = null );
      void cmdChooseFileNameCase ( GAbstractCommand* cmd = null );
      void cmdChooseFileTime ( GAbstractCommand* cmd = null );

      /**
       * Write a complete INI-file to a new and unique temporary file
       * of which will be used as the INI-file of the new instance of
       * Larsen Commander that we will request to be launched.
       */
      void cmdCloneProgramInstance ( GAbstractCommand* cmd = null );

      void cmdCloseAndExit ( GAbstractCommand* cmd = null );
      void cmdCommandHistory ( GAbstractCommand* cmd = null );
      void cmdCompareDirs ( GAbstractCommand* cmd = null );
      void cmdCompareFiles ( GAbstractCommand* cmd = null );
      void cmdCopy ( GAbstractCommand* cmd = null );
      void cmdCopyCurFileName ( GAbstractCommand* cmd = null );
      void cmdCopyCurFilePath ( GAbstractCommand* cmd = null );
      void cmdCopyFile ( GAbstractCommand* cmd = null );
      void cmdCopyFileToSameDir ( GAbstractCommand* cmd = null );
      void cmdCreateSysObject ( GAbstractCommand* cmd = null );
      void cmdCurPanelCycleIconTypes ( GAbstractCommand* cmd = null );
      void cmdCurPanelGotoRecentDir ( GAbstractCommand* cmd = null );
      void cmdCurPanelModeBrief ( GAbstractCommand* cmd = null );
      void cmdCurPanelModeFull ( GAbstractCommand* cmd = null );
      void cmdCurPanelModeTree ( GAbstractCommand* cmd = null );
      void cmdCurPanelModeWide ( GAbstractCommand* cmd = null );
      void cmdCurPanelOptionsSort ( GAbstractCommand* cmd = null );
      void cmdCurPanelReread ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortByDate ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortByExtension ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortByName ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortBySize ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortByTime ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortReverse ( GAbstractCommand* cmd = null );
      void cmdCurPanelSortUnsorted ( GAbstractCommand* cmd = null );
      void cmdDebugOptions ( GAbstractCommand* cmd = null );
      void cmdDebugShowLogFile ( GAbstractCommand* cmd = null );
      void cmdDeleteFile ( GAbstractCommand* cmd = null );
      void cmdDirHistory ( GAbstractCommand* cmd = null );
      void cmdEditFile ( GAbstractCommand* cmd = null );
      void cmdEscape ( GAbstractCommand* cmd = null );
      void cmdFeedChildProcessStdIn ( GAbstractCommand* cmd = null );
      void cmdFindDuplicateFiles ( GAbstractCommand* cmd = null );
      void cmdFindFile ( GAbstractCommand* cmd = null );
      void cmdGotoLineNum ( GAbstractCommand* cmd = null );
      void cmdHelpAbout ( GAbstractCommand* cmd = null );
      void cmdHelpContents ( GAbstractCommand* cmd = null );
      void cmdHelpCmdLineArgs ( GAbstractCommand* cmd = null );
      void cmdHelpIndex ( GAbstractCommand* cmd = null );
      void cmdHelpKeyboard ( GAbstractCommand* cmd = null );
      void cmdHelpShowTip ( GAbstractCommand* cmd = null );
      void cmdInsertItemNameInCmdLine ( GAbstractCommand* cmd = null );
      void cmdInsertItemPathInCmdLine ( GAbstractCommand* cmd = null );
      void cmdInvertSelection ( GAbstractCommand* cmd = null );
      void cmdLeftChooseDrive ( GAbstractCommand* cmd = null );
      void cmdLeftChooseFileAttrib ( GAbstractCommand* cmd = null );
      void cmdLeftChooseFileTime ( GAbstractCommand* cmd = null );
      void cmdLeftCycleIconTypes ( GAbstractCommand* cmd = null );
      void cmdLeftGotoRecentDir ( GAbstractCommand* cmd = null );
      void cmdLeftModeBrief ( GAbstractCommand* cmd = null );
      void cmdLeftModeFull ( GAbstractCommand* cmd = null );
      void cmdLeftModeInfo ( GAbstractCommand* cmd = null );
      void cmdLeftModeTree ( GAbstractCommand* cmd = null );
      void cmdLeftModeWide ( GAbstractCommand* cmd = null );
      void cmdLeftOptionsColor ( GAbstractCommand* cmd = null );
      void cmdLeftOptionsFileFilter ( GAbstractCommand* cmd = null );
      void cmdLeftOptionsFont ( GAbstractCommand* cmd = null );
      void cmdLeftOptionsSort ( GAbstractCommand* cmd = null );
      void cmdLeftOptionsStartup ( GAbstractCommand* cmd = null );
      void cmdLeftOptionsView ( GAbstractCommand* cmd = null );
      void cmdLeftReread ( GAbstractCommand* cmd = null );
      void cmdLeftShowFileInfo ( GAbstractCommand* cmd = null );
      void cmdLeftSortByDate ( GAbstractCommand* cmd = null );
      void cmdLeftSortByExtension ( GAbstractCommand* cmd = null );
      void cmdLeftSortByName ( GAbstractCommand* cmd = null );
      void cmdLeftSortBySize ( GAbstractCommand* cmd = null );
      void cmdLeftSortByTime ( GAbstractCommand* cmd = null );
      void cmdLeftSortReverse ( GAbstractCommand* cmd = null );
      void cmdLeftSortUnsorted ( GAbstractCommand* cmd = null );
      void cmdLeftWalkDirectoryUp ( GAbstractCommand* cmd = null );
      void cmdLeftWalkRootDirectory ( GAbstractCommand* cmd = null );
      void cmdMakeDir ( GAbstractCommand* cmd = null );
      void cmdMoveFile ( GAbstractCommand* cmd = null );
      void cmdNavigateBetwConAndCmdLine ( GAbstractCommand* cmd = null );
      void cmdNavigateDown ( GAbstractCommand* cmd = null );
      void cmdNavigateEnd ( GAbstractCommand* cmd = null );
      void cmdNavigateEnter ( GAbstractCommand* cmd = null );
      void cmdNavigateEnterSysObj ( GAbstractCommand* cmd = null );
      void cmdNavigateHome ( GAbstractCommand* cmd = null );
      void cmdNavigateLeft ( GAbstractCommand* cmd = null );
      void cmdNavigateNextMatch ( GAbstractCommand* cmd = null );
      void cmdNavigatePageDown ( GAbstractCommand* cmd = null );
      void cmdNavigatePageUp ( GAbstractCommand* cmd = null );
      void cmdNavigatePrevMatch ( GAbstractCommand* cmd = null );
      void cmdNavigateRight ( GAbstractCommand* cmd = null );
      void cmdNavigateUp ( GAbstractCommand* cmd = null );
      void cmdOpenSysFolder ( GAbstractCommand* cmd = null );
      void cmdOppositePanelInfo ( GAbstractCommand* cmd = null );
      void cmdOptionsAliases ( GAbstractCommand* cmd = null );
      void cmdOptionsColor ( GAbstractCommand* cmd = null );
      void cmdOptionsConfirmations ( GAbstractCommand* cmd = null );
      void cmdOptionsDialog ( GAbstractCommand* cmd = null );
      void cmdOptionsEnvironmentVars ( GAbstractCommand* cmd = null );
      void cmdOptionsFont ( GAbstractCommand* cmd = null );
      void cmdOptionsInternalCommands ( GAbstractCommand* cmd = null );
      void cmdOptionsSSA ( GAbstractCommand* cmd = null );
      void cmdOptionsSaveOnExit ( GAbstractCommand* cmd = null );
      void cmdOptionsFileCopyMv ( GAbstractCommand* cmd = null );
      void cmdOptionsFileDel ( GAbstractCommand* cmd = null );
      void cmdOptionsDirCache ( GAbstractCommand* cmd = null );
      void cmdOptionsStyle ( GAbstractCommand* cmd = null );
      void cmdOptionsVarious ( GAbstractCommand* cmd = null );
      void cmdOptionsView ( GAbstractCommand* cmd = null );
      void cmdOptionsConsoleVarious ( GAbstractCommand* cmd = null );
      void cmdOptionsVfsVarious ( GAbstractCommand* cmd = null );
      void cmdOptionsVfsZip ( GAbstractCommand* cmd = null );
      void cmdProcessFilesByCommand ( GAbstractCommand* cmd = null );
      void cmdProgOptionsEditor ( GAbstractCommand* cmd = null );
      void cmdProgOptionsViewer ( GAbstractCommand* cmd = null );
      void cmdPullDownMenu ( GAbstractCommand* cmd = null );
      void cmdRenameFile ( GAbstractCommand* cmd = null );
      void cmdRightChooseDrive ( GAbstractCommand* cmd = null );
      void cmdRightChooseFileAttrib ( GAbstractCommand* cmd = null );
      void cmdRightChooseFileTime ( GAbstractCommand* cmd = null );
      void cmdRightCycleIconTypes ( GAbstractCommand* cmd = null );
      void cmdRightGotoRecentDir ( GAbstractCommand* cmd = null );
      void cmdRightModeBrief ( GAbstractCommand* cmd = null );
      void cmdRightModeFull ( GAbstractCommand* cmd = null );
      void cmdRightModeInfo ( GAbstractCommand* cmd = null );
      void cmdRightModeTree ( GAbstractCommand* cmd = null );
      void cmdRightModeWide ( GAbstractCommand* cmd = null );
      void cmdRightOptionsColor ( GAbstractCommand* cmd = null );
      void cmdRightOptionsFileFilter ( GAbstractCommand* cmd = null );
      void cmdRightOptionsFont ( GAbstractCommand* cmd = null );
      void cmdRightOptionsSort ( GAbstractCommand* cmd = null );
      void cmdRightOptionsStartup ( GAbstractCommand* cmd = null );
      void cmdRightOptionsView ( GAbstractCommand* cmd = null );
      void cmdRightReread ( GAbstractCommand* cmd = null );
      void cmdRightShowFileInfo ( GAbstractCommand* cmd = null );
      void cmdRightSortByDate ( GAbstractCommand* cmd = null );
      void cmdRightSortByExtension ( GAbstractCommand* cmd = null );
      void cmdRightSortByName ( GAbstractCommand* cmd = null );
      void cmdRightSortBySize ( GAbstractCommand* cmd = null );
      void cmdRightSortByTime ( GAbstractCommand* cmd = null );
      void cmdRightSortReverse ( GAbstractCommand* cmd = null );
      void cmdRightSortUnsorted ( GAbstractCommand* cmd = null );
      void cmdRightWalkDirectoryUp ( GAbstractCommand* cmd = null );
      void cmdRightWalkRootDirectory ( GAbstractCommand* cmd = null );
      void cmdRunCommandInNewSession ( GAbstractCommand* cmd = null );
      void cmdSaveOptions ( GAbstractCommand* cmd = null );
      void cmdScrollCmdHistDown ( GAbstractCommand* cmd = null );
      void cmdScrollCmdHistUp ( GAbstractCommand* cmd = null );
      void cmdSearch ( GAbstractCommand* cmd = null );
      void cmdSearchNext ( GAbstractCommand* cmd = null );
      void cmdSelectAll ( GAbstractCommand* cmd = null );
      void cmdSelectGroup ( GAbstractCommand* cmd = null );
      void cmdSelfTestAll ( GAbstractCommand* cmd = null );
      void cmdSelfTestLowLevelAPIs ( GAbstractCommand* cmd = null );
      void cmdSelfTestGarbageCollector ( GAbstractCommand* cmd = null );
      void cmdSelfTestObjectSynch ( GAbstractCommand* cmd = null );
      void cmdSelfTestContainerClasses ( GAbstractCommand* cmd = null );
      void cmdSelfTestLowLevelFileIO ( GAbstractCommand* cmd = null );
      void cmdSelfTestTextViewer ( GAbstractCommand* cmd = null );
      void cmdSelfTestWorkerThread ( GAbstractCommand* cmd = null );
      void cmdSelfTestCopySingleFile ( GAbstractCommand* cmd = null );
      void cmdSelfTestMoveSingleFile ( GAbstractCommand* cmd = null );
      void cmdSelfTestChangeFilenameCase ( GAbstractCommand* cmd = null );
      void cmdSelfTestChangeFileAttributes ( GAbstractCommand* cmd = null );
      void cmdSelfTestChangeFileTime ( GAbstractCommand* cmd = null );
      void cmdShowContextMenu ( GAbstractCommand* cmd = null );

      /**
       * Pop up a box of where to display some information about current
       * selected file in the active panel.
       */
      void cmdShowFileInfo ( GAbstractCommand* cmd = null );

      void cmdStoreDir0 ( GAbstractCommand* cmd = null );
      void cmdStoreDir1 ( GAbstractCommand* cmd = null );
      void cmdStoreDir2 ( GAbstractCommand* cmd = null );
      void cmdStoreDir3 ( GAbstractCommand* cmd = null );
      void cmdStoreDir4 ( GAbstractCommand* cmd = null );
      void cmdStoreDir5 ( GAbstractCommand* cmd = null );
      void cmdStoreDir6 ( GAbstractCommand* cmd = null );
      void cmdStoreDir7 ( GAbstractCommand* cmd = null );
      void cmdStoreDir8 ( GAbstractCommand* cmd = null );
      void cmdStoreDir9 ( GAbstractCommand* cmd = null );
      void cmdStoreDirIntoFirstFreeIndex ( GAbstractCommand* cmd = null );
      void cmdStoredDirs ( GAbstractCommand* cmd = null );

      /**
       * Swap the two file panels so that the left panel will be the right
       * panel and vice versa.
       */
      void cmdSwapPanels ( GAbstractCommand* cmd = null );

      void cmdTest ( GAbstractCommand* cmd = null );

      void cmdToggleCommandLine ( GAbstractCommand* cmd = null );
      void cmdToggleFullScreen ( GAbstractCommand* cmd = null );
      void cmdToggleHiddenFiles ( GAbstractCommand* cmd = null );
      void cmdToggleItemMark ( GAbstractCommand* cmd = null );
      void cmdToggleItemMarkAndMoveDown ( GAbstractCommand* cmd = null );
      void cmdToggleKeybar ( GAbstractCommand* cmd = null );
      void cmdTogglePanels ( GAbstractCommand* cmd = null );

      /**
       * Toggle on/off if we shall automatically save setup upon
       * program exit.
       */
      void cmdToggleSaveOptionsOnExit ( GAbstractCommand* cmd = null );

      void cmdToggleStatusbar ( GAbstractCommand* cmd = null );
      void cmdToggleToolbar ( GAbstractCommand* cmd = null );
      void cmdUndeleteFile ( GAbstractCommand* cmd = null );
      void cmdUnselectGroup ( GAbstractCommand* cmd = null );
      void cmdUnselectInConMon ( GAbstractCommand* cmd = null );
      void cmdUserCommand ( GAbstractCommand* cmd = null );
      void cmdUserMenu ( GAbstractCommand* cmd = null );
      void cmdViewFile ( GAbstractCommand* cmd = null );
      void cmdWalkDirectoryDown ( GAbstractCommand* cmd = null );
      void cmdWalkDirectoryUp ( GAbstractCommand* cmd = null );
      void cmdWalkCurDirIntoOppositePanel ( GAbstractCommand* cmd = null );
      void cmdWalkOppositeDirectory ( GAbstractCommand* cmd = null );
      void cmdWalkRootDirectory ( GAbstractCommand* cmd = null );
      void cmdViewTypedFileName ( GAbstractCommand* cmd = null );
      void cmdEditTypedFileName ( GAbstractCommand* cmd = null );
      void cmdWalkLeftToRightDir ( GAbstractCommand* cmd = null );
      void cmdWalkRightToLeftDir ( GAbstractCommand* cmd = null );

   DECLARE_COMMAND_TABLE(LCmdMainWindow);
};

#endif // #ifndef __LCMD_MAINWINDOW
