/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_INTERNALVIEWER
#define __LCMD_INTERNALVIEWER

#include "lcmd/LCmdDlgTextSearch.h"

#include "glib/gui/GFrameWindow.h"
#include "glib/gui/GTextViewer.h"
#include "glib/gui/GWorkerThread.h"
#include "glib/vfs/GFile.h"

/**
 * This is the class that implements the internal
 * text viewer of Larsen Commander.
 *
 * @author  Leif Erik Larsen
 * @since   2002.03.03
 */
class LCmdInternalViewer : public GFrameWindow
{
   private:

      /**
       * We must sub-class GTextViewer in order to make the popup menu of
       * the text viewer frame window being poipped up when the user
       * right-clicks his mouse. This is important to make sure that the
       * popup menu commands are enabled/disabled correctly upon
       * {@link LCmdInternalViewer#updateCommandStates}.
       *
       * @author  Leif Erik Larsen
       * @since   2002.03.10
       */
      class Viewer : public GTextViewer
      {
         private:

            class LCmdInternalViewer& pwin;

         public:

            Viewer ( GTextViewer::SearchParams& sp, 
                     class LCmdInternalViewer& pwin );

            virtual ~Viewer ();

            virtual bool onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
            virtual void onCaretPositionChanged ();
            virtual bool onPaint ( class GGraphics& g, const GRectangle& rect );
      };

      /**
       * This class implements the background worker thread that is used 
       * to load the content of a file of which to view.
       *
       * @author  Leif Erik Larsen
       * @since   2004.01.18
       */
      class FileLoader : public GWorkerThread
      {
         friend class LCmdInternalViewer;
         class LCmdInternalViewer& owner;
         explicit FileLoader ( class LCmdInternalViewer& owner );
         virtual ~FileLoader ();
         void loadFile ( const GString& path );
         virtual void runTheWorkerThread ( class GWorkerThread& worker );
         virtual void onWorkerThreadCommand ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& cmd );
         virtual void onWorkerThreadUpdateMonitor ( class GWorkerThread& worker, class GDialogPanel& monitor );
         virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& msgID, GObject* userParam );
      };

      /**
       * Class used to remember caret position information and viewer 
       * window size and position, when reopening a file that was 
       * recently viewed in another file window.
       *
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       */
      class FileOptions : public GObject
      {
         friend class LCmdInternalViewer;
         const GTextViewer::Pos caretPos;
         const GDimension windowSize;
         const GPoint windowPos;
         const GFrameWindow::State windowState;
         const int firstVisibleLine;
         const int firstVisibleColumn;
         FileOptions ( const GTextViewer::Pos& caretPos,
                       const GDimension& windowSize,
                       const GPoint& windowPos,
                       GFrameWindow::State windowState,
                       int firstVisibleLine,
                       int firstVisibleColumn );
         virtual ~FileOptions ();
      };

      friend class Viewer;
      friend class FileLoader;
      friend class FileOptions;

      /** The ID of the message used to append text lines from background thread to GUI. */
      const GString MSGID_APPENDLINES;

      /** The path of the file that is to be viewed by us. */
      GString filePath;

      /** The dialog used to let the user specify the properties for a search command. */
      LCmdDlgTextSearch dlgSearch;

      /** The wrapper window that containes the GTextViewer window. */
      GWindow wrapperWin;

      /** The client area primitive text viewer window. */
      Viewer viewerWin;

      /** The background worker thread used to load content of files. */
      FileLoader fileLoader;

      /** Remember caret position etc. of recently viewed files. The file path is used as the keybag key. */
      static GKeyBag<LCmdInternalViewer::FileOptions> FileStore;

   public:

      LCmdInternalViewer ( const GString& filePath );
      virtual ~LCmdInternalViewer ();

   protected:

      virtual bool onActivate ();
      virtual bool onClose ();

   public:

      void cmdFileOpen ( class GAbstractCommand* cmd = null );
      void cmdExit ( class GAbstractCommand* cmd = null );
      void cmdCopy ( class GAbstractCommand *cmd = null );
      void cmdSearch ( class GAbstractCommand* cmd = null );
      void cmdSearchNext ( class GAbstractCommand* cmd = null );
      void cmdGotoLineNum ( class GAbstractCommand* cmd = null );
      void cmdSelectAll ( class GAbstractCommand* cmd = null );
      void cmdUnselect ( class GAbstractCommand* cmd = null );
      void cmdShowContextMenu ( class GAbstractCommand* cmd = null );
      void cmdToggleToolbar ( class GAbstractCommand* cmd = null );
      void cmdToggleKeybar ( class GAbstractCommand* cmd = null );
      void cmdToggleStatusbar ( class GAbstractCommand* cmd = null );
      void cmdToggleShowLineFeeds ( class GAbstractCommand* cmd = null );
      void cmdToggleShowSpaces ( class GAbstractCommand* cmd = null );
      void cmdToggleShowTabs ( class GAbstractCommand* cmd = null );
      void cmdToggleQuickScrollMode ( class GAbstractCommand* cmd = null );
      void cmdToggleWordWrap ( class GAbstractCommand* cmd = null );
      void cmdOptionTextViewerFont ( class GAbstractCommand* cmd = null );
      void cmdNavigateUp ( class GAbstractCommand* cmd = null );
      void cmdNavigateDown ( class GAbstractCommand* cmd = null );
      void cmdNavigateLeft ( class GAbstractCommand* cmd = null );
      void cmdNavigateRight ( class GAbstractCommand* cmd = null );
      void cmdNavigateHome ( class GAbstractCommand* cmd = null );
      void cmdNavigateEnd ( class GAbstractCommand* cmd = null );
      void cmdNavigatePrevWord ( class GAbstractCommand* cmd = null );
      void cmdNavigateNextWord ( class GAbstractCommand* cmd = null );
      void cmdNavigatePageUp ( class GAbstractCommand* cmd = null );
      void cmdNavigatePageDown ( class GAbstractCommand* cmd = null );
      void cmdNavigateTop ( class GAbstractCommand* cmd = null );
      void cmdNavigateBottom ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectUp ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectDown ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectLeft ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectRight ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectHome ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectEnd ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectPrevWord ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectNextWord ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectPageUp ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectPageDown ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectTop ( class GAbstractCommand* cmd = null );
      void cmdNavigateSelectBottom ( class GAbstractCommand* cmd = null );

      virtual void queryProfile ( const GString& sectName );
      virtual void writeProfile ( const GString& sectName, bool force = false );

      virtual void ensureFocusOK ();
      virtual void updateCommandStates ();

      /**
       * @author  Leif Erik Larsen
       * @since   2005.02.04
       */
      class GTextViewer& getTextViewer ();
      class LCmdDlgTextSearch& getSearchParamsEditor ();

   DECLARE_COMMAND_TABLE(LCmdInternalViewer);
};

#endif // #ifndef __LCMD_INTERNALVIEWER
