/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_INTERNALCOMMAND
#define __LCMD_INTERNALCOMMAND

#include "glib/primitives/GString.h"

/**
 * This class is used to represent command line commands that are
 * internal to the Larsen Commander application.
 *
 * Such commands will not actually be launched as a child processs, but
 * will instead be execute directly and internally by Larsen Commander.
 * Either by the main thread or by the command launcher thread.
 *
 * @author  Leif Erik Larsen
 * @since   1998.12.12
 */
class InternalCommand : public GObject
{
   private:

      /** The name of the command. */
      GString name;

      /** The short language dependent help text of the command. */
      GStringl shortHelp;

      /** The long language dependent help text of the command. */
      GStringl longHelp;

   protected:

      /** True if the command is disabled. In that case we will try to execute the command normally as an external command. */
      bool disabled;

      /** True if command is to be traversed to "CMD.EXE". */
      bool transferToShell;

   public:

      explicit InternalCommand ( const char* name, bool transferToShell = false );
      virtual ~InternalCommand ();

   public:

      const GString& getCommandStr () const;
      const GString& getShortHelp () const;
      const GString& getLongHelp () const;
      bool isDisabled ();
      bool isTransferToShell ();
      virtual void readProfile ( class GSectionBag& ini );
      void setDisabled ( bool disabled );
      virtual void writeProfile ( class GSectionBag& ini, bool doIt = true );
};

/**
 * This class is used to represent command line commands that are
 * internal to the Operating System dependent Shell.
 *
 * For OS/2, this is "CMD.EXE".
 * Therefore, such commands must be launched via the Shell.
 *
 * @author  Leif Erik Larsen
 * @since   1998.12.12
 */
class InternalCommandToShell : public InternalCommand
{
   public:

      InternalCommandToShell ( const char* cmd, bool transferToShell = true );
      virtual ~InternalCommandToShell ();

   public:

      virtual void writeProfile ( class GSectionBag& ini, bool doIt = true );
      virtual void readProfile ( class GSectionBag& ini );
};

#endif // #ifndef __LCMD_INTERNALCOMMAND
