/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdInfoPanel.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmd.h"

#include "glib/util/GStringUtil.h"
#include "glib/io/GFileNotFoundException.h"

// --------------------------------------------------------------------------
// AbstractItem

LCmdInfoPanel::AbstractItem::AbstractItem ( LCmdInfoPanel& info, UpdateType updt )
                            :ypos(0),
                             updt(updt),
                             txtColor(GColor::BLACK),
                             txtBoldColor(GColor::DRED),
                             info(info)
{
}

LCmdInfoPanel::AbstractItem::~AbstractItem ()
{
}

int LCmdInfoPanel::AbstractItem::getHeight ()
{
   return info.ownerPanel.viewInfo.getHeightOfString("X");
}

bool LCmdInfoPanel::AbstractItem::update ()
{
   return false;
}

GString LCmdInfoPanel::AbstractItem::getItemString ()
{
   return GString::Empty;
}

void LCmdInfoPanel::AbstractItem::paint ()
{
   GGraphics g(info.ownerPanel.viewInfo);
   paint(g);
}

int LCmdInfoPanel::AbstractItem::getWidthOfString ( const GString& str1,
                                                    const GString& str2,
                                                    const GString& str3,
                                                    const GString& str4 )
{
   int w1 = info.ownerPanel.viewInfo.getWidthOfString(str1);
   int w2 = info.ownerPanel.viewInfo.getWidthOfString(str2);
   int w3 = info.ownerPanel.viewInfo.getWidthOfString(str3);
   int w4 = info.ownerPanel.viewInfo.getWidthOfString(str4);
   return w1 + w2 + w3 + w4;
}

void LCmdInfoPanel::AbstractItem::paintText ( GGraphics& g,
                                              const GString& str1, const GColor& c1,
                                              const GString& str2, const GColor& c2,
                                              const GString& str3, const GColor& c3,
                                              const GString& str4, const GColor& c4 )
{
   int widthStr1 = g.getWidthOfString(str1);
   int widthStr2 = g.getWidthOfString(str2);
   int widthStr3 = g.getWidthOfString(str3);
   int widthStr4 = g.getWidthOfString(str4);
   int widthTot = widthStr1 + widthStr2 + widthStr3 + widthStr4;

   GRectangle r = info.ownerPanel.viewInfo.filesRect;
   r.height = getHeight();
   r.y = ypos + info.ypos0 - r.height;

   int xposStart = r.x + (r.width/2) - (widthTot/2);
   if (xposStart <= r.x)
      xposStart = r.x + 1;

   xposStart -= info.xpos0;
   r.x -= info.xpos0;

   g.drawFilledRectangle(r, info.ownerPanel.colors.itemBck);

   r.x = xposStart;
   r.width = widthStr1;
   g.drawText(str1, r, c1);

   r.x += widthStr1;
   r.width = widthStr2;
   g.drawText(str2, r, c2);

   r.x += widthStr2;
   r.width = widthStr3;
   g.drawText(str3, r, c3);

   r.x += widthStr3;
   r.width = widthStr4;
   g.drawText(str4, r, c4);
}

// --------------------------------------------------------------------------
// Separator

LCmdInfoPanel::Separator::Separator ( LCmdInfoPanel& info )
                         :AbstractItem(info, NEVER),
                          str1("%Txt_FPInfoMode_Separator") // oOo
{
}

int LCmdInfoPanel::Separator::getWidth ()
{
   return getWidthOfString(str1);
}

void LCmdInfoPanel::Separator::paint ( GGraphics& g )
{
   paintText(g, str1, GColor::DGRAY);
}

// --------------------------------------------------------------------------
// LCmdVersion

LCmdInfoPanel::LCmdVersion::LCmdVersion ( LCmdInfoPanel& info )
                           :AbstractItem(info, NEVER),
                            str1(lcmd->aboutBox.getVersionString())
{
}

int LCmdInfoPanel::LCmdVersion::getWidth ()
{
   return getWidthOfString(str1);
}

GString LCmdInfoPanel::LCmdVersion::getItemString ()
{
   return str1 + '\n';
}

void LCmdInfoPanel::LCmdVersion::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor);
}

// --------------------------------------------------------------------------
// LCmdCopyright

LCmdInfoPanel::LCmdCopyright::LCmdCopyright ( LCmdInfoPanel& info )
                             :AbstractItem(info, NEVER),
                              str1(lcmd->aboutBox.getCopyrightString())
{
}

int LCmdInfoPanel::LCmdCopyright::getWidth ()
{
   return getWidthOfString(str1);
}

GString LCmdInfoPanel::LCmdCopyright::getItemString ()
{
   return str1 + '\n';
}

void LCmdInfoPanel::LCmdCopyright::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor);
}

// --------------------------------------------------------------------------
// TotalPhysMem

LCmdInfoPanel::TotalPhysMem::TotalPhysMem ( LCmdInfoPanel& info )
                            :AbstractItem(info, NEVER),
                             str2("%Txt_FPInfoMode_TotalPhysMem"),
                             totalPhysRam(0)
{
   update();
}

int LCmdInfoPanel::TotalPhysMem::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::TotalPhysMem::update ()
{
   totalPhysRam = lcmd->sysInfo.getPhysicalRAM();
   str1 = GStringUtil::ToByteCountString(totalPhysRam);
   return true;
}

GString LCmdInfoPanel::TotalPhysMem::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::TotalPhysMem::paint ( GGraphics& g )
{
   paintText(g, str1, txtBoldColor, str2, txtColor);
}

// --------------------------------------------------------------------------
// FreePhysMem

LCmdInfoPanel::FreePhysMem::FreePhysMem ( LCmdInfoPanel& info )
                           :AbstractItem(info, UPON_1HZ),
#if __IBMCPP4__
                            str2("%Txt_FPInfoMode_FreeVirtMem"),
#else
                            str2("%Txt_FPInfoMode_FreePhysMem"),
#endif
                            freePhysRam(0)
{
   update();
}

int LCmdInfoPanel::FreePhysMem::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::FreePhysMem::update ()
{
   ulonglong newVal = lcmd->sysInfo.getFreePhysicalRAM();
   if (newVal == freePhysRam)
      return false;
   freePhysRam = newVal;
   str1 = GStringUtil::ToByteCountString(freePhysRam);
   return true;
}

GString LCmdInfoPanel::FreePhysMem::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::FreePhysMem::paint ( GGraphics& g )
{
   paintText(g, str1, txtBoldColor, str2, txtColor);
}

// --------------------------------------------------------------------------
// TotalBytesOnDrive

LCmdInfoPanel::TotalBytesOnDrive::TotalBytesOnDrive ( LCmdInfoPanel& info )
                                 :AbstractItem(info, UPON_REREAD),
                                  str2("%Txt_FPInfoMode_TotalBytesOnDrive")
{
}

int LCmdInfoPanel::TotalBytesOnDrive::getWidth ()
{
   return getWidthOfString(str1, str2, str3);
}

bool LCmdInfoPanel::TotalBytesOnDrive::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   const GVfsLocal& fs = op.vfs.root();
   str1 = GStringUtil::ToByteCountString(fs.getCurrentDriveInfo().diskSize);
   str3 = fs.getCurrentDriveName();
   return true;
}

GString LCmdInfoPanel::TotalBytesOnDrive::getItemString ()
{
   update();
   return str1 + str2 + str3 + '\n';
}

void LCmdInfoPanel::TotalBytesOnDrive::paint ( GGraphics& g )
{
   paintText(g, str1, txtBoldColor, str2, txtColor, str3, txtBoldColor);
}

// --------------------------------------------------------------------------
// BytesFreeOnDrive

LCmdInfoPanel::BytesFreeOnDrive::BytesFreeOnDrive ( LCmdInfoPanel& info )
                                :AbstractItem(info, UPON_1HZ),
                                 str2("%Txt_FPInfoMode_BytesFreeOnDrive"),
                                 diskFree(0)
{
}

int LCmdInfoPanel::BytesFreeOnDrive::getWidth ()
{
   return getWidthOfString(str1, str2, str3);
}

bool LCmdInfoPanel::BytesFreeOnDrive::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   const GVfsLocal& fs = op.vfs.root();
   const GDriveInfo& di = fs.getCurrentDriveInfo();
   longlong newDiskFree = di.diskFree;
   if (diskFree == newDiskFree)
      return false;
   diskFree = newDiskFree;
   str1 = GStringUtil::ToByteCountString(newDiskFree);
   str3 = fs.getCurrentDriveName();
   return true;
}

GString LCmdInfoPanel::BytesFreeOnDrive::getItemString ()
{
   update();
   return str1 + str2 + str3 + '\n';
}

void LCmdInfoPanel::BytesFreeOnDrive::paint ( GGraphics& g )
{
   paintText(g, str1, txtBoldColor, str2, txtColor, str3, txtBoldColor);
}

// --------------------------------------------------------------------------
// PanelFilesStatistic

LCmdInfoPanel::PanelFilesStatistic::PanelFilesStatistic ( LCmdInfoPanel& info )
                                   :AbstractItem(info, UPON_CALC_SIZE),
                                    str2("%Txt_FPInfoMode_PanelFilesStatistic1"),
                                    str4("%Txt_FPInfoMode_PanelFilesStatistic2")
{
}

int LCmdInfoPanel::PanelFilesStatistic::getWidth ()
{
   return getWidthOfString(str1, str2, str3, str4);
}

bool LCmdInfoPanel::PanelFilesStatistic::update ()
{
   longlong numBytes = 0;
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int numFiles = op.items.getCount();
   GString s1 = GInteger::ToString(numFiles);
   const LCmdFileItemContainer& items = op.items; // Make this a fast one
   const int num = items.getCount();
   for (int i=0; i<num; i++)
      numBytes += items[i].fileSize;
   GString s3 = GStringUtil::ToByteCountString(numBytes);
   if (str1 == s1 && str3 == s3)
      return false;
   str1 = s1;
   str3 = s3;
   return true;
}

GString LCmdInfoPanel::PanelFilesStatistic::getItemString ()
{
   update();
   return str1 + str2 + str3 + str4;
}

void LCmdInfoPanel::PanelFilesStatistic::paint ( GGraphics& g )
{
   paintText(g, str1, txtBoldColor, str2, txtColor, str3, txtBoldColor, str4, txtColor);
}

// --------------------------------------------------------------------------
// CurrentDir

LCmdInfoPanel::CurrentDir::CurrentDir ( LCmdInfoPanel& info )
                          :AbstractItem(info, UPON_REREAD)
{
}

int LCmdInfoPanel::CurrentDir::getWidth ()
{
   return getWidthOfString(str1);
}

bool LCmdInfoPanel::CurrentDir::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   str1 = op.getCurrentSysDirectory(false);
   return true;
}

GString LCmdInfoPanel::CurrentDir::getItemString ()
{
   update();
   return str1 + '\n';
}

void LCmdInfoPanel::CurrentDir::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor);
}

// --------------------------------------------------------------------------
// CurrentFileName

LCmdInfoPanel::CurrentFileName::CurrentFileName ( LCmdInfoPanel& info )
                               :AbstractItem(info, UPON_SELECTION)
{
}

int LCmdInfoPanel::CurrentFileName::getWidth ()
{
   return getWidthOfString(str1);
}

bool LCmdInfoPanel::CurrentFileName::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int curSel = op.getCurrentSelectedIndex();
   if (curSel >= 0)
      str1 = op.items[curSel].getFileName();
   else
      str1 = "";
   return true;
}

void LCmdInfoPanel::CurrentFileName::paint ( GGraphics& g )
{
   paintText(g, str1, txtBoldColor);
}

// --------------------------------------------------------------------------
// CurrentFileAttributes

LCmdInfoPanel::CurrentFileAttributes::CurrentFileAttributes ( LCmdInfoPanel& info )
                                     :AbstractItem(info, UPON_SELECTION),
                                      str1("%Txt_FPInfoMode_CurrentFileAttributes")
{
}

int LCmdInfoPanel::CurrentFileAttributes::getWidth ()
{

   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::CurrentFileAttributes::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int curSel = op.getCurrentSelectedIndex();
   if (curSel >= 0)
      str2 = op.items[curSel].getFileAttributesDescrStr();
   else
      str2 = "";
   return true;
}

void LCmdInfoPanel::CurrentFileAttributes::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// CurrentFileCreated

LCmdInfoPanel::CurrentFileCreated::CurrentFileCreated ( LCmdInfoPanel& info )
                                  :AbstractItem(info, UPON_SELECTION),
                                   str1("%Txt_FPInfoMode_CurrentFileCreated")
{
}

int LCmdInfoPanel::CurrentFileCreated::getWidth ()
{

   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::CurrentFileCreated::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int curSel = op.getCurrentSelectedIndex();
   if (curSel >= 0)
   {
      const LCmdFileItem& fi = op.items[curSel];
      str2 = fi.timeCreate.getTimeStampString();
   }
   else
      str2 = "";
   return true;
}

void LCmdInfoPanel::CurrentFileCreated::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// CurrentFileUpdated

LCmdInfoPanel::CurrentFileUpdated::CurrentFileUpdated ( LCmdInfoPanel& info )
                                  :AbstractItem(info, UPON_SELECTION),
                                   str1("%Txt_FPInfoMode_CurrentFileUpdated")
{
}

int LCmdInfoPanel::CurrentFileUpdated::getWidth ()
{

   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::CurrentFileUpdated::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int curSel = op.getCurrentSelectedIndex();
   if (curSel >= 0)
   {
      const LCmdFileItem& fi = op.items[curSel];
      str2 = fi.timeWrite.getTimeStampString();
   }
   else
      str2 = "";
   return true;
}

void LCmdInfoPanel::CurrentFileUpdated::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// CurrentFileAccessed

LCmdInfoPanel::CurrentFileAccessed::CurrentFileAccessed ( LCmdInfoPanel& info )
                                   :AbstractItem(info, UPON_SELECTION),
                                    str1("%Txt_FPInfoMode_CurrentFileAccessed")
{
}

int LCmdInfoPanel::CurrentFileAccessed::getWidth ()
{

   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::CurrentFileAccessed::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int curSel = op.getCurrentSelectedIndex();
   if (curSel >= 0)
   {
      const LCmdFileItem& fi = op.items[curSel];
      str2 = fi.timeAccess.getTimeStampString();
   }
   else
      str2 = "";
   return true;
}

void LCmdInfoPanel::CurrentFileAccessed::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// CurrentFileSize

LCmdInfoPanel::CurrentFileSize::CurrentFileSize ( LCmdInfoPanel& info )
                               :AbstractItem(info, UPON_SELECTION),
                                str1("%Txt_FPInfoMode_CurrentFileSize")
{
}

int LCmdInfoPanel::CurrentFileSize::getWidth ()
{

   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::CurrentFileSize::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   int curSel = op.getCurrentSelectedIndex();
   if (curSel >= 0)
   {
      const LCmdFileItem& fi = op.items[curSel];
      str2 = GStringUtil::ToByteCountString(fi.fileSize);
      if (fi.fileSize >= 1024*1024)
      {
         // Add alternative number with thousand separators only.
         GString buff = GStringUtil::ToByteCountString(fi.fileSize, true);
         str2 += GString(" (%s)", GVArgs(buff));
      }
   }
   else
      str2 = "";
   return true;
}

void LCmdInfoPanel::CurrentFileSize::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// DriveLabel

LCmdInfoPanel::DriveLabel::DriveLabel ( LCmdInfoPanel& info )
                          :AbstractItem(info, UPON_REREAD),
                           str1("%Txt_FPInfoMode_DriveLabel")
{
}

int LCmdInfoPanel::DriveLabel::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::DriveLabel::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   const GVfsLocal& fs = op.vfs.root();
   str2 = fs.getCurrentDriveInfo().volumeName;
   return true;
}

GString LCmdInfoPanel::DriveLabel::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::DriveLabel::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// DriveFileSystem

LCmdInfoPanel::DriveFileSystem::DriveFileSystem ( LCmdInfoPanel& info )
                               :AbstractItem(info, UPON_REREAD),
                                str1("%Txt_FPInfoMode_DriveFileSystem")
{
}

int LCmdInfoPanel::DriveFileSystem::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::DriveFileSystem::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   const GVfsLocal& fs = op.vfs.root();
   str2 = fs.getCurrentDriveInfo().fsName;
   return true;
}

GString LCmdInfoPanel::DriveFileSystem::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::DriveFileSystem::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// DriveSerialNumber

LCmdInfoPanel::DriveSerialNumber::DriveSerialNumber ( LCmdInfoPanel& info )
                                 :AbstractItem(info, UPON_REREAD),
                                  str1("%Txt_FPInfoMode_DriveSerialNumber")
{
}

int LCmdInfoPanel::DriveSerialNumber::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::DriveSerialNumber::update ()
{
   const LCmdFilePanel& op = info.ownerPanel.getOppositePanel();
   const GVfsLocal& fs = op.vfs.root();
   const GDriveInfo& dinf = fs.getCurrentDriveInfo();
   str2 = GString("%04X:%04X", GVArgs(USHORT((dinf.serialNo >> 16) & 0xFFFF))
                                 .add(USHORT(dinf.serialNo & 0xFFFF)));
   return true;
}

GString LCmdInfoPanel::DriveSerialNumber::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::DriveSerialNumber::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// Uptime

LCmdInfoPanel::Uptime::Uptime ( LCmdInfoPanel& info )
                      :AbstractItem(info, UPON_1HZ),
                       str1("%Txt_FPInfoMode_Uptime")
{
   update();
}

int LCmdInfoPanel::Uptime::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::Uptime::update ()
{
   const GTimeAmount& ta = lcmd->sysInfo.getUpTime();
   if (ta.numDays == uptime.numDays && ta.numHours == uptime.numHours && ta.numMinutes == uptime.numMinutes && ta.numSeconds == uptime.numSeconds)
      return false;
   uptime = ta;
   const char *timeSeparator = lcmd->getLocaleData().timeSeparator;
   if (uptime.numDays >= 1)
      str2 = GString("%d %s, %02d%s%02d%s%02d", GVArgs(uptime.numDays).add(uptime.numDays > 1 ? "Days" : "Day").add(uptime.numHours).add(timeSeparator).add(uptime.numMinutes).add(timeSeparator).add(uptime.numSeconds));
   else
      str2 = GString("%02d%s%02d%s%02d", GVArgs(uptime.numHours).add(timeSeparator).add(uptime.numMinutes).add(timeSeparator).add(uptime.numSeconds));
   return true;
}

GString LCmdInfoPanel::Uptime::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::Uptime::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// NumberOfProcesses

LCmdInfoPanel::NumberOfProcesses::NumberOfProcesses ( LCmdInfoPanel& info )
                                 :AbstractItem(info, UPON_1HZ),
                                  str1("%Txt_FPInfoMode_NumberOfProcesses"),
                                  numProcesses(0)
{
   update();
}

int LCmdInfoPanel::NumberOfProcesses::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::NumberOfProcesses::update ()
{
   int pcount = lcmd->sysInfo.getProcessCount();
   if (pcount == numProcesses)
      return false;
   numProcesses = pcount;
   str2 = long(numProcesses);
   return true;
}

GString LCmdInfoPanel::NumberOfProcesses::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::NumberOfProcesses::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// NumberOfThreads

LCmdInfoPanel::NumberOfThreads::NumberOfThreads ( LCmdInfoPanel& info )
                               :AbstractItem(info, UPON_1HZ),
                                str1("%Txt_FPInfoMode_NumberOfThreads"),
                                numThreads(0)
{
   update();
}

int LCmdInfoPanel::NumberOfThreads::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::NumberOfThreads::update ()
{
   int tcount = lcmd->sysInfo.getThreadCount();
   if (tcount == numThreads)
      return false;
   numThreads = tcount;
   str2 = long(numThreads);
   return true;
}

GString LCmdInfoPanel::NumberOfThreads::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::NumberOfThreads::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// SwapFileSize

LCmdInfoPanel::SwapFileSize::SwapFileSize ( LCmdInfoPanel& info )
                            :AbstractItem(info, UPON_REREAD),
                             str1("%Txt_FPInfoMode_SwapFileSize"),
                             swapSize(0)
{
}

int LCmdInfoPanel::SwapFileSize::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::SwapFileSize::update ()
{
   // Prevent GFileNotFoundException in case we runs on a system that is
   // configured not to use any swap-file.
   GVfsLocal& localVfs = info.ownerPanel.vfs.root();
   longlong s = -1;
   if (localVfs.existFile(info.swapPath))
   {
      // Ok, check the size of the swap-file.
      try {
         GFileItem fi(localVfs, info.swapPath);
         s = fi.fileSize;
      } catch (const GFileNotFoundException& /*e*/) {
         s = -1;
      }
   }
   if (s == -1)
   {
      if (GLog::Filter(GLog::TEST))
         GLog::Log(this, "Swap file not found: '%s' (%s:%d)", GVArgs(info.swapPath).add(__FUNCTION__).add(__LINE__));
      s = 0;
   }
   if (s == swapSize)
      return false;
   swapSize = s;
   str2 = GStringUtil::ToByteCountString(swapSize);
   return true;
}

GString LCmdInfoPanel::SwapFileSize::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::SwapFileSize::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

// --------------------------------------------------------------------------
// FreeSwapSpace

LCmdInfoPanel::FreeSwapSpace::FreeSwapSpace ( LCmdInfoPanel& info )
                             :AbstractItem(info, UPON_REREAD),
                              str1("%Txt_FPInfoMode_FreeSwapSpace"),
                              swapFree(0)
{
   update();
}

int LCmdInfoPanel::FreeSwapSpace::getWidth ()
{
   return getWidthOfString(str1, str2);
}

bool LCmdInfoPanel::FreeSwapSpace::update ()
{
   longlong newVal = 0;
   if (info.swapPath[0])
   {
      GDriveInfo swapDrive;
      int drive = GCharacter::ToUpperCase(info.swapPath[0]) - 'A' + 1;
      if (swapDrive.update(drive) == NO_ERROR)
      {
         newVal = swapDrive.numAvailUnits *
                  swapDrive.numSectorsPerUnit *
                  swapDrive.bytesPerSector;
      }
   }
   if (newVal < longlong(info.minFree * 1024L))
      newVal = 0;
   else
      newVal = newVal - longlong(info.minFree * 1024);
   if (newVal == swapFree)
      return false;
   swapFree = newVal;
   str2 = GStringUtil::ToByteCountString(swapFree);
   return true;
}

GString LCmdInfoPanel::FreeSwapSpace::getItemString ()
{
   update();
   return str1 + str2 + '\n';
}

void LCmdInfoPanel::FreeSwapSpace::paint ( GGraphics& g )
{
   paintText(g, str1, txtColor, str2, txtBoldColor);
}

LCmdInfoPanel::LCmdInfoPanel ( LCmdFilePanel& filePanel )
              :ownerPanel(filePanel),
               minFree(0),
               swapPath(QuerySwapPath(&minFree)),
               items(25, 10),
               xpos0(0),
               ypos0(0)
{
   items.add(new Separator(*this));
   items.add(new LCmdVersion(*this));
   items.add(new LCmdCopyright(*this));

   items.add(new Separator(*this));
   items.add(new CurrentFileName(*this));
   items.add(new CurrentFileCreated(*this));
   items.add(new CurrentFileUpdated(*this));
   items.add(new CurrentFileAccessed(*this));
   items.add(new CurrentFileAttributes(*this));
   items.add(new CurrentFileSize(*this));

   items.add(new Separator(*this));
   items.add(new TotalBytesOnDrive(*this));
   items.add(new BytesFreeOnDrive(*this));
   items.add(new PanelFilesStatistic(*this));
   items.add(new CurrentDir(*this));

   items.add(new Separator(*this));
   items.add(new DriveLabel(*this));
   items.add(new DriveFileSystem(*this));
   items.add(new DriveSerialNumber(*this));

   items.add(new Separator(*this));
   items.add(new Uptime(*this));
   items.add(new TotalPhysMem(*this));
   items.add(new FreePhysMem(*this));
   items.add(new NumberOfProcesses(*this));
   items.add(new NumberOfThreads(*this));
   items.add(new SwapFileSize(*this));
   items.add(new FreeSwapSpace(*this));

   items.add(new Separator(*this));
}

LCmdInfoPanel::~LCmdInfoPanel ()
{
}

void LCmdInfoPanel::updateItemRects ()
// Will be called whenever the Owner File Panel has changed size.
{
   GRectangle fr = ownerPanel.viewInfo.filesRect;
   int ypos = fr.y + fr.height - 1;
   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      AbstractItem& pItem = items[i]; // Make this a fast one
      pItem.ypos = ypos;
      ypos -= pItem.getHeight();
   }
}

void LCmdInfoPanel::updateUponSelection ()
{
   if (ownerPanel.view.viewMode != LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      AbstractItem& item = items[i]; // Make this a fast one
      if (item.updt == AbstractItem::UPON_SELECTION)
      {
         if (item.update())
            item.paint();
      }
   }
}

void LCmdInfoPanel::updateUpon1Hz ()
{
   if (ownerPanel.view.viewMode != LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   LCmdFilePanel& panel = ownerPanel.getOppositePanel();
   GVfsLocal& fs = panel.vfs.root();
   fs.updateInternalDriveInfo(false); // Update only if needed.

   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      AbstractItem& item = items[i]; // Make this a fast one
      if (item.updt == AbstractItem::UPON_1HZ)
      {
         if (item.update())
            item.paint();
      }
   }
}

void LCmdInfoPanel::updateUponReread ()
{
   if (ownerPanel.view.viewMode != LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   LCmdFilePanel& panel = ownerPanel.getOppositePanel();
   GVfsLocal& fs = panel.vfs.root();
   fs.updateInternalDriveInfo(true); // Force update!

   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      AbstractItem& item = items[i]; // Make this a fast one
      if (item.updt == AbstractItem::UPON_REREAD ||
          item.updt == AbstractItem::UPON_CALC_SIZE ||
          item.updt == AbstractItem::UPON_SELECTION)
      {
         if (item.update())
            item.paint();
      }
   }
}

void LCmdInfoPanel::updateUponCalcSize ()
{
   if (ownerPanel.view.viewMode != LCmdFilePanelViewOptions::VIEWMODE_INFO)
      return;

   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      AbstractItem& item = items[i]; // Make this a fast one
      if (item.updt == AbstractItem::UPON_CALC_SIZE ||
          item.updt == AbstractItem::UPON_SELECTION)
      {
         if (item.update())
            item.paint();
      }
   }
}

int LCmdInfoPanel::widthOfWidestItem ()
{
   int widest = 0;
   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      int width = items[i].getWidth() + 1;
      if (width > widest)
         widest = width;
   }
   return widest;
}

int LCmdInfoPanel::sumHeightOfItems ()
{
   int ret = 0;
   const int num = items.getCount();
   for (int i=0; i<num; i++)
      ret += items[i].getHeight();
   return ret;
}

void LCmdInfoPanel::paint ( GGraphics& g, const GRectangle* /*rect*/ )
{
   const int num = items.getCount();
   for (int i=0; i<num; i++)
      items[i].paint(g);
}

GString LCmdInfoPanel::QuerySwapPath ( ulonglong* minFree )
{
   *minFree = 0; // Until the actual value is fetched.
   GString sysval = GSystem::ScanSystemConfig("SWAPPATH");
   if (sysval == "")
      sysval = "C:\\OS2\\SYSTEM 0";

   // Find the index of the start of the first SWAPPATH-parameter,
   // which defines the minFree value.
   int spacePos = sysval.indexOf(' ');
   if (spacePos >= 0)
   {
      int pos = spacePos;
      while (sysval[++pos] == ' ');
      GString arg1(16);
      while (GCharacter::IsDigit(sysval[pos]))
         arg1 += sysval[pos++];
      if (arg1 != "")
         *minFree = GULong::ParseULong(arg1);
      sysval.cutTailFrom(spacePos);
   }

   // Now the sysval variable contains only the directory part
   // of the SWAPPATH-parameter.
   GString ret(sysval.length() + 12);
   ret = sysval;
   GFile::Slash(ret);
   ret += "swapper.dat";
   return ret;
}
