/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_FILEPANELMODEABSTRACT
#define __LCMD_FILEPANELMODEABSTRACT

#include "glib/primitives/GObject.h"

/**
 * The base class of all the different file panel view mode panels.
 * It implements all the common behaviour, as well as declaring all 
 * abstract methods that needs different implementations in the 
 * different subclassing view modes.
 *
 * @author  Leif Erik Larsen
 * @since   2006.01.15
 */
class LCmdFilePanelModeAbstract
{
   public:

      LCmdFilePanelModeAbstract ( class LCmdFilePanel& fpanel );
      virtual ~LCmdFilePanelModeAbstract ();

      /**
       * Find the index of which filename item is positioned at the 
       * specified pixel position (typically a mouse position).
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.03
       * @return  The index of the positioned item, or -1 if the 
       *          specified position is not occupied by any item.
       */
      virtual int calcItemIdxFromPos ( int xpos, int ypos ) const = 0;

      /**
       * Calculate the rectangle of where the indexed file item is
       * contained within the file item container.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @return  True if the indexed item is within visible area,
       *          or else false.
       */
      virtual bool calcItemRect ( int itemIndex, class GRectangle& rect ) const = 0;

      /**
       * Draw the single indexed filename item.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.16
       * @param   itemIndex Index (0..) of which file item to paint.
       */
      virtual void drawItem ( int itemIndex ) = 0;
      virtual void drawItem ( int itemIndex, class GGraphics& g, const class GRectangle& itemRect, bool isDragOver = false ) = 0;

      /**
       * Get the index of the current selected filename item in the current
       * view mode, or -1 if there is currently no such selected item.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       */
      virtual int getCurrentSelectedIndex () const = 0;

      /**
       * Get the index of the first item in the list that is more or 
       * less visible within the current view, or -1 of there are 
       * no items being visible at all.
       *
       * @author  Leif Erik Larsen
       * @since   2006.01.24
       */
      virtual int getFirstVisibleIndex () const = 0;

      /**
       * Request all parts of the view to be repainted asynchronously.
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.26
       * @see     GWindow#invalidateAll
       */
      virtual void invalidateAll ( bool inclChildren ) const = 0;

      /**
       * Request the specified rectangle of the view to be 
       * repainted asynchronously.
       *
       * @author  Leif Erik Larsen
       * @since   2006.04.26
       * @see     GWindow#invalidateRect
       */
      virtual void invalidateRect ( const GRectangle& rect ) const = 0;

      /**
       * Return true if and only if the content of the view is horizontally
       * scrollable. That is usually the same as returning true if the width 
       * of the view data is larger than the current width of the view window.
       * Should return true if the view need to handle the VK_LEFT and/or
       * VK_RIGHT keyboard codes.
       * 
       * @author  Leif Erik Larsen
       * @since   2006.02.01
       */
      virtual bool isHorizontallyScrollable () const = 0;

      /**
       * This method is called by {@link LCmdFilePanel#fillList} when the
       * filenames list has been reloaded and refreshed.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.02
       */
      virtual void itemsListHasBeenRefreshed () = 0;

      virtual void layout () = 0;

      virtual int navigateDown () = 0;
      virtual int navigateUp () = 0;
      virtual int navigateEnd () = 0;
      virtual int navigateHome () = 0;
      virtual int navigateLeft () = 0;
      virtual int navigateRight () = 0;
      virtual int navigatePageDown () = 0;
      virtual int navigatePageUp () = 0;
      virtual int navigateRandom ( int index ) = 0;

      /**
       * Called by {@link LCmdFilePanel#setViewMode} when this 
       * view list has been activated.
       * 
       * @author  Leif Erik Larsen
       * @since   2006.02.02
       */
      virtual void onViewHasBeenActivated () = 0;
};

#endif // #ifndef __LCMD_FILEPANELMODEABSTRACT
