/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelInfoBar.h"
#include "lcmd/LCmdFilePanelFrame.h"
#include "lcmd/LCmd.h"

#include "glib/util/GStringUtil.h"

LCmdFilePanelInfoBar::LCmdFilePanelInfoBar ( LCmdFilePanelFrame& parentWin, 
                                             const GString& constraints )
                     :GStatusbar("InfoBar", constraints, parentWin),
                      panelFrame(parentWin)
{
   addCell(new Button(*this, "FileSize", "cmdShowFileInfo", "cmdLeftShowFileInfo", "cmdRightShowFileInfo"));
   addCell(new Button(*this, "FileAttrs", "cmdChooseFileAttrib", "cmdLeftChooseFileAttrib", "cmdRightChooseFileAttrib"));
   addCell(new Button(*this, "FileDate", "cmdChooseFileTime", "cmdLeftChooseFileTime", "cmdRightChooseFileTime"));
   addCell(new Button(*this, "FileTime", "cmdChooseFileTime", "cmdLeftChooseFileTime", "cmdRightChooseFileTime"));
   setSynchronizedHint(true);
   GWindow& hintWin = getHintWindow();
   hintWin.setBackgroundColor(hintWin.defaultBackgroundColor = GColor::DRED);
   hintWin.setForegroundColor(hintWin.defaultForegroundColor = GColor::YELLOW);
}

LCmdFilePanelInfoBar::~LCmdFilePanelInfoBar ()
{
}

LCmdFilePanelInfoBar::Button::Button ( LCmdFilePanelInfoBar& ownerTB, 
                                       const GString& cellName,
                                       const GString& cmdIdCurrent,
                                       const GString& cmdIdLeft,
                                       const GString& cmdIdRight )
                             :GToolbarButton(cellName, // name
                                             GString::Empty, 
                                             GWindow::GetHiddenDummy(), // parentWin, is set upon addCell()
                                             GString::Empty, 
                                             GString::Empty, // text
                                             GString::Empty, // iconID
                                             GString::Empty, 
                                             GToolbarButton::IP_CENTER),
                              infoBar(ownerTB),
                              cmdIdCurrent(cmdIdCurrent),
                              cmdIdLeft(cmdIdLeft),
                              cmdIdRight(cmdIdRight)
{
}

LCmdFilePanelInfoBar::Button::~Button ()
{
}

GString LCmdFilePanelInfoBar::Button::getTooltipText () const
{
   Button* self = const_cast<Button*>(this);
   self->updateIDString();
   GString id(getIDString());
   GString txt(getKeyboardShortcutKeyText());
   txt += GProgram::LoadText("%" + id, GResourceTable::LT_PREFER_TEXT, 0);
   return txt;
}

void LCmdFilePanelInfoBar::Button::updateIDString ()
{
   LCmdFilePanel& fp = infoBar.panelFrame.fpanel;
   if (fp.isCurrentPanel())
      setIDString(cmdIdCurrent);
   else
   if (fp.isLeftPanel())
      setIDString(cmdIdLeft);
   else
      setIDString(cmdIdRight);
}

void LCmdFilePanelInfoBar::Button::performAction ()
{
   updateIDString();
   GAbstractCommand& cmd = getCommand();
   postCommand(&cmd);
}

bool LCmdFilePanelInfoBar::Button::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (!isEnabled())
      infoBar.panelFrame.fpanel.activatePanel();
   return GToolbarButton::onButton1Down(xpos, ypos, flags);
}

void LCmdFilePanelInfoBar::updateAllFileItemInfoCells_NoSelection ()
{
   setText("%Txt_FPInfo_NoSelectedFile"); // "No selected file"
   setHint(GString::Empty);
   setCellText("FileSize", GString::Empty);
   setCellText("FileDate", GString::Empty);
   setCellText("FileTime", GString::Empty);
   setCellText("FileAttrs", GString::Empty);
}

void LCmdFilePanelInfoBar::updateAllFileItemInfoCells_SingleItem ( LCmdFileItem& item )
{
   GString sizeStr;
   if (item.isDirectory() && !(item.getInternalFlags() & LCmdFileItem::FINFO_SHOWSIZE))
      sizeStr = (item.isUpDir() ? lcmd->txtUpDir : lcmd->txtSubDir);
   else
      sizeStr = GStringUtil::ToByteCountString(item.fileSize, false, false, false);

   GString flagStr(4);
   flagStr += item.isArchive() ? 'a' : '.';
   flagStr += item.isReadOnly() ? 'r' : '.';
   flagStr += item.isHidden() ? 'h' : '.';
   flagStr += item.isSystem() ? 's' : '.';

   setText(item.getFileName());
   setHint(GString::Empty);
   setCellText("FileSize", sizeStr);
   setCellText("FileDate", item.timeWrite.getDateString());
   setCellText("FileTime", item.timeWrite.getTimeString());
   setCellText("FileAttrs", flagStr);
}

void LCmdFilePanelInfoBar::updateAllFileItemInfoCells ()
{
   LCmdFilePanel& panel = panelFrame.fpanel;
   int curSel = panel.getCurrentSelectedIndex();
#if __ENABLE_TREE_VIEW
   if (panel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_TREE)
   {
      // In the tree-view, always display information for the current 
      // selected single node only.
      LCmdFileItem* item = panel.viewTree.getCurrentSelectedFileItem();
      if (item == null)
         updateAllFileItemInfoCells_NoSelection();
      else
         updateAllFileItemInfoCells_SingleItem(*item);
   }
   else
#endif
   if (panel.items.getCount() <= 0)
   {
      setText("%Txt_FPInfo_NoFiles"); // "No files"
      setHint(GString::Empty);
      setCellText("FileSize", GString::Empty);
      setCellText("FileDate", GString::Empty);
      setCellText("FileTime", GString::Empty);
      setCellText("FileAttrs", GString::Empty);
   }
   else
   if (panel.markedFilesCount == 1)
   {
      GVArgs args;
      args.add(GStringUtil::ToByteCountString(panel.sizeOfMarkedFiles, false, false, false));
      GStringl txt("%Txt_FPInfo_BytesInFile", args);
      setHint(txt);
      setText(GString::Empty);
   }
   else
   if (panel.markedFilesCount > 1)
   {
      GVArgs args;
      args.add(GStringUtil::ToByteCountString(panel.sizeOfMarkedFiles, false, false, false));
      args.add(panel.markedFilesCount);
      GStringl txt("%Txt_FPInfo_BytesInFiles", args);
      setHint(txt);
      setText(GString::Empty);
   }
   else
   if (curSel < 0 ||
       curSel >= panel.items.getCount())
   {
      updateAllFileItemInfoCells_NoSelection();
   }
   else
   {
      LCmdFileItem& item = panel.items.get(curSel);
      updateAllFileItemInfoCells_SingleItem(item);
   }

   // Layout the cells synchronously, to make it look crispy to the user.
   layout();
   updateWindow();
}

int LCmdFilePanelInfoBar::getPreferredHeight () const
{
   LCmdFilePanelViewOptions& vo = panelFrame.fpanel.view;
   if (vo.showInfobar && 
       vo.viewMode != LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      return GStatusbar::getPreferredHeight();
   }
   return 0;
}
