/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFilePanelHeader.h"
#include "lcmd/LCmdDynaFileSearch.h"
#include "lcmd/LCmdFilePanelFrame.h"
#include "lcmd/LCmd.h"

#include "glib/gui/border/GLineBorder.h"
#include "glib/util/GMath.h"

LCmdFilePanelHeader::LCmdFilePanelHeader ( LCmdFilePanelFrame& parentWin, 
                                           const GString& constraints )
                    :GAbstractToolbarWindow("Header", 
                                            constraints, 
                                            parentWin, 
                                            true, 
                                            WS_VISIBLE, 
                                            WS2_IGNORE_COLORS_PROFILE | WS2_AUTO_SHOW_TOOLTIP | WS2_OS2Y),
                     panelFrame(parentWin),
                     driveButt(null),
                     upDirButt(null),
                     rootButt(null),
                     overriddenTooltipPos(GTooltip::PosInheritFromParent),
                     dynaFileSearch(null)
{
   setInsets(new GInsets(1, 0, 2, 0), true);
   setBorder(new GLineBorder(GLineBorder::AlignInvisible, GLineBorder::AlignOuter, GLineBorder::AlignInvisible, GLineBorder::AlignInvisible), true);

   setFlatButtonBorders(true);
   defaultFontNameSize = lcmd->options.DefaultFontFilePanelHeader;
   setFontNameSize(defaultFontNameSize);

   addButton(driveButt = new DriveButton(*this), true);
   addButton(upDirButt = new UpDirButton(*this), true);
   addButton(rootButt = new RootButton(*this), true);
   dynaFileSearch = new LCmdDynaFileSearch(*this);

   driveButt->setTooltipPosition(GTooltip::PosBelow);
   upDirButt->setTooltipPosition(GTooltip::PosBelow);
   rootButt->setTooltipPosition(GTooltip::PosBelow);
}

LCmdFilePanelHeader::~LCmdFilePanelHeader ()
{
   delete dynaFileSearch;
}

LCmdFilePanelHeader::Button::Button ( LCmdFilePanelHeader& ownerTB, 
                                      const GString& text, 
                                      const GString& iconID )
                            :GToolbarButton(GString::Empty, 
                                            GString::Empty, 
                                            ownerTB, 
                                            GString::Empty, 
                                            text, 
                                            iconID, 
                                            GString::Empty, 
                                            GToolbarButton::IP_CENTER),
                             headerBar(ownerTB)

{
   setInsets(new GInsets(1, 2, 1, 2), true);
   setBorder(new GLineBorder(), true);
}

LCmdFilePanelHeader::Button::~Button ()
{
}

void LCmdFilePanelHeader::Button::performAction ()
{
   updateIDString();
   GAbstractCommand& cmd = getCommand();
   postCommand(&cmd);
}

bool LCmdFilePanelHeader::Button::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags )
{
   if (!isEnabled())
      headerBar.panelFrame.fpanel.activatePanel();
   return GToolbarButton::onButton1Down(xpos, ypos, flags);
}

int LCmdFilePanelHeader::Button::getPreferredWidth () const
{
   return GToolbarButton::getPreferredWidth() + 4;
}

LCmdFilePanelHeader::DriveButton::DriveButton ( LCmdFilePanelHeader& ownerTB )
                                 :Button(ownerTB, GString::Empty/*, "IDP_DRIVE_HARDDISK"*/)
{
}

LCmdFilePanelHeader::DriveButton::~DriveButton ()
{
}

void LCmdFilePanelHeader::DriveButton::updateIDString ()
{
   if (headerBar.panelFrame.fpanel.isLeftPanel())
      setIDString("cmdLeftChooseDrive");
   else
      setIDString("cmdRightChooseDrive");
}

GString LCmdFilePanelHeader::DriveButton::getTooltipText () const
{
   LCmdFilePanelHeader::Button* self = const_cast<LCmdFilePanelHeader::DriveButton*>(this);
   self->updateIDString();
   GString id(getIDString());
   GString txt(getKeyboardShortcutKeyText());
   txt += GProgram::LoadText("%" + id, GResourceTable::LT_PREFER_TEXT, 0);
   return txt;
}

GString LCmdFilePanelHeader::DriveButton::getText () const
{
   const LCmdFilePanel& panel = headerBar.panelFrame.fpanel;
   if (panel.vfs.isEmpty())
      return GString::Empty;
   const GVfsLocal& fs = panel.vfs.root();
   return fs.getCurrentDriveName();
}

LCmdFilePanelHeader::UpDirButton::UpDirButton ( LCmdFilePanelHeader& ownerTB )
                                 :Button(ownerTB, GString::Empty, "IDP_DIRUP")
{
}

LCmdFilePanelHeader::UpDirButton::~UpDirButton ()
{
}

void LCmdFilePanelHeader::UpDirButton::updateIDString ()
{
   if (headerBar.panelFrame.fpanel.isLeftPanel())
      setIDString("cmdLeftWalkDirectoryUp");
   else
      setIDString("cmdRightWalkDirectoryUp");
}

GString LCmdFilePanelHeader::UpDirButton::getKeyboardShortcutKeyText () const
{
   LCmdFilePanelHeader::Button* self = const_cast<LCmdFilePanelHeader::UpDirButton*>(this);
   self->updateIDString();
   GString id(getIDString());
   GString txt(getKeyboardShortcutKeyTextForID(id)); // First, try the true (left/right specific) command ID
   if (txt == "")
   {
      // There are no keyboard code specifically for the left/right command,
      // so use the keyboard code as of the "current" file panel, but
      // only if we actually are the current file panel then.
      if (headerBar.panelFrame.fpanel.isCurrentPanel())
         txt = getKeyboardShortcutKeyTextForID("cmdWalkDirectoryUp");
   }
   return txt;
}

GString LCmdFilePanelHeader::UpDirButton::getTooltipText () const
{
   GString txt(getKeyboardShortcutKeyText());
   GString txtID = "%" + getIDString();
   txt += GProgram::LoadText(txtID, GResourceTable::LT_PREFER_TEXT);
   return txt;
}

LCmdFilePanelHeader::RootButton::RootButton ( LCmdFilePanelHeader& ownerTB )
                                :Button(ownerTB, GString::Empty, "IDP_DIRROOT")
{
}

LCmdFilePanelHeader::RootButton::~RootButton ()
{
}

void LCmdFilePanelHeader::RootButton::updateIDString ()
{
   if (headerBar.panelFrame.fpanel.isLeftPanel())
      setIDString("cmdLeftWalkRootDirectory");
   else
      setIDString("cmdRightWalkRootDirectory");
}

GString LCmdFilePanelHeader::RootButton::getKeyboardShortcutKeyText () const
{
   LCmdFilePanelHeader::Button* self = const_cast<LCmdFilePanelHeader::RootButton*>(this);
   self->updateIDString();
   GString id(getIDString());
   GString txt(getKeyboardShortcutKeyTextForID(id)); // First, try the true (left/right specific) command ID
   if (txt == "")
   {
      // There are no keyboard code specifically for the left/right command,
      // so use the keyboard code as of the "current" file panel, but
      // only if we actually are the current file panel then.
      if (headerBar.panelFrame.fpanel.isCurrentPanel())
         txt = getKeyboardShortcutKeyTextForID("cmdWalkRootDirectory");
   }
   return txt;
}

GString LCmdFilePanelHeader::RootButton::getTooltipText () const
{
   GString txt(getKeyboardShortcutKeyText());
   GString txtID = "%" + getIDString();
   txt += GProgram::LoadText(txtID, GResourceTable::LT_PREFER_TEXT, 0);
   return txt;
}

bool LCmdFilePanelHeader::onButton1Down ( int /*xpos*/, int /*ypos*/, const GWindowMessage::InputFlags& /*flags*/ )
{
   GProgram& prg = GProgram::GetProgram();
   GWindow& mwin = prg.getMainWindow();
   mwin.setActive();
   LCmdFilePanel& fp = panelFrame.fpanel;
   fp.activatePanel();
   return true;
}

bool LCmdFilePanelHeader::onBackgroundColorChanged ( const GColor& color )
{
   LCmdFilePanel& panel = panelFrame.fpanel;
   if (panel.vfs.isRootOnly() ||
       panel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      if (panel.isCurrentPanel())
         panel.colors.headerActiveBck = color;
      else
         panel.colors.headerInactiveBck = color;
   }
   else
   {
      if (panel.isCurrentPanel())
         panel.colors.headerActiveBckWhenInsideVfs = color;
      else
         panel.colors.headerInactiveBckWhenInsideVfs = color;
   }
   return true;
}

bool LCmdFilePanelHeader::onForegroundColorChanged ( const GColor& color )
{
   LCmdFilePanel& panel = panelFrame.fpanel;
   if (panel.vfs.isRootOnly() ||
       panel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      if (panel.isCurrentPanel())
         panel.colors.headerActiveTxt = color;
      else
         panel.colors.headerInactiveTxt = color;
   }
   else
   {
      if (panel.isCurrentPanel())
         panel.colors.headerActiveTxtWhenInsideVfs = color;
      else
         panel.colors.headerInactiveTxtWhenInsideVfs = color;
   }
   return true;
}

const GColor& LCmdFilePanelHeader::getPreferredBckColor () const
{
   LCmdFilePanel& panel = panelFrame.fpanel;
   if (panel.vfs.isRootOnly() ||
       panel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      if (panel.isCurrentPanel())
         return panel.colors.headerActiveBck;
      else
         return panel.colors.headerInactiveBck;
   }
   else
   {
      if (panel.isCurrentPanel())
         return panel.colors.headerActiveBckWhenInsideVfs;
      else
         return panel.colors.headerInactiveBckWhenInsideVfs;
   }
}

const GColor& LCmdFilePanelHeader::getPreferredTxtColor () const
{
   LCmdFilePanel& panel = panelFrame.fpanel;
   if (panel.vfs.isRootOnly() ||
       panel.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      if (panel.isCurrentPanel())
         return panel.colors.headerActiveTxt;
      else
         return panel.colors.headerInactiveTxt;
   }
   else
   {
      if (panel.isCurrentPanel())
         return panel.colors.headerActiveTxtWhenInsideVfs;
      else
         return panel.colors.headerInactiveTxtWhenInsideVfs;
   }
}

void LCmdFilePanelHeader::invalidateClientAreaForText ()
{
   GRectangle r = getWindowFaceRect();
   invalidateRect(r);
}

bool LCmdFilePanelHeader::onPaint ( GGraphics& g, const GRectangle& rect )
{
   LCmdFilePanel& fp = panelFrame.fpanel;

   GRectangle r = titleRect;
   g.setColor(GColor::DGRAY);
   g.drawRectangle(r);
   r.inflateRect(-1);
   GColor bckColor = getPreferredBckColor();
   g.drawFilledRectangle(r, bckColor);

   GString txt;
   int x, y;
   if (fp.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO)
   {
      txt = GStringl("%Txt_FPInfoMode_TitleInHeader");
      setTooltipText(txt);
      GDimension txtDim = g.getTextDim(txt);
      x = r.x + r.width/2 - txtDim.width/2;
      y = r.y + r.height/2 - txtDim.height/2;
   }
   else
   {
      // If the Drives-Button is visible we shouldn't paint the drive as part
      // of the directory. That would have been redundant, since the current
      // drive is already painted on the visible Drives-Button.
      txt = fp.vfs.getFullVirtualPath(false);
      setTooltipText(txt);
      if (fp.view.showHeaderDrivButt && GFile::ContainsDrive(txt))
         txt = txt.substring(2); // Remove the drive from the path.
      GDimension txtDim = g.getTextDim(txt);
      r.nudgeX(2);
      x = r.x;
      y = r.y + r.height/2 - txtDim.height/2;
   }

   if (panelFrame.fpanel.isCurrentPanel())
      g.setColor(GColor::BLACK);
   else
      g.setColor(GColor::DGRAY);

   // Make the text fit into visible area (if needed).
   if (g.getWidthOfString(txt) > r.width)
      txt = g.getShortenedText(txt, r.width);

   // Draw the text.
   g.drawText(x+1, y-1, txt);
   GColor txtColor = getPreferredTxtColor();
   g.setColor(txtColor);
   g.drawText(x, y, txt);

   return true;
}

void LCmdFilePanelHeader::layout ()
{
   LCmdFilePanel& fp = panelFrame.fpanel;
   bool isInfoMode = (fp.view.viewMode == LCmdFilePanelViewOptions::VIEWMODE_INFO);
   bool showDriveButt = driveButt != null && fp.view.showHeaderDrivButt && !isInfoMode;
   bool showUpDirButt = upDirButt != null && fp.view.showHeaderUpDirButt && !isInfoMode;
   bool showRootButt = rootButt != null && fp.view.showHeaderRootButt && !isInfoMode;
   int widthDrive = showDriveButt ? driveButt->getPreferredWidth() : 0;
   int widthUpDir = showUpDirButt ? upDirButt->getPreferredWidth() : 0;
   int widthRootDir = showRootButt ? rootButt->getPreferredWidth() : 0;

   GRectangle r = getWindowFaceRect();

   // Rectangle of the Drives button
   if (driveButt != null)
      driveButt->setWindowBounds(r.x, r.y, widthDrive, r.height);

   // Rectangle of the Title-String area of the header
   titleRect.x = r.x + widthDrive;
   titleRect.width = r.width - widthRootDir - widthUpDir - titleRect.x;
   titleRect.y = r.y;
   titleRect.height = r.height;
   if (!showDriveButt)
      titleRect.nudgeX(1);
   if (!showUpDirButt && !showRootButt)
      titleRect.width -= 1;

   // Rectangle of the Up-Directory button
   int xpos = r.width - widthRootDir - widthUpDir;
   if (upDirButt != null)
      upDirButt->setWindowBounds(xpos, r.y, widthUpDir, r.height);

   // Rectangle of the Root-Directory button
   xpos = r.width - widthRootDir;
   if (rootButt != null)
      rootButt->setWindowBounds(xpos, r.y, widthRootDir, r.height);

   // Adjust the size of the entryfield window to use when user performs a
   // dynamic filesearch.
   if (dynaFileSearch != null)
      dynaFileSearch->setWindowBounds(r.x, r.y, r.width, r.height);
}

int LCmdFilePanelHeader::getPreferredHeight () const
{
   LCmdFilePanel& fp = panelFrame.fpanel;
   LCmdFilePanelViewOptions& viewOpt = fp.view;
   if (!viewOpt.showHeaderbar)
      return 0;
   int driveButtH = (driveButt == null ? 0 : driveButt->getPreferredHeight());
   int titleBarH = getHeightOfString("X") + 7;
   int upDirButtH = (upDirButt == null ? 0 : upDirButt->getPreferredHeight());
   int rootButtH = (rootButt == null ? 0 : rootButt->getPreferredHeight());
   return GMath::Max(driveButtH, GMath::Max(titleBarH, GMath::Max(upDirButtH, rootButtH)));
}

void LCmdFilePanelHeader::dismissDynamicSearch ()
{
   if (dynaFileSearch != null)
      dynaFileSearch->setVisible(false);
   LCmdMainWindow& mwin = LCmdMainWindow::GetMainWindow();
   mwin.ensureFocusOK();
}

bool LCmdFilePanelHeader::isPerformingDynamicSearch ()
{
   if (dynaFileSearch == null)
      return false;
   return dynaFileSearch->isVisible();
}

GString LCmdFilePanelHeader::getTooltipText () const
{
   // By default, use the tooltip position as set in our super class.
   overriddenTooltipPos = GTooltip::PosInheritFromParent;

   // ---
   GString text = GWindow::getTooltipText();
   GGraphics g(this);
   GDimension dim(titleRect.width, titleRect.height);
   GDimension textDim = g.getTextDim(text);

   // Show the current text as a tooltip if the text is too long to be all 
   // visible within the window face area.
   if (textDim.height >= dim.height || textDim.width >= dim.width)
   {
      // Show the full length of the text in a tooltip box that is 
      // overlapping our self.
      overriddenTooltipPos = GTooltip::PosOverlap;
      return text;
   }

   // The component is wide enough to show all parts of the text, so we 
   // don't need to show the text as a tooltip.
   return GString::Empty;
}

GTooltip::Position LCmdFilePanelHeader::getTooltipPosition () const
{
   if (overriddenTooltipPos != GTooltip::PosInheritFromParent)
      return overriddenTooltipPos;
   return GAbstractToolbarWindow::getTooltipPosition();
}

bool LCmdFilePanelHeader::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags ) 
{ 
   if (titleRect.isPointInRect(xpos, ypos))
      GWindow::onMouseMove(xpos, ypos, flags);
   else
   if (isMouseCapture())
      captureMouse(false);
   return true;
}
