/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdFileItemContainer.h"
#include "lcmd/LCmdFilePanel.h"

#include "glib/gui/GDialogPanel.h"
#include "glib/util/GHashtable.h"
#include "glib/util/GMath.h"
#include "glib/vfs/GVfsLocal.h"

LCmdFileItemContainer::LCmdFileItemContainer ( LCmdFilePanel& fpanel )
                      :GWorkerThread("DlgFileNameReaderStatus", 200),
                       fpanel(fpanel),
                       items(4096),
                       listeners(16),
                       runStep(FIND_FIRST),
                       cancelSem(false),
                       preLoadCounter(0),
                       synchPaintPanel(false)
{
}

LCmdFileItemContainer::~LCmdFileItemContainer ()
{
}

void LCmdFileItemContainer::addFileItemContainerListener ( LCmdFileItemContainer::Listener* l )
{
   if (!listeners.contains(l))
      listeners.add(l);
}

void LCmdFileItemContainer::removeFileItemContainerListener ( LCmdFileItemContainer::Listener* l )
{
   int idx = listeners.indexOf(l);
   if (idx >= 0)
      listeners.remove(idx);
}

LCmdFileItem& LCmdFileItemContainer::get ( int index ) const throw(GArrayIndexOutOfBoundsException) 
{ 
   return items.get(index); 
}

LCmdFileItem& LCmdFileItemContainer::operator[] ( int index ) const 
{ 
   return items[index]; 
}

int LCmdFileItemContainer::getCount () const 
{ 
   return items.getCount(); 
}

void LCmdFileItemContainer::onWorkerThreadInitDialog ( GWorkerThread& worker, 
                                                       GDialogPanel& monitor )
{
   monitor.setComponentText("101", "%Txt_DlgFileNameReaderStatus_Leader1");
   monitor.setComponentText("102", GString::Empty);
   monitor.setComponentText("DLG_CANCEL", "%DLG_STOP");
   monitor.setComponentEnabled("DLG_CANCEL", true);

   // Request panel area to repaint. This is to remove all
   // old filenames and paint the "Reading Filenames..."
   // message while we wait for the reader thread to finish.
   LCmdFilePanelModeAbstract& view = fpanel.getCurrentView();
   view.invalidateAll(true);

   // Paint synchronously if requested.
   if (synchPaintPanel)
   {
      fpanel.updateWindow();
      synchPaintPanel = false; // Back to default.
   }
}

void LCmdFileItemContainer::onWorkerThreadCommand ( GWorkerThread& worker, 
                                                    GDialogPanel& monitor, 
                                                    const GString& cmd )
{
   if (cmd == "DLG_CANCEL")
   {
      // Give the worker thread a minimum of some milliseconds so
      // that at least a very few filename items are loaded.
      if (GWorkerThread::getTimeMillisSinceStart(false) >= 250)
      {
         cancelSem = true;
         requestStop();
         monitor.setComponentDisabled("DLG_CANCEL");
      }
   }
}

void LCmdFileItemContainer::onWorkerThreadUpdateMonitor ( GWorkerThread& worker, 
                                                          GDialogPanel& monitor )
{
   if (runStep == SORT_ITEMS)
   {
      // No need to update anything while the background thread
      // is sorting the loaded filename items.
   }
   else
   if (runStep == FIND_FIRST && !fpanel.vfs.isRootOnly())
   {
      int newCount = preLoadCounter;
      monitor.setComponentValue("102", newCount);
   }
   else
   {
      int newCount = items.getCount();
      monitor.setComponentValue("102", newCount);
   }
}

void LCmdFileItemContainer::onWorkerThreadUserEvent ( GWorkerThread& worker, 
                                                      GDialogPanel& monitor, 
                                                      const GString& msgID, 
                                                      GObject* /*userParam*/ )
{
   if (msgID == "ALL_ITEMS_HAVE_BEEN_REMOVED")
   {
      // All filename items have now been removed by the background thread.
      // Call every external listener about this event.
      for (int i=0, num=listeners.getCount(); i<num; i++)
      {
         Listener* l = listeners.elementAt(i);
         if (l != null)
            l->allItemsHaveBeenRemoved();
      }
   }
   else
   if (msgID == "STARTING_FINDFIRST")
   {
      // Change "Number of files" to "Preloading VFS items".
      monitor.setComponentValue("101", GStringl("%Txt_DlgFileNameReaderStatus_Leader1_VfsPreLoad"));
      monitor.setComponentValue("102", "");
   }
   else
   if (msgID == "FINISHED_FINDFIRST")
   {
      // Change "Preloading VFS items" back to "Number of files".
      monitor.setComponentValue("101", GStringl("%Txt_DlgFileNameReaderStatus_Leader1"));
      monitor.setComponentValue("102", "");
   }
   else
   if (msgID == "NOW_SORTING")
   {
      // "Sorting items..."
      runStep = SORT_ITEMS;
      monitor.setComponentValue("101", GStringl("%Txt_DlgFileNameReaderStatus_Leader1_Sorting"));
      monitor.setComponentValue("102", items.getCount());
      monitor.setComponentDisabled("DLG_CANCEL");
   }
}

void LCmdFileItemContainer::runTheWorkerThread ( GWorkerThread& worker )
{
   // Don't keep file items selection if directory has changed.
   GString curDir = fpanel.vfs.getFullVirtualPath(false);
   if (curDir != previousDir)
      fpanel.markedFilesCount = 0;
   previousDir = curDir;

   // To be able to preserve the marked items we must first get and keep an
   // array of all all the currently marked items.
   GHashtable<GString, GString> markedItems(GMath::Max(1, fpanel.markedFilesCount));
   if (fpanel.markedFilesCount > 0)
   {
      const int num = items.getCount();
      for (int i=0; i<num; i++)
      {
         LCmdFileItem& fitem = items[i]; // Make this a fast one
         if (fitem.isMarked())
         {
            GString* str = new GString(fitem.getFileName());
            markedItems.put(str, null, true, false);
         }
      }
   }

   // Free all previous entries from the table of files.
   freeItems();

   // Read filenames from the current directory of the active file system.
   GFileItem fitem;
   runStep = FIND_FIRST;
   cancelSem = false;
   preLoadCounter = 0;
   GVfs& fs = fpanel.vfs.peek();
   if (fs.findFirstMightNeedLongTime())
      sendUserMessageToMonitor("STARTING_FINDFIRST");
   bool caseFilter = !fs.isFileNameCasePreserved();
   int hdir = fs.findFirst(fitem, GString::Empty, &cancelSem, &preLoadCounter);
   runStep = FIND_NEXT;
   if (fs.findFirstMightNeedLongTime())
      sendUserMessageToMonitor("FINISHED_FINDFIRST");
   GVfsLocal& sfs = fpanel.vfs.root();
   bool ignoreUpDir = (sfs.isCurrentDirectoryTheRoot() && fpanel.vfs.isRootOnly());
   bool foundUpDir = false; // Will be true when we have found the ".." directory.
   int countAddedItems = 0;
   if (hdir != 0) do 
   {
      bool ignore = false;   // Don't ignore filename until opposite is proven.

      if (fitem.isDirectory())
      {
         if (!fpanel.filter.showDirs)
            ignore = true;   // User has choosen not to include directories.
         else
         if (fitem.isThisDir())
            ignore = true;   // Don't include the "dummy" this-directory.
         else
         if (ignoreUpDir && fitem.isUpDir())
            ignore = true;   // Don't include the "dummy" up-directory in root.
      }
      else
      {
         if (!fpanel.filter.showFiles)
            ignore = true;      // User has choosen not to include any files.
      }

      if (!ignore)           // Test if this item is to be hidden from panel.
      {
         const LCmdFilePanelFilterOptions& filt = fpanel.filter;
         if ((filt.hideArchive  && fitem.isArchive()) ||
             (filt.hideHidden   && fitem.isHidden()) ||
             (filt.hideSystem   && fitem.isSystem()) ||
             (filt.hideReadOnly && fitem.isReadOnly()))
         {
            if (!fitem.isUpDir()) // Never hide the ".." directory!
               ignore = true;
         }

         else
         if (!fitem.isDirectory()) // Always include directories.
         {
            const GString& fname = fitem.getFileName();
            if (!GFile::IsFileNameInFilterList(fname, filt.include) ||
                 GFile::IsFileNameInFilterList(fname, filt.exclude))
            {
               ignore = true;
            }
         }
      }

      if (ignore) // If this filename is to be filtered out.
      {
         if (GLog::Filter(GLog::DEBUG))
            GLog::Log(this, "Filename is filtered out: '%s' (%s:%d)", GVArgs(fitem.getFileName()).add(__FUNCTION__).add(__LINE__));
      }
      else
      {
         if (caseFilter) // Convert filename in memory to lowercase if needed.
         {
            GString name = fitem.getName();
            GString ext = fitem.getExtension();
            fitem.setName(name.toLowerCase());
            fitem.setExtension(ext.toLowerCase());
         }
         if (fitem.getName() == "..")
            foundUpDir = true;
         LCmdFileItem* fi = new LCmdFileItem(fpanel, fitem);
         items.add(fi);
         fi->unsortedIndex = countAddedItems++;
         if (GLog::Filter(GLog::DEBUG))
            GLog::Log(this, "Include filename ##%04d: '%s' (%s:%d)", GVArgs(countAddedItems).add(fitem.getFileName()).add(__FUNCTION__).add(__LINE__));

         // Re-mark the item, if it was marked when we was called.
         if (markedItems.size() > 0)
         {
            GString fname = fi->getFileName();
            if (markedItems.containsKey(fname))
               fpanel.tagItem(fi->unsortedIndex, true, false);
         }
      }

      // By checking of stop-request here instead of in the top of the
      // do/while-block we ensure that at least the ".." in archive files
      // will be added to the file panel so that the user will always get
      // an easy way to walk out of the archive file. This is true because
      // {@link GVfs#findFirst} is documented to return
      // the ".." item if possible.
      if (isStopRequested()) // If we have been requested to cancel.
         break;
   } 
   while (fs.findNext(hdir, fitem));

   if (hdir != 0)
      fs.findClose(hdir);

   // Programatically add the ".." if directory is not the roor, and we 
   // did not find it. This may happen sometimes on some file systems.
   // I have seen it e.g. on "unfinished CD's" on Windows XP.
   if (!foundUpDir && !ignoreUpDir)
   {
      fitem = GFileItem();
      fitem.setName("..");
      fitem.setDirectoryFlag(true);
      LCmdFileItem* fi = new LCmdFileItem(fpanel, fitem);
      items.add(fi);
      fi->unsortedIndex = countAddedItems++;
   }

   int firstIdx = countAddedItems > 0 ? 0 : -1;
   fpanel.viewBrief.curSelectedItem = firstIdx;
   fpanel.viewBrief.firstVisibleItem = firstIdx;
   fpanel.viewFull.list.curSelectedItem = firstIdx;
   fpanel.viewFull.list.firstVisibleItem = firstIdx;
   fpanel.viewWide.list.curSelectedItem = firstIdx;
   fpanel.viewWide.list.firstVisibleItem = firstIdx;

   // Sort the list of newly read filenames, using sorting options of the panel.
   sendUserMessageToMonitor("NOW_SORTING");
   fpanel.sortList(false, false);
}

void LCmdFileItemContainer::freeItems ()
{
   freeItemIcons();
   items.removeAll();

   // Notify the GUI thread that we have removed all items.
   // The GUI thread will then call every external listener about the same.
   sendUserMessageToMonitor("ALL_ITEMS_HAVE_BEEN_REMOVED");
}

void LCmdFileItemContainer::freeItemIcons ()
{
   const int num = items.getCount();
   for (int i=0; i<num; i++)
   {
      LCmdFileItem& fitem = items.get(i);
      fitem.unloadIcon();
   }
}

void LCmdFileItemContainer::sort ()
{
   items.sortItems((GComparator&) *this);
}

int LCmdFileItemContainer::compare2Objects ( const GObject& obj1, 
                                             const GObject& obj2 ) const
{
   return compareItems(dynamic_cast<const LCmdFileItem&>(obj1), 
                       dynamic_cast<const LCmdFileItem&>(obj2));
}

int LCmdFileItemContainer::compareItems_ ( const LCmdFileItem& file1, 
                                           const LCmdFileItem& file2, 
                                           int fldIdx ) const
{
   int res = 0;

   switch (fpanel.sortOpt.what[fldIdx])
   {
      case LCmdFilePanelSortOptions::PSW_UNSORTED:
      {
         if (file1.unsortedIndex < file2.unsortedIndex)
            res = -1;
         else
         if (file1.unsortedIndex > file2.unsortedIndex)
            res = 1;
         break;
      }

      case LCmdFilePanelSortOptions::PSW_TYPE:
      {
         bool dir1 = file1.isDirectory();
         bool dir2 = file2.isDirectory();
         if (dir1 != dir2)
         {
            if (dir1 > dir2)
               res = -1;
            else
               res = 1;
         }
         break;
      }

      case LCmdFilePanelSortOptions::PSW_NAME:
      {
         GString name1 = file1.getName();
         if (file1.isDirectory())
            name1 += file1.getExtension();
         GString name2 = file2.getName();
         if (file2.isDirectory())
            name2 += file2.getExtension();
         res = name1.compareIgnoreCase(name2);
         break;
      }

      case LCmdFilePanelSortOptions::PSW_EXTENTION:
      {
         GString ext1;
         if (!file1.isDirectory())
            ext1 = file1.getExtension();
         GString ext2;
         if (!file2.isDirectory())
            ext2 = file2.getExtension();
         res = ext1.compareIgnoreCase(ext2);
         break;
      }

      case LCmdFilePanelSortOptions::PSW_DATE:
      {
         const GDate& date1 = file1.timeWrite.getDate();
         const GDate& date2 = file2.timeWrite.getDate();
         if (date1 < date2)
            res = -1;
         else
         if (date1 > date2)
            res = 1;
         break;
      }

      case LCmdFilePanelSortOptions::PSW_TIME:
      {
         const GTime& time1 = file1.timeWrite.getTime();
         const GTime& time2 = file2.timeWrite.getTime();
         if (time1 < time2)
            res = -1;
         else
         if (time1 > time2)
            res = 1;
         break;
      }

      case LCmdFilePanelSortOptions::PSW_SIZE:
      {
         if (file1.fileSize < file2.fileSize)
            res = -1;
         else
         if (file1.fileSize > file2.fileSize)
            res = 1;
         break;
      }

      case LCmdFilePanelSortOptions::PSW_UNKNOWN:
      default:
         break;
   }

   if (fpanel.sortOpt.how[fldIdx] == LCmdFilePanelSortOptions::PSH_ASCENDING)
      return res;
   else
      return -res;
}

int LCmdFileItemContainer::compareItems ( const LCmdFileItem& file1, 
                                          const LCmdFileItem& file2 ) const
{
   // The up-dir ("..") should always be the very first item in the list
   if (file1.isUpDir() || file2.isUpDir())
      return -1;

   // Compare first, second, third and fourth sorting order field
   int res;
   for (int i=0; i<4; i++)
      if ((res = compareItems_(file1, file2, i)) != 0)
         return res;
   return 0;
}
