/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DLGQUICKDIRLIST
#define __LCMD_DLGQUICKDIRLIST

#include "glib/gui/event/GDialogMessageHandler.h"
#include "glib/gui/event/GListSelectionListener.h"
#include "glib/util/GArray.h"

/**
 * This dialog is used to let the user select from a short list of
 * frequently used directories for quick and easy access.
 *
 * The list has support for storing up to ten directories for fast access.
 *
 * The dialog also includes logic needed to
 * let the user edit the directory items, and change their position.
 *
 * @author  Leif Erik Larsen
 * @since   2000.06.25
 */
class LCmdDlgQuickDirList : private GDialogMessageHandler,
                            private GListSelectionListener
{
   private:

      /** Ten stored directories for fast access. */
      GArray<GString> storedDirs;

   public:

      LCmdDlgQuickDirList ();
      virtual ~LCmdDlgQuickDirList ();

   private:

      /**
       * This is the message handler method that is called to handle
       * messages from the dialog box window.
       */
      virtual bool handleDialogMessage ( class GDialogMessage& msg );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionChanged ( class GAbstractListBox& abstractListb );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionChangedByCharKey ( class GAbstractListBox& abstractListb );

      /** Part of {@link GListSelectionListener}. */
      virtual void onListBoxSelectionDblClick ( class GAbstractListBox& abstractListb );

      /**
       * Execute the modal dialog.
       * @return Index of which item that was selected, or -1 if canceled.
       */
      int execute ( class GWindow& parentWin );

   public:

      /**
       * Add the current directory of the specified file panel
       * to the list of stored directories.
       *
       * The directory will be stored at the specifed index if it is
       * between zero and nine. Else the directory will be inserted to
       * the first free position.
       *
       * @return  If there are no more free positions then we will return -1.
       *          Else we will return the index of the position where the
       *          directory was actually added.
       */
      int add ( class LCmdFilePanel& panel, int index );

      /**
       * Re-activate the indexed stored directory in the specified panel.
       *
       * @param   panel  Reference to which file panel where the indexed
       *                 directory is to be activated.
       * @param   index  Index of which stored directory to activate.
       *                 Must be between zero and nine.
       * @return  True if the directory was successfully activated,
       *          or else false if something went wrong. Possible problems
       *          can be that the stored directory does no longer exist,
       *          an illegal index was specified, etc.
       */
      bool retrieve ( class LCmdFilePanel& panel, int index );

      /**
       * Show the dialog and let the user select one of the stored
       * directories and then activate it on the specified file panel.
       *
       * @return   True on "OK" and success, or else false on "CANCEL" or
       *           failure.
       */
      bool pickFromDialog ( class LCmdFilePanel& panel );

      /**
       * Load the set of stored directories from the specified section bag,
       * under the specified section name.
       */
      void loadFromProfile ( class GSectionBag& ini, const GString& sectName );

      /**
       * Write the set of stored directories to the specified section bag,
       * under the specified section name.
       */
      void writeToProfile ( class GSectionBag& ini, const GString& sectName );
};

#endif // #ifndef __LCMD_DLGQUICKDIRLIST
