/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_DLGCALCDIRSIZE
#define __LCMD_DLGCALCDIRSIZE

#include "glib/gui/GWorkerThread.h"
#include "glib/util/GDriveInfo.h"
#include "lcmd/LCmdCopyFileItem.h"

/**
 * Calculate total number of bytes in use by a given directory on media.
 *
 * @author  Leif Erik Larsen
 * @since   2001.01.05
 */
class LCmdDlgCalcDirSize : public GWorkerThread
{
   public:

      class Result : public GObject
      {
         friend class LCmdDlgCalcDirSize;

         public:

            /** */
            int countFiles;

            /** */
            int countDirs;

            /** Ideal space actually needed by all the files/dirs. */
            longlong totalSize;

            /** Allocated space, incl. wasteage due to clusters. */
            longlong totalSizeAllocated;

            /** */
            int deepestSubDirLevel;

         private:

            Result ();
            virtual ~Result ();
      };

   private:

      class RootEntry {
      public:
         virtual void doIt (  LCmdDlgCalcDirSize& obj ) = 0;
      };

   private:

      /** 
       * Can be null (if list of items to calculate does not originate 
       * from a file panel.
       */
      class LCmdFilePanel* fpanel;

      class GVfs& vfs;

      GString curDir;

      Result res;

      /** True as long as there is no error. */
      bool statusOK;

      bool finished;

      /** True if we shall wait for the user to click "OK" when the worker thread has finished. */
      bool waitWhenFinished;

      /** Semaphore used by the worker thread to wait for the user to click OK, if {@link #waitWhenFinished} is true. */
      GEventSemaphore waitForOK;

      /** The entry point of which to perform couting. */
      RootEntry& rootEntry;

      /** Current sub-dir level (relative to starting dir). */
      int subDirLevel;

      /** True if the caller need number of files only, and not the total size. */
      bool needFileCountOnly;

   private:

      /**
       * @param  fpanel    A pointer to the source file panel. Can be null
       *                   if the specified filename items don't belong to
       *                   a source file panel.
       * @param  vfs       The VFS on which the specified "startDir" is to 
       *                   be recursed.
       * @param  startDir  The top level directory of which to recurse 
       *                   and count size and number of contained files.
       */
      LCmdDlgCalcDirSize ( class LCmdFilePanel* fpanel, 
                           class GVfs& vfs,
                           const GString& startDir, 
                           RootEntry& rootEntry,
                           bool needFileCountOnly );

      virtual ~LCmdDlgCalcDirSize ();

   private:

      virtual void runTheWorkerThread ( class GWorkerThread& worker );
      virtual void onWorkerThreadInitDialog ( class GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadUpdateMonitor ( class GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadCommand ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& cmd );
      virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& msgID, GObject* userParam );

   private:

      /**
       * @author  Leif Erik Larsen
       * @since   2005.01.31
       * @param   item              Which file item to calculate.
       * @param   idxFilePanelItem  Index of where the item is located in 
       *                            the file panel, or -1 if the item is 
       *                            not located in the file panel at all.
       *                            We will invalidate the item rectangle 
       *                            if this index is >= 0.
       */
      void calcItem ( class LCmdFileItem& item, int idxFilePanelItem = -1 );

      void updateMonitor ( class GDialogPanel& monitor );

   public:

      void calcItems ( GArray<LCmdCopyFileItem>& items );
      void calcSelectedItems ();
      bool calcSubDir ();

      /**
       * Calculate the total number of bytes that are contained by all the
       * source files in the specified array of files.
       *
       * Typically, this function is to be called just before a copy or
       * move progress is started, to know how much data is to be
       * transfered before begin.
       *
       * @param  fpanel   A pointer to the source file panel, or null
       *                  if the list of items does not originate from 
       *                  any file panel.
       * @param  itemsVfs The VFS of where the specified list of 
       *                  filename items is contained. If fpanel != null
       *                  then this VFS should always be the current VFS
       *                  of the specified file panel, or else this VFS 
       *                  can be any VFS.
       * @param  items    Array of filenames. We will only use the source
       *                  paths of the array objects.
       * @param  size     The total number of bytes will be returned in
       *                  this parameter. If this parameter is null we will
       *                  calculate only the file count and not the size.
       *                  This should be true when the caller doesn't depend 
       *                  on file sizes (e.g. if we only need to coundt
       *                  number of files to delete) because it might 
       *                  be used for some major speed optimizations.
       * @param  count    The total counted number of files + directories
       *                  will be returned in this parameter.
       * @return True on success, or else false on any error.
       */
      static bool CalcSizeOfAllItems ( class LCmdFilePanel* fpanel, 
                                       class GVfs& itemsVfs,
                                       GArray<LCmdCopyFileItem>& items, 
                                       longlong* size,
                                       int* count );

      static bool CmdCalcSubDirSize ( class LCmdFilePanel& fpanel );
      static bool CmdCalcRootDirSize ( class LCmdFilePanel& fpanel );
};

#endif // #ifndef __LCMD_DLGCALCDIRSIZE
