/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "lcmd/LCmdDelete.h"
#include "lcmd/LCmdDlgCalcDirSize.h"
#include "lcmd/LCmdDlgConfirmFileDelete.h"
#include "lcmd/LCmdFilePanel.h"
#include "lcmd/LCmdDirCache.h"
#include "lcmd/LCmd.h"

#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GProgressBar.h"
#include "glib/exceptions/GThreadStartException.h"
#include "glib/util/GMath.h"

LCmdDelete::LCmdDelete ( LCmdFilePanel& panel )
           :GWorkerThread("DlgFileDelProgressBar", 0),
            panel(panel),
            vfs(panel.vfs.peek()),
            hdel(0),
            prevSelected(0),
            statusOK(true),
            lastConfirmAnswer(GMessageBox::IDNONE),
            answDelTree(false),
            confirmNextItem(true),
            trashCan(LCmdOptions::GetOptions().fileDel.defaultDeleteToTrashCan),
            countItemsToDeleteTotally(0),
            countDeletedItems(0),
            prevTimeSecRemaining(0),
            ConfirmDelTags("Yncy!")
{
   hdel = vfs.openDeletion();
}

LCmdDelete::~LCmdDelete ()
{
   if (hdel != 0)
      vfs.closeDeletion(hdel);
}

void LCmdDelete::runTheWorkerThread ( GWorkerThread& worker )
{
   bool ok = false;
   int curSel = panel.getCurrentSelectedIndex();
   if (panel.markedFilesCount > 0)
   {
      // Delete all files that are currently marked in panel.
      const int num = panel.items.getCount();
      for (int i=0; i<num && !isStopRequested(); i++)
      {
         LCmdFileItem& file = panel.items[i];
         if (file.isMarked())
         {
            answDelTree = false;
            ok = deleteTopLevelItem(file);
            if (!ok)
               break;
         }
      }
   }
   else
   if (curSel >= 0)
   {
      LCmdFileItem& file = panel.items[curSel];
      ok = deleteTopLevelItem(file);
   }

   // ---
   if (ok)
   {
      GError err = vfs.performDeletion(hdel);
      if (err == GError::Ok)
      {
         // Update the progress dialog with "100% finished".
         sendUserMessageToMonitor("updtMonitor");
      }
      else
      {
         GString msg = err.getErrorMessage();
         GString details = err.getErrorDetails();
         if (details != "")
            msg += "\n\n" + details;
         showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR);
      }
   }
}

bool LCmdDelete::deleteTopLevelItem ( LCmdFileItem& file )
{
   // Can't delete "this" and "up" directories
   if (file.isThisDir() || file.isUpDir())
      return true;

   curFile = '.';
   curFile += GFile::SlashChar;
   curFile += file.getFileName();
   sendUserMessageToMonitor("updtMonitor");

   if (file.isDirectory())
   {
      answDelTree = false;
      return deleteCurDir(file.attr, &file, true);
   }
   else
   {
      return deleteCurFile(file.attr, &file, true);
   }
}

bool LCmdDelete::deleteCurFile ( int attr, LCmdFileItem* file, bool calledByTopLevel )
{
   if (!statusOK)
      return false; // We have probably been breaked/canceled by the user

   while (GFile::IsSlash(curFile.lastChar()))
      curFile.removeLastChar();

   LCmdOptions& opt = LCmdOptions::GetOptions();

   bool isdir = ((attr & GVfs::FAttrDirectory) != 0);
   if (isdir || !opt.fileDel.showDirsProgressOnly)
      sendUserMessageToMonitor("updtMonitor");

   if (!answDelTree && 
       (confirmNextItem || !calledByTopLevel) && 
       lastConfirmAnswer != GMessageBox::IDYESTOALL)
   {
      if (isdir)
      {
         // Don't need to confirm deletion of a directory, because that has
         // already been done by {@link #deleteCurDir}.
      }
      else
      {
         if (opt.confirm.delFileInSubDirs ||
            (opt.confirm.delFile && calledByTopLevel))
         {
            // "Delete file '%s'?"
            GStringl msg("%Txt_Del_Confirm_DelFile", GVArgs(curFile));
            lastConfirmAnswer = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, ConfirmDelTags);
            switch (lastConfirmAnswer)
            {
               case GMessageBox::IDCANCEL:
                    return false;

               case GMessageBox::IDYES:
               case GMessageBox::IDYESTOALL:
                    break;

               case GMessageBox::IDNO:
               default:
                    return true; // Pretend the file was deleted, and continue
            }
         }
      }
   }

   // Need a variable at this scope to remember if we did remove the 
   // ReadOnly-flag from the file, before we attempt to delete the file.
   bool hasRemovedReadOnlyFlag = false;

   if (attr & (GVfs::FAttrReadOnly | GVfs::FAttrHidden | GVfs::FAttrSystem))
   {
      if (lastConfirmAnswer != GMessageBox::IDYESTOALL)
      {
         if (opt.confirm.delHidSysReadFile)
         {
            int countFlags = 0;
            GString msgFlags(64);
            if (attr & GVfs::FAttrReadOnly)
            {
               countFlags++;
               msgFlags += GStringl("%TxtFileAttrReadOnly");
            }

            if (attr & GVfs::FAttrHidden)
            {
               if (countFlags++ > 0)
                  msgFlags += "+";
               msgFlags += GStringl("%TxtFileAttrHidden");
            }

            if (attr & GVfs::FAttrSystem)
            {
               if (countFlags++ > 0)
                  msgFlags += "+";
               msgFlags += GStringl("%TxtFileAttrSystem");
            }

            // "File '%s' is marked as %s. Delete it anyway?"
            GStringl msg("%Txt_Del_Confirm_DelSysFile", GVArgs(curFile).add(msgFlags));
            lastConfirmAnswer = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, ConfirmDelTags);
            switch (lastConfirmAnswer)
            {
               case GMessageBox::IDCANCEL:
                    return false;

               case GMessageBox::IDYES:
               case GMessageBox::IDYESTOALL:
                    break;

               case GMessageBox::IDNO:
               default:
                    return true; // Pretend the file was deleted, and continue
            }
         }
      }

      if (attr & GVfs::FAttrReadOnly)
      {
         // Must remove the read-only attribute before any attempt to
         // delete the file. 
         attr &= ~GVfs::FAttrReadOnly;
         GError rc = vfs.setFileAttributes(curFile, attr);
         hasRemovedReadOnlyFlag = true;
         if (rc != GError::Ok)
         {
            GStringl msg("%Txt_Del_Error_RemoveReadOnlyAttr", GVArgs(curFile).add(rc.getErrorMessage()));
            showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR);
            return false;
         }
      }
   }

   // We should respect the "trashCan" variable only if called at top level.
   // Make a local variable out of this to make things easier to read.
   bool shouldDeleteToTrashCan = (trashCan && file != null);
   if (shouldDeleteToTrashCan)
   {
      // This can only happen when called from {@link #deleteTopLevelItem}.
      // When the "Delete to Trash Can/Recycle Bin option is on we should 
      // let the system remove the file into there. In order for the 
      // file to be "undoable" (from the system point of view) we must 
      // make sure to use the fully qualified path.
      GString fullPath = file->getFullPath();
      GError rc = vfs.removeFile(hdel, fullPath, true);
      for (;;)
      {
         if (rc == GError::Ok)
         {
            countDeletedItems++;
            confirmNextItem = true;
            // Update the dircache, in case the file represents a VFS (e.g. zip-files).
            if (lcmd->options.dirCache.autoStripDirsAfterDirDeletions)
               lcmd->dirCache->removeDirectoryFromCache(fullPath, true);
            return true;
         }
         // "Error on delete file or directory '%s'.\nMessage from the system: %s".
         GStringl msg("%Txt_Del_Error_DelFileOrDir", GVArgs(curFile).add(rc.getErrorMessage()));
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
         switch (answ)
         {
            case GMessageBox::IDSKIP:
               return true;

            case GMessageBox::IDABORT:
               if (hasRemovedReadOnlyFlag)
               {
                  // Restore the ReadOnly-flag befire we return with the error.
                  attr |= GVfs::FAttrReadOnly;
                  vfs.setFileAttributes(curFile, attr);
               }
               return false;

            case GMessageBox::IDRETRY:
            default:
               continue;
         }
      }
   }

   for (bool skip=false; !skip;)
   {
      // Don't attempt to delete to the Recycle Bin/Trash Can at this point,
      // because we never reach here if that option is on anyway.
      GError rc;
      if (isdir)
         rc = vfs.removeDirectory(hdel, curFile, false);
      else
         rc = vfs.removeFile(hdel, curFile, false);
      if (rc == GError::Ok)
      {
         countDeletedItems++;
         confirmNextItem = true;
         if (lcmd->options.dirCache.autoStripDirsAfterDirDeletions)
         {
            GString fullPath = panel.vfs.getFullVirtualPath(true) + curFile;
            GString walkedPath(256);
            const GVfsLocal& localVfs = panel.vfs.root();
            rc = localVfs.getWalkedPath(fullPath, walkedPath);
            if (rc == GError::Ok)
               lcmd->dirCache->removeDirectoryFromCache(walkedPath, true);
         }
         break;
      }
      else
      {
         GStringl msg("%Txt_Del_Error_DelFileOrDir", GVArgs(curFile).add(rc.getErrorMessage()));
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
         switch (answ)
         {
            case GMessageBox::IDSKIP:
                 skip = true;
                 break;

            case GMessageBox::IDABORT:
                 return false;

            case GMessageBox::IDRETRY:
            default:
                 continue;
         }
      }
   }

   // Cut the last filename/dirname from the character buffer that contains
   // current path of file/dir that is about to be deleted.
   curFile.removeLastChar();
   while (curFile != "" && !GFile::IsSlash(curFile.lastChar()))
      curFile.removeLastChar();

   return true;
}

bool LCmdDelete::deleteCurDir ( int attr, LCmdFileItem* file, bool calledByTopLevel )
{
   LCmdOptions& opt = LCmdOptions::GetOptions();

   if (!answDelTree &&
       confirmNextItem && 
       lastConfirmAnswer != GMessageBox::IDYESTOALL &&
       opt.confirm.delEmptyDir)
   {
      // "Are you sure you want to delete the directory '%s'?"
      GStringl msg("%Txt_Del_Confirm_DelDir", GVArgs(curFile));
      lastConfirmAnswer = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, ConfirmDelTags);
      switch (lastConfirmAnswer)
      {
         case GMessageBox::IDCANCEL:
              return false;

         case GMessageBox::IDYES:
              break;

         case GMessageBox::IDYESTOALL:
              answDelTree = true;
              break;

         case GMessageBox::IDNO:
         default:
              return true; // Pretend the file was deleted, and continue
      }
   }

   GFile::Slash(curFile);
   sendUserMessageToMonitor("updtMonitor");

   // We should respect the "trashCan" variable only if called at top level.
   // Make a local variable out of this to make things easier to read.
   bool shouldDeleteToTrashCan = (trashCan && file != null);
   if (shouldDeleteToTrashCan)
   {
      // This can only happen when called from {@link #deleteTopLevelItem}.
      // When the "Delete to Trash Can/Recycle Bin option is on we should 
      // let the system remove the directory into there. In order for the 
      // directory to be "undoable" (from the system point of view) we must 
      // make sure to use the fully qualified path.
      GString fullPath = file->getFullPath(); 

      // Do an extra confirmation if the directory is not empty.
      if (confirmNextItem &&
          lastConfirmAnswer != GMessageBox::IDYESTOALL &&
          opt.confirm.delDirContainingFiles &&
          !vfs.isDirectoryEmpty(fullPath))
      {
         // "Directory '%s' is not empty. Delete it anyway?"
         GStringl msg("%Txt_Del_Confirm_DelNonEmptyDir", GVArgs(fullPath));
         lastConfirmAnswer = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, ConfirmDelTags);
         switch (lastConfirmAnswer)
         {
            case GMessageBox::IDCANCEL:
               return false;

            case GMessageBox::IDYES:
            case GMessageBox::IDYESTOALL:
               break;

            case GMessageBox::IDNO:
            default:
               return true; // Pretend the file was deleted, and continue.
         }
      }

      // Remove the directory.
      for (;;)
      {
         GError rc = vfs.removeDirectory(hdel, fullPath, true, lcmd->mainWin.getHWND());
         if (rc == GError::Ok)
         {
            countDeletedItems++;
            confirmNextItem = true;
            if (lcmd->options.dirCache.autoStripDirsAfterDirDeletions)
               lcmd->dirCache->removeDirectoryFromCache(fullPath, true);
            return true;
         }
         GStringl msg("%Txt_Del_Error_DelFileOrDir", GVArgs(curFile).add(rc.getErrorMessage()));
         GMessageBox::Answer answ = showWorkerMessageBox(msg, GMessageBox::TYPE_ERROR, "Rsa");
         switch (answ)
         {
            case GMessageBox::IDSKIP:
               return true;

            case GMessageBox::IDABORT:
               return false;

            case GMessageBox::IDRETRY:
            default:
               continue;
         }
      }
   }

   GString pattern(curFile.length() + 3);
   pattern = curFile;
   pattern += "*.*";

   GVfs::List list;
   vfs.fillList(list, pattern, true, true, true, true);

   for (int i=0, num=list.size();
        i<num && statusOK && !isStopRequested();
        i++)
   {
      if (i == 0 &&
          !answDelTree &&
          confirmNextItem && calledByTopLevel &&
          lastConfirmAnswer != GMessageBox::IDYESTOALL &&
          opt.confirm.delDirContainingFiles)
      {
         // "Directory '%s' is not empty. Delete it anyway?"
         GStringl msg("%Txt_Del_Confirm_DelNonEmptyDir", GVArgs(curFile));
         lastConfirmAnswer = showWorkerMessageBox(msg, GMessageBox::TYPE_QUESTION, ConfirmDelTags);
         switch (lastConfirmAnswer)
         {
            case GMessageBox::IDCANCEL:
                 return false;

            case GMessageBox::IDYES:
                 break;

            case GMessageBox::IDYESTOALL:
                 answDelTree = true;
                 break;

            case GMessageBox::IDNO:
            default:
                 // Cut the last dirname from the character buffer that contains
                 // current path of file/dir that is about to be deleted.
                 curFile.removeLastChar();
                 while (curFile != "" && !GFile::IsSlash(curFile.lastChar()))
                    curFile.removeLastChar();
                 return true; // Pretend the file was deleted, and continue.
         }
      }

      const GVfs::List::Item& item = list[i];
      const GString& fname = item.getFName();
      int flags = item.getFlags();

      if (item.isDirectory())
      {
         // Recurse into this directory to delete all items in the
         // directory including its subdirectories.
         curFile += fname;
         if (!deleteCurDir(flags, null))
            return false;
      }
      else
      {
         // Delete this file
         curFile += fname;
         if (!deleteCurFile(flags, null))
            return false;
      }
   }

   if (!statusOK)
      return false; // We have probably been breaked/canceled by the user

   // Delete the directory of which we was requested to kill.
   // The directory should now be empty.
   return deleteCurFile(attr, null);
}

void LCmdDelete::onWorkerThreadCommand ( GWorkerThread& worker,
                                         GDialogPanel& monitor,
                                         const GString& cmdID )
{
   if (cmdID == "DLG_CANCEL")
   {
      monitor.setComponentEnabled(cmdID, false);
      requestStop(false);
      statusOK = false;
   }
}

void LCmdDelete::onWorkerThreadUserEvent ( GWorkerThread& worker,
                                           GDialogPanel& monitor,
                                           const GString& msgID,
                                           GObject* /*userParam*/ )
{
   if (msgID == "updtMonitor")
   {
      monitor.setComponentValue("102", curFile);
      int remainingCount = countItemsToDeleteTotally - countDeletedItems;
      monitor.setComponentValue("RemainingCount", remainingCount, false);
      monitor.setComponentValue("Progress", countDeletedItems, false);

      // Estimate remaining time.
      double timeSecSinceStart = double(getTimeMillisSinceStart(true)) / 1000.0;
      if (countDeletedItems > 2 && timeSecSinceStart >= 1.0)
      {
         double timeSecPerFileAvg = timeSecSinceStart / countDeletedItems;
         double timeSecRemaining = (remainingCount * timeSecPerFileAvg) + 1.0;
         double diff = GMath::Abs(prevTimeSecRemaining - timeSecRemaining);
         if (diff < 1.0)
         {
            // Don't update the Eta when new estimated remaining time 
            // differs less than a second from the previous estimated 
            // remaining time. To prevent uggly "flashing" of the time.
         }
         else
         {
            GString etaStr;
            prevTimeSecRemaining = timeSecRemaining;
            if (timeSecRemaining > 86400) // If more than 24 hours.
            {
               if (timeSecSinceStart > 4.0) // Wait some more seconds before showing this.
                  etaStr = GStringl("%Txt_DlgFileCopyProgressBar_MoreThan24HoursLeft");
            }
            else
            if (timeSecRemaining < 1.0)
            {
               monitor.setComponentVisible("Eta", false);
               monitor.setComponentVisible("EtaLeader", false);
            }
            else
            {
               const GLocaleData& ld = GProgram::GetProgram().getLocaleData();
               ulonglong etaMillis = ulonglong(timeSecRemaining * 1000);
               etaStr = ld.getTimeStringFromMillis(etaMillis, false);
            }
            if (etaStr != "")
            {
               monitor.setComponentValue("Eta", etaStr, false);
               monitor.setComponentVisible("Eta", true);
               monitor.setComponentVisible("EtaLeader", true);
            }
         }
      }
   }
}

void LCmdDelete::onWorkerThreadInitDialog ( GWorkerThread& worker, 
                                            GDialogPanel& monitor )
{
   prevTimeSecRemaining = 0;
   monitor.setComponentValue("RemainingCount", countItemsToDeleteTotally, false);
   monitor.setComponentVisible("EtaLeader", false);
   monitor.setComponentVisible("Eta", false);
   GProgressBar& pbar = dynamic_cast<GProgressBar&>(monitor.getComponentByID("Progress"));
   pbar.setMinValue(0);
   pbar.setMaxValue(countItemsToDeleteTotally);
   pbar.setCurrentValue(0);
}

bool LCmdDelete::confirmDelete ( const GString& question, GMessageBox::Answer* answ )
{
   // Make sure that the "answ" parameter is != null.
   GMessageBox::Answer tempAnsw = GMessageBox::IDNONE;
   if (answ == null)
      answ = &tempAnsw;

   LCmdOptions& opt = LCmdOptions::GetOptions();
   LCmdDlgConfirmFileDelete dlg(question, opt.fileDel.defaultDeleteToTrashCan);
   *answ = dlg.execute(lcmd->mainWin);
   if (*answ != GMessageBox::IDYES && *answ != GMessageBox::IDYESTOALL)
      return false;

   lastConfirmAnswer = *answ;
   trashCan = dlg.isDeleteToTrashCan();
   return true;
}

bool LCmdDelete::CmdDelete ( LCmdFilePanel& panel )
{
   GVfs& vf = panel.vfs.peek();
   int curSel = panel.getCurrentSelectedIndex();
   GString strConfirm;
   bool confirmFirstItem = true;
   if (panel.markedFilesCount > 0)
   {
      // You are about to delete %d tagged file(s). Are you sure?
      strConfirm = GStringl("%Txt_Del_Confirm_DelTaggedFiles", GVArgs(panel.markedFilesCount));
      confirmFirstItem = true; // Re-confirm each tagged item.
   }
   else
   if (curSel >= 0)
   {
      LCmdFileItem& fitem = panel.items[curSel];
      if (fitem.isUpDir())
         return false;
      GString fname = fitem.getFileName();
      LCmdOptions& opt = LCmdOptions::GetOptions();
      LCmdOptions::ConfirmOpts& copt = opt.confirm;
      if (fitem.isDirectory())
      {
         GString dir = fitem.getFullPath();
         if (copt.delDirContainingFiles && !vf.isDirectoryEmpty(dir))
         {
            // "Directory '%s' is not empty. Delete it anyway?"
            strConfirm = GStringl("%Txt_Del_Confirm_DelNonEmptyDir", GVArgs(fname));
            confirmFirstItem = false; // Don't confirm the single top-level item again.
         }
         else
         if (copt.delEmptyDir)
         {
            // "Are you sure you want to delete the directory '%s'?"
            strConfirm = GStringl("%Txt_Del_Confirm_DelDir", GVArgs(fname));
            confirmFirstItem = false; // Don't confirm the single top-level item again.
         }
      }
      else
      {
         if (copt.delFile)
         {
            // "Delete file '%s'?"
            strConfirm = GStringl("%Txt_Del_Confirm_DelFile", GVArgs(fname));
            confirmFirstItem = false; // Don't confirm the single top-level item again.
         }
      }
   }
   else
   {
      return false;
   }

   // Ask the user if he is sure to let us perform the deletion.
   LCmdDelete del(panel);
   if (strConfirm != "")
   {
      del.confirmNextItem = confirmFirstItem;
      if (!del.confirmDelete(strConfirm))
         return false; // User answered "No" or "Cancel".
   }

   // ---
   aptr<GArray<LCmdCopyFileItem> > items = panel.makeArrayOfCopyFileItems();
   if (items.get() == null)
      return false;
   int countItemsToDeleteTotally = 0;
   if (del.trashCan) // If "delete to trashcan".
      countItemsToDeleteTotally = (panel.markedFilesCount > 0 ? panel.markedFilesCount : 1);
   else
   if (!LCmdDlgCalcDirSize::CalcSizeOfAllItems(&panel, del.vfs, *items, null, &countItemsToDeleteTotally))
      return false;

   // ---
   if (panel.vfs.isRootOnly())
   {
      // Set current drive and directory.
      GString dir = vf.getCurrentDirectory(false); // Remember which directory was active.
      if (!panel.activatePanelDriveAndDir(true))
         return false;

      // activatePanelDriveAndDir() may have changed the directory buffer
      // even if it returns OK. In that case, threate this as an error and do
      // nothing but fill panel with filenames from that directory set (return
      // without deleteing any files or directories).
      if (!vf.getCurrentDirectory(false).equalsIgnoreCase(dir))
      {
         // Fill in the filenames of the current dir in selected drive.
         panel.reRead();
         return false;
      }
   }

   // ---
   del.countItemsToDeleteTotally = countItemsToDeleteTotally;
   del.prevSelected = panel.getCurrentSelectedIndex();

   GString prevSelected = panel.getCurItemName();

   // Start the background worker thread and show the progress bar dialog.
   try {
      del.workModal(lcmd->mainWin);
   } catch (GThreadStartException& /*e*/) {
      GStringl msg("%Txt_Del_Error_CreateThread");
      lcmd->mainWin.showMessageBox(msg, GMessageBox::TYPE_ERROR);
      return false;
   }

   if (del.countDeletedItems > 0)
   {
      // Reread filenames into panel.
      panel.reRead();

      // If the opposite panel is the same directory then reread that
      // panel as well.
      GString dir1 = panel.getCurrentSysDirectory(false);
      GString dir2 = panel.getOppositePanel().getCurrentSysDirectory(false);
      if (dir1.equalsIgnoreCase(dir2))
         panel.getOppositePanel().reRead();
   }

   // Try to keep the previously selected item select, but if it
   // fails (that is if the previous item has been deleted) then
   // keep the selection as close to the previous position as
   // possible.
   if (panel.selectItem(prevSelected) == -1)
   {
      if (del.prevSelected < panel.items.getCount())
         panel.selectItem(del.prevSelected);
      else
         panel.selectItem(panel.items.getCount() - 1);
   }

   return true;
}
