/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_COPY
#define __LCMD_COPY

#include "glib/vfs/GVfs.h"
#include "glib/gui/GWorkerThread.h"
#include "glib/util/GDriveInfo.h"
#include "lcmd/LCmdCopyFileItem.h"

/**
 * This class is used to perform copy or move of file and directy items.
 *
 * @author  Leif Erik Larsen
 * @since   1999.01.01
 */
class LCmdCopy : public GWorkerThread
{
   public:

      enum ErrType
      {
         ERR_NONE = 0,    // Successfull copy.
         ERR_CANCELED,    // Canceled by user.
         ERR_IOERROR,     // Some io operation failed.
         ERR_CREATETHREAD // Failed on execute the copy worker thread.
      };

      class SpecifyDestDir
      {
         public:

            bool move;
            class LCmdFilePanel& sourcePanel;
            GString destPath;
            GArray<LCmdCopyFileItem>& items;
            ErrType iError;

         public:

            SpecifyDestDir ( bool move,
                             class LCmdFilePanel& srcPanel,
                             const GString& destPath,
                             GArray<LCmdCopyFileItem>& items );
      };

   private:

      // Items that must be set.

      /** The Virtual File System for which to copy/move from. */
      class GVfs& vfsSrc;

      /** The Virtual File System for which to copy/move to. */
      class GVfs& vfsDst;

      /** The handle used for all file- and directory deletion from source operations (in case of moving files). */
      GVfs::DeletionHandle hdelSrc;

      /** True if we are about to MOVE rather than COPY. */
      bool bMove;

      /** True if we should accept "atmic move" only (copy-then-delete not acceptible). */
      bool moveAtomicOnly;

      /** True if we shall remove the ".LONGNAME" EA of the file(s). */
      bool removeEAName;

      /** True if operation is to rename one single file only. */
      bool renameOnly;

      /** Source panel. */
      class LCmdFilePanel* fpanel;

      /**
       * True if the file panel is the source of which file items to
       * copy/move, or else false if the copy/move operation has another
       * source such as when copy/move is due to a drag-n-drop operation.
       */
      bool fpanelIsSrc;

      /** True if we shall not copy the Write Protection Flag (if any). */
      bool dontCopyWriteProtect;

      /** Array of filename path's of which to copy or move. */
      GArray<LCmdCopyFileItem>* items;

   private:

      // Some static helper constants.

      static const char* ILLEGAL_FS_CHARS;

      /**
       * List of EA's which will be removed when a file or directory is
       * being copied or moved to a new name.
       */
      static const char* REMOVE_LONGNAME_EAS[];

   private:

      // Items that will be used internally during the copy/move progress.

      /** Total number of bytes to copy. */
      longlong totalBytesToCopy;

      /** Total number of files/directories to copy. */
      int fileCount;

      /** True if user has requested us not to confirm anything. */
      bool noConfirmations;

      /** True if we shall display the "Statistics" fields in the File Copy/Move progress bar. */
      bool showStatistics;

      /** True if we shall wait for user pressing OK when the copy/move progress has finished. */
      bool waitOnFinish;

      /**
       * True if we are currently waiting for the user to click
       * on the "Finished!" button after the copy or move progress has
       * actually finished.
       *
       * This is only possible if the variable "waitOnFinish" is true,
       * however.
       */
      bool waitingForFinishCommand;

      /**
       * This is the event semaphore that is used to make the background
       * thread wait for the user to click the "Finished!" button before
       * return from {@link #runTheWorkerThread} if waitOnFinish is true.
       */
      GEventSemaphore waitingForFinishedSem;

      /** True if user has confirmed "yes to all" on replace file. */
      bool bReplaceAll;

      /** True if user has confirmed "yes to all" on replace read only, system or hidden file. */
      bool bReplaceAllReadOnly;

      /** True if user has confirmed "yes to all" on merge source with existing destination directory. */
      bool bMergeAllDirs;

      /** True if user has confirmed "yes to all" on delete write protected source file (upon moving files only). */
      bool bRemoveAllReadOnly;

      /** True if user has confirmed "yes to all" on question if we shall discard EAs because destination drive does not support OS/2 Extended Attributes. */
      bool bIgnoreAllEAs;

      /** False if user has answered "yes to all" on question if we shall auto shorten filename that is to long for the destination file system. */
      bool confirmAutoFilenameShorten;

      /** True when entire copy progress has finished. */
      bool bHasFinished;

      /** Will be incremented upon every WM_TIMER message. */
      int timerCounter;

      /** Index of source file item in source panel being copied/moved now. */
      int iCurItemIdx;

      /** Used to remember which item index was recently put into iCurItemIdx. To avoid unneccessary updates. */
      int iPrevItemIdx;

      /** Path of which source file is currently about being copied from. */
      GString curSourcePath;

      /** Path of which destination file is currently about being copied to. */
      GString curDestPath;

      /** Info of the source drive. */
      GDriveInfo srcDriveInfo;

      /** Info of the destination drive. */
      GDriveInfo dstDriveInfo;

      /** The maximum buffer size. */
      int maximumBuffSize;

      /** The number of bytes in {@link #copyBuffer} to actually use for buffering. */
      int activeBufferSize;

      /** The number of bytes allocated in {@link #copyBuffer}. */
      int allocatedBufferSize;

      /** The buffer used to copy bytes from source to destination file. */
      char* copyBuffer;

      /** Number of bytes that has been copied. */
      longlong bytesCopied;

      /** Previous number of bytes that has been copied. */
      longlong prevBytesCopied;

      /** Number of files/directories that has been copied. */
      int countFilesCopied;

      /** Number of files/directories that has been skipped. */
      int countFilesSkipped;

      /** Previous time-since-start, in milliseconds. */
      ulonglong prevTimeSinceStartMillis;

      /** Peak BPS (bytes-per-second). */
      longlong bpsPeak;

      /** Items that are to return a value. */
      ErrType iError;

   private:

      LCmdCopy ( GVfs& vfsSrc, GVfs& vfsDst, const GString& monDlgID );

   public:

      virtual ~LCmdCopy ();

   private:

      void clear ();

      /**
       * Prepare the object for a new copy/move operation.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.30
       */
      bool reInit ( GArray<LCmdCopyFileItem>* items,
                    class LCmdFilePanel& fpanel,
                    bool fpanelIsSrc,
                    bool move,
                    bool renameOnly,
                    bool removeEAName );

   public:

      /**
       * Copy or move the files as are specified in the given array object.
       *
       * The files will be copied or moved to the contained
       * destination path's.
       *
       * @param  vfsSrc       The Virtual File System of where to copy from.
       * @param  vfsDst       The Virtual File System of where to copy to.
       * @param  items        Array of which files to copy or move. Source and
       *                      destination path, as of within the specified 
       *                      source and destination VFS, respectively.
       * @param  move         True if we shall move instead of copy the files.
       * @param  fpanel       A reference to the file panel that is involved
       *                      in the copy/move operation.
       * @param  fpanelIsSrc  True if the file panel is the source of which
       *                      file items to copy/move, or else false if the
       *                      copy/move operation has another source such as
       *                      when copy/move is due to a drag-n-drop operation.
       * @param  renameOnly   True if, and only if, the caller knows that there
       *                      are one and only one file or directory in the
       *                      array and that file is to be renamed only (not
       *                      to be moved by any means).
       * @param  removeEAName If this parameter is true then we will remove the
       *                      ".LONGNAME" EA (if any) from the destination file
       *                      item(s).
       * @param  autoSelect1  The name of the file item in the left file panel
       *                      that we should automatically select when the
       *                      copy/move progress has finsihed and at least one
       *                      file/directory has actually been copied/moved. If
       *                      this parameter is an empty string then we will
       *                      try to keep the current selcted item selected.
       * @param  autoSelect2  Same as <i>autoSelect1</i> except this one is
       *                      for the right file pane.
       * @param  countFilesSkipped Returnes the number of skipped files and/or
       *                      directories, due to user answering "skip" to one 
       *                      or more copy/move progress message boxes.
       *                      If null is specified we will not touch 
       *                      this argument.
       * @param  refillPanel  True if we shall automatically refill the 
       *                      current file panel after the copy/move has
       *                      finished, or else false if this should not 
       *                      be done.
       * @return True on success, or else false on any error.
       */
      static bool CopyOrMoveFiles ( class GVfs& vfsSrc,
                                    class GVfs& vfsDst,
                                    GArray<LCmdCopyFileItem>& items,
                                    bool move,
                                    LCmdFilePanel& fpanel,
                                    bool fpanelIsSrc,
                                    bool renameOnly,
                                    bool removeEAName,
                                    const GString& autoSelect1 = GString::Empty,
                                    const GString& autoSelect2 = GString::Empty,
                                    int* countFilesSkipped = null,
                                    bool refillPanel = true );

   private:

      GError copyFile_ ( GVfs::FileHandle srcFile,
                         GVfs::FileHandle dstFile,
                         const GString& originalFName,
                         const GString& shortenedFName,
                         const GString& dstFilePath,
                         bool yesCopyEAsIfAny );

      /**
       * @param  srcSize     Number of bytes in the source file.
       * @param  sourcePath  Path of which file to copy from.
       * @param  destPath    Path of which file to copy to. Upon return this
       *                     parameter will contain the actual destination
       *                     filename.
       * @param  skipped     Returns true if file wasn't actually copied,
       *                     even if we return {@link #ERR_NONE}.
       * @param  isLevel1    True if and only if this is the first
       *                     recursive level.
       */
      ErrType copyFile ( longlong srcSize,
                         const class GFile& sourcePath,
                         class GFile& destPath,
                         bool* skipped,
                         bool isLevel1 );

      // sourcePath  Entry: Directory of where to copy from.
      // destPath    Entry: Directory of where to copy to.
      // skipped     Return: True if file wasn't actually copied, even if we return {@link #ERR_NONE}.
      // isLevel1    Entry: True if, and only if, this is the first recursive level.
      ErrType copyDirectory ( const class GFile& sourcePath,
                              class GFile& destPath,
                              bool* skipped,
                              bool isLevel1 );

      ErrType doCopyOrMoveItem ( bool isDir,
                                 longlong srcFileSize,
                                 const GString& srcPath,
                                 const GString& dstPath,
                                 bool* skipped,
                                 bool renameOnly );

      void updateMonitor ( class GDialogPanel& monitor, bool forceUpdateAll );

      /**
       * This is the main entry point of the secondary thread.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.09
       */
      virtual void runTheWorkerThread ( class GWorkerThread& worker );

      /**
       * This method is automatically called by our base class whenever
       * we should update the status/progress components of the
       * copy/move progress dialog.
       *
       * @author  Leif Erik Larsen
       * @since   2000.10.09
       */
      virtual void onWorkerThreadUserEvent ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& msgID, GObject* userParam );
      virtual void onWorkerThreadInitDialog ( class GWorkerThread& worker, class GDialogPanel& monitor );
      virtual void onWorkerThreadRequestStop ( class GWorkerThread& worker, class GDialogPanel& monitor, bool success );
      virtual void onWorkerThreadCtrlChanged ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& compID, const GString& newValue );
      virtual void onWorkerThreadCommand ( class GWorkerThread& worker, class GDialogPanel& monitor, const GString& cmd );
};

#endif // #ifndef __LCMD_COPY

