/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_CMDLINEENTRY
#define __LCMD_CMDLINEENTRY

#include "glib/gui/GMultiLineEditor.h"
#include "glib/gui/event/GCommandMap.h"
#include "glib/gui/event/GMouseListener.h"
#include "glib/primitives/GAtomicCounter.h"
#include "glib/sys/GSystem.h"
#include "lcmd/LCmdCmdLineEntryColorOptions.h"
#include "lcmd/LCmdAliases.h"
#include "lcmd/LCmdInternalCmd.h"
#include "lcmd/LCmdCmdHist.h"
#include "lcmd/LCmdDirHist.h"
#include "lcmd/LCmdSepSessionApps.h"

/**
 * List of some commands that are known not to be compatible with the
 * Console Monitor of Larsen Commander.
 */
struct KnownSSA
{
   bool viaMain;
   bool viaPipe;
   const char* name;
};

extern KnownSSA knownSSA[];

/**
 * The entryfield window of where the user can enter and edit a command for
 * which to be executed when he press Enter.
 *
 * We use <i>GMultilineEditor</i> because the standard OS/2 ENTRYFIELD
 * Control has a number of stupid limitations that the MLE control hasn't.
 * Just a few examples are support for the MLM_INSERT message and
 * default handling of the "Ctrl+Left" and the "Ctrl+Right" keyboard
 * scancodes.
 *
 * This class also keeps track of the running child processes, and has the
 * responsibility to change the color of the text entry window with respect
 * to the current child program states.
 *
 * @author  Leif Erik Larsen
 * @since   1998.10.24
 */
class LCmdCmdLineEntry : public GMultiLineEditor
{
   friend class LCmdProcessLauncher;

   private:

      /**
       * Inner class used to display the popup menu when user 
       * clicks the right mouse button on our contained peer.
       *
       * @author  Leif Erik Larsen
       * @since   2004.09.07
       */
      class MyMouseListener : public GMouseListener, public GObject
      {
         friend class LCmdCmdLineEntry;
         class LCmdCmdLineEntry& owner;
         MyMouseListener ( class LCmdCmdLineEntry& owner );
         virtual ~MyMouseListener ();
         virtual bool mousePressed ( const GMouseEvent& ev );
      };

      /** Listens for mouse messages from our contained peer. */
      MyMouseListener mouseListener;

      /** A reference to our parent window. */
      class LCmdCmdLine& cmdLine;

      /** True if ECHO is on. */
      bool echoIsOn;

      /** Current number of running child processes. This variable is maintained by {@link LCmdProcessLauncher#run}. */
      GAtomicCounter<int> runningCounter;

      /** Max number of characters in a single command line command. */
      enum { MAXCOMMANDSIZE = 1024 };

   public:

      /** Colors of the Command Line Window. */
      LCmdCmdLineEntryColorOptions colorOpt;

      /** 
       * The thread of where to call {GThread#guiUserMessageHandlerHasFinished} 
       * when data has been feeded to child process that is waiting for input 
       * from its STDIN. 
       */
      class LCmdProcessLauncher* childWaitingForInput;

      /** If the process selection dialog is active, this is its HWND. */
      class GDialogPanel* childProcessSelectorDlg;

      /** All the currently defined aliases. */
      LCmdAliases aliases;

      /** The bag of internal commands to be recognized by executeCommand(). */
      GKeyBag<InternalCommand> internalCommands;

      /** Bag of all programs that will always be launched in a new session. */
      GKeyBag<LCmdSepSessionApps> sepSessionApps;

      /**
       * The array of child processes that we have launched.
       *
       * Not all of these are neccessarily still running. Each time a new
       * program is to be launched we will loop though this array to see if
       * any of them has finished their task. If so, we will remove the object
       * from the array and destroy it.
       */
      GArray<LCmdProcessLauncher> runningChildren;

      /** The command history list. */
      CommandHistory cmdHist;

      /** The most recent visited directories. */
      DirectoryHistory dirHist;

      /** The directory stack to use with the PUSHD and POPD commands. */
      GArray<GString> dirStack;

   public:

      LCmdCmdLineEntry ( class LCmdCmdLine& parentWin, 
                         const GString& constraints );

      virtual ~LCmdCmdLineEntry ();

   private:

      /** Disable the copy-contructor. */
      LCmdCmdLineEntry ( const LCmdCmdLineEntry& );

      /** Disable the assignment operator. */
      void operator= ( const LCmdCmdLineEntry& );

   private:

      /**
       * Find the next occurence of any of the characters specified in
       * <i>chr</i> in the source string <i>str</i>.
       *
       * If there are no such characters in the string then we will
       * return NULL.
       *
       * We will ignore any occurence of the characters if it is contained
       * within a pair of enclosing quotes, however.
       */
      static const char* FindNext ( const char* str, const char* chr );

      /**
       * We will automatically call this method before a new task is actually
       * launched by <i>executeCommand</i>. It will loop though the array of
       * child processes to see if any of them has finished. If so, we will
       * remove the finished child process from the array and delete the
       * object of it.
       */
      void garbageCollectChildren ();

      /**
       * Get the index of the newest child program that is still running.
       * If no child is running then we will return -1.
       */
      int getIndexOfNewestChild ();

      /**
       * Get a reference to the <i>LCmdSepSessionApps</i> object of the
       * specified program, or null if the program is not contained in
       * the bag of programs that are incompatible with the console
       * monitor of Larsen Commander.
       */
      LCmdSepSessionApps* getNoneConsoleProgram ( const GString& progPath );

   private:

      /**
       * Execute the indexed command in the command history.
       *
       * This method is called by {@link #executeCommand} if the user
       * specified command is found to be "H" or "HIST".
       */
      bool executeHistoricCommand ( int index, const GString& workDir, int flags );

   public:

      /**
       * Perform an "automatic filename completion" ala 4OS2.
       */
      void autoCompleteFileName ( bool foreward );

      /**
       * Break/kill the child process of choice (if any).
       *
       * This method will be called by the GUI thread when the user has
       * pressed CTRL+C to break one of the (potentially several) child
       * processes.
       *
       * We will prompt the user to select which child process to break.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.25
       * @param   defaultBt The break type of which to be used by default.
       *                    That is, if the user doesn't select any other 
       *                    break mehod. This default break method is the 
       *                    one that will be initially selected in the 
       *                    dialog where user can select which process 
       *                    to break, and how to break it.
       */
      void breakChildProg ( GSystem::BreakType defaultBt );

      /** Flags to be used with {@link #executeCommand}. */
      enum { ECF_FORCE_NEW_SESSION = 0x0001 }; // forceNewSession
      enum { ECF_IS_INTERNAL = 0x00002 }; // isInternal
      enum { ECF_RUN_AS_SHELL_OBJECT = 0x0004 }; // runAsObject
      enum { ECF_CLOSEON_EXIT = 0x0010 }; // closeOnExit
      enum { ECF_FORCE_RUN_VIA_SHELL = 0x0020 }; // forceRunViaShell
      enum { ECF_DO_ECHO = 0x0040 }; // doEcho
      enum { ECF_DONT_ADD_TO_CMD_HIST = 0x0080 }; // dontAddToCmdHist

   private:

      /**
       * Parse and execute the specified command line.
       *
       * Execute the specified command with the specified set of parameters
       * after they have been parsed and put into an array of separated
       * arguments.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.04
       * @return  True on success, or esle false on any error.
       */
      bool executeCommand ( const GString& originalCmd,
                            const GString& workDir,
                            const GString& prgName,
                            const GString& paramStr,
                            int flags );

   public:

      /**
       * Parse and execute the specified command line.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.04
       * @return  True on success, or esle false on any error.
       */
      bool executeCommand ( const GString& cmd,
                            const GString& workDir,
                            int flags );

      /**
       * Write the specified text to the STDIN stream of the specified 
       * child process.
       *
       * This method must be used by the GUI-thread only.
       *
       * @author  Leif Erik Larsen
       * @since   2006.02.01
       * @param   child   The child process of where to write the text,
       *                  or null to let the user select the child process 
       *                  from an interactive list.
       * @param   txt     The text of which to feed into the STDIN stream 
       *                  of the specified child process.
       * @return  True on success, or false if the text could not be 
       *          written due to some error. Possible errors are e.g.
       *          that the user did cancel the process select dialog,
       *          or if there are no current running child process at all.
       */
      bool feedChildProcessStdIn ( LCmdProcessLauncher* child, const GString& txt );

      /**
       * Get a reference to the command line entry of Larsen Commander.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.16
       */
      static LCmdCmdLineEntry& GetCmdLineEntry ();

      /**
       * Override the isRunning() method of our super class.
       * We need to test on our own internal flag in addition, in case we are
       * waiting for the handling of the UM_PROCESSLAUNCHERTHREADHASFINISHED to
       * finish.
       */
      int getRunningChildrenCount () const;

      /**
       * Test if the specified program w/parameters should be forced to
       * run in a new session or not.
       */
      bool isNoneConsoleCommand ( const GString& progPath, 
                                  const GString& params, 
                                  bool testPipedCommandsOnly = false );

      /**
       * Overrides GMultiLineEditor::paste() in order to prevent multiline
       * texts from being pasted.
       */
      void paste ();

      /**
       * Load and activate the profile variables for the commandline entry
       * window, from the specified section name.
       * @see #writeProfile
       */
      void queryProfile ( const GString& sectName );

      /**
       * Replace the current selected text in the Text Entry with the
       * specified text, but only if the command line is currently
       * configured to be visible.
       *
       * If there is no selected text at the time being then we
       * will insert the specified text at the current cursor position
       * without deleteing any existing text.
       */
      void replaceSelection ( const GString& text );

      /**
       * Show a dialog box of where the user can select one of the currently
       * running child processes. If the user select CANCEL then we will
       * return NULL.
       *
       * @author  Leif Erik Larsen
       * @since   2004.06.25
       * @param   bt   <b>On input:</b> The default break type of which to 
       *               be used if user doesn't specify anything else.<br>
       *               <b>On output:</b> The break type of which to be 
       *               used by the caller to break the child process.<br>
       *               If this argument is null then we will not touch it,
       *               and the user will not be able to choose any 
       *               break-type.
       * @param   tree <b>On input:</b> Default state of the "Process Tree"
       *               toggle button.<br>
       *               <b>On output:</b> User selected state of the 
       *               "Process Tree" toggle button.<br>
       *               If this argument is null then we will not touch it,
       *               and the user will not be able to toggle "tree".
       * @return  The user selected process, or null if user canceled 
       *          the dialog.
       */
      LCmdProcessLauncher* selectChildProcess ( GSystem::BreakType* bt, bool* tree );

      /**
       * Show the long help for the specified command.
       *
       * This method is typically called for commands when the "/?" or "-?"
       * argument has been given by user.
       */
      void showHelpForCommand ( const GString& cmdName );

      /**
       * Updates the command line to its correct foreground and background 
       * colours with respect to the current state of child programs, 
       * if any.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.29
       */
      void updateCommandLineColours ();

      /**
       * Write the profile variables for the commandline entry window,
       * under the specified section name.
       * @see #queryProfile
       */
      void writeProfile ( const GString& sectName, bool force );

   protected:

      virtual bool onUserMessage ( GUserMessage& msg );
      virtual bool onBackgroundColorChanged ( const GColor& color );
      virtual bool onForegroundColorChanged ( const GColor& color );
      virtual bool onInitMenu ();

      /**
       * If the command line entry is empty then the VK_DELETE key
       * will cause the "cmdDeleteFile" command method to be called
       * on the main window, but only if the option
       * {@link LCmdOptions::FileDelOpts#useDelKey} is on.
       *
       * There are a few keyboard commands that we should give to the default
       * message handler of <i>GMultiLineEditor</i> regardless if it is
       * defined in the accelerator table of this window or not.
       *
       * These keyboard codes should be filtered when the file panels are off.
       *
       * <pre>
       * VK_HOME<br>
       * VK_END<br>
       * VK_LEFT<br>
       * VK_RIGHT<br>
       * </pre>
       *
       * These keyboard codes should be filtered when there are some text in
       * the editor window:
       *
       * <pre>
       * SHIFT+VK_HOME<br>
       * SHIFT+VK_END<br>
       * SHIFT+VK_LEFT<br>
       * SHIFT+VK_RIGHT<br>
       * </pre>
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.17
       */
      virtual bool onKeyDown ( const class GKeyMessage& key );

   public:

      void cmdClear ( class GAbstractCommand *cmd );
      void cmdCopy ( class GAbstractCommand *cmd );
      void cmdCopyAllCommandLineText ( class GAbstractCommand *cmd = null );
      void cmdCut ( class GAbstractCommand *cmd );
      void cmdMoveCaretRight ( class GAbstractCommand *cmd = null );
      void cmdMoveCaretLeft ( class GAbstractCommand *cmd = null );
      void cmdMoveCaretHome ( class GAbstractCommand *cmd = null );
      void cmdMoveCaretEnd ( class GAbstractCommand *cmd = null );
      void cmdMoveCaretWordRight ( class GAbstractCommand *cmd = null );
      void cmdMoveCaretWordLeft ( class GAbstractCommand *cmd = null );
      void cmdPaste ( class GAbstractCommand *cmd );
      void cmdScrollConMonHome ( class GAbstractCommand *cmd );
      void cmdScrollConMonTop ( class GAbstractCommand *cmd );
      void cmdScrollConMonEnd ( class GAbstractCommand *cmd );
      void cmdScrollConMonBottom ( class GAbstractCommand *cmd );
      void cmdScrollConMonUp ( class GAbstractCommand *cmd );
      void cmdScrollConMonDown ( class GAbstractCommand *cmd );
      void cmdScrollConMonLeft ( class GAbstractCommand *cmd );
      void cmdScrollConMonRight ( class GAbstractCommand *cmd );
      void cmdScrollConMonPageUp ( class GAbstractCommand *cmd );
      void cmdScrollConMonPageDown ( class GAbstractCommand *cmd );
      void cmdSelectAll ( class GAbstractCommand *cmd );

      /**
       * Toggle on/off the insert-mode of the command line text entry field.
       */
      void cmdToggleInsertMode ( class GAbstractCommand *cmd );

   DECLARE_COMMAND_TABLE(LCmdCmdLineEntry);
};

#endif // #ifndef __LCMD_CMDLINEENTRY
