/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __LCMD_CMDLINE
#define __LCMD_CMDLINE

#include "lcmd/LCmdCmdLineEntry.h"
#include "glib/gui/GAbstractToolbarWindow.h"

/**
 * This is the Command Line class.
 *
 * It contains two child windows: The prompt window of (which is used to paint
 * the current directory) and the command line entry window which is a text
 * input field of where the user can enter command line commands.
 *
 * The reason why we extends <i>GAbstractToolbarWindow</i> is to make it
 * easier for the prompt area window to actually be a toolbar button. When
 * the user clicks on that button then the directory history window will
 * be displayed.
 *
 * @author  Leif Erik Larsen
 * @since   1998.10.24
 */
class LCmdCmdLine : public GAbstractToolbarWindow
{
   friend class LCmdProcessLauncher;

   private:

      /** 
       * False until constructor has finished creating our child windows, 
       * to prevent null-pointer exception e.g. in {@link #layout}.
       */
      bool initialized;

   public:

      /** A reference to our owner container window. */
      class LCmdCmdContainer& ownerCont;

      /** A reference to the console monitor text viewer window. */
      class LCmdConsoleMonitor& conmon;

      /** The prompt that contains the current directory text, and also forms a button that the user can press in order to get easy access to the directory history. */
      class LCmdCmdLinePrompt& prompt;

      /** The MLE entry field of the command line. */
      LCmdCmdLineEntry entry;

      /** The command history push button. */
      class LCmdCmdLineButton& butt;

   public:

      LCmdCmdLine ( class LCmdCmdContainer& ownerCont, 
                    const GString& constraints, 
                    class LCmdConsoleMonitor& conmon );

     virtual ~LCmdCmdLine ();

   private:

      /** Disable the copy constructor. */
      LCmdCmdLine ( const LCmdCmdLine& src ) 
         : GAbstractToolbarWindow(GString::Empty, GString::Empty, *this), ownerCont(src.ownerCont), conmon(src.conmon), prompt(src.prompt), entry(*this, GString::Empty), butt(src.butt) {}

      /** Disable the assignment operator. */
      LCmdCmdLine& operator= ( const LCmdCmdLine& ) { return *this; }

   public:

      /**
       * This function will be called whenever the current Drive and/or
       * Directory has been changed.
       *
       * We will perform the code neccessary to update the Command Line
       * Window (if it is currently in use at all) and request the
       * window to be repainted.
       */
      void currentPathHasChanged ();

      /**
       * Get the preferred height of the window, in pixels.
       */
      int getPreferredHeight () const;

      /**
       * We override the layout() method of our super class just to
       * displace the entry field so that its bottom line as one pixel
       * "above" the bottom line of the command line buttons. This is just
       * to make the thin line between the bottons and the keybar as nice as
       * "possible".
       *
       * However, this displacement is only done if the keybar is
       * currently configured (by the user) to be visible.
       */
      void layout ();

      /**
       * Replace the current text (if any) in the Text Entry with the
       * specified text.
       */
      void setText ( const GString& text ) { entry.setText(text); }
};

#endif // #ifndef __LCMD_CMDLINE

