/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GVfsStack.h"
#include "glib/vfs/GVfsLocal.h"
#include "glib/util/GLog.h"
#include "glib/io/GIOException.h"

GVfsStack::GVfsStack ()
          :GStack<GVfs>(4)
{
   GVfsLocal* root = new GVfsLocal();
   push(root);
}

GVfsStack::~GVfsStack ()
{
}

GString GVfsStack::getFullVirtualPath ( bool slash ) const
{
   const GVfsLocal& localVfs = root();
   GString buff(128);
   for (int i=0, num=getCount(); i<num; i++)
   {
      const GVfs& fs = get(i);
      if (i > 0)
      {
         GFile::Slash(buff);
         buff += fs.getLogicalSelfName();
      }
      GString cur = fs.getCurrentDirectory(slash);
      if (cur != "")
      {
         GFile::Slash(buff);
         buff += cur;
      }
   }

   if (slash)
      GFile::Slash(buff);

   // Make sure all slashes respects the root file system.
   localVfs.translateSlashes(buff);
   return buff;
}

GVfsLocal& GVfsStack::backToRoot () 
{ 
   while (!isRootOnly()) 
      pop(); 
   return root(); 
}

bool GVfsStack::isRootOnly () const 
{ 
   return getCount() == 1; 
}

void GVfsStack::pop () 
{ 
   bool destroy = !isRootOnly();
   GStack<GVfs>::pop(destroy); 
}

void GVfsStack::push ( GVfs* v, bool autoDelete ) 
{ 
   GStack<GVfs>::push(v, autoDelete); 
}

const GVfsLocal& GVfsStack::root () const 
{ 
   return dynamic_cast<GVfsLocal&>(get(0)); 
}

GVfsLocal& GVfsStack::root () 
{ 
   return dynamic_cast<GVfsLocal&>(get(0)); 
}

const GVfs& GVfsStack::peek () const 
{ 
   return GStack<GVfs>::peek(); 
}

GVfs& GVfsStack::peek () 
{ 
   return GStack<GVfs>::peek(); 
}

GError GVfsStack::walkPath ( const GString& pathToWalk_, GString* autoSelect )
{
   GString pathToWalk = pathToWalk_;
   if (pathToWalk == GString::Empty)
      return GError::Ok;

   // If the pathToWalk contains a drive specification, activate the drive.
   GFile file(pathToWalk);
   if (file.containsDrive())
   {
      // Drive is specified, so activate the specified drive!
      GString driveStr = file.getDrive();
      int drive = GCharacter::ToUpperCase(driveStr[0]) - 'A' + 1;
      GString dir;
      try {
         dir = GFile::GetCurrentDir(drive);
      } catch (APIRET& rc) {
         return rc;
      }

      GVfsLocal& root = backToRoot();
      GError err = root.setCurrentDirectory(dir);
      if (err != GError::Ok)
      {
         // Could not activate the "current directory" of that drive.
         // If the "current directory" is not the root then retry by 
         // activating the root of that drive.
         if (dir.length() > 3 && dir[1] == ':' && GFile::IsSlash(dir[2]))
         {
            dir.cutTailFrom(3);
            err = root.setCurrentDirectory(dir);
         }
         if (err != GError::Ok)
         {
            // The root could not be activated either. 
            // This probably means that the drive is invalid.
            return err;
         }
      }

      // Remove the drive part of the pathToWalk.
      pathToWalk = file.getDirAndFileName();
   }

   // Separate each slash-separated directory element of the 
   // specified walking-path into a string-vector for easier access.
   bool walkFromRoot = false;
   GArray<GString> dirList(16);
   GString dirElm(128);
   for (int i=0, len=pathToWalk.length(); i<len; i++)
   {
      while (GFile::IsSlash(pathToWalk[i]))
      {
         if (i == 0) // If the very first character is a slash.
            walkFromRoot = true;
         i++;
      }
      dirElm.clear();
      while (i<len && !GFile::IsSlash(pathToWalk[i]))
         dirElm += pathToWalk[i++];
      if (dirElm.length() > 0)
         dirList.add(new GString(dirElm));
   }

   // Jumpt to the physical root directory if the walk path 
   // did start with a slash.
   if (walkFromRoot)
   {
      GVfsLocal& root = backToRoot();
      root.setCurrentDirectory(GFile::SlashStr);
   }

   // Remember the name of the last walked-down directory element.
   // Typically used by the caller to e.g. auto-select an item 
   // in something like a directory list after activating a parent directory.
   GString autoSelectLocal;
   if (autoSelect == null)
      autoSelect = &autoSelectLocal;

   // Loop through each slash-separated directory element of the 
   // specified walking-path, and walk respectively.
   for (int i=0, num=dirList.getCount(); i<num; i++)
   {
      dirElm = dirList.get(i);
      if (GFile::IsThisDir(dirElm)) // If dirElm == "."
         continue;
      GVfs& fs = peek();
      if (GFile::IsUpDir(dirElm)) // If dirElm == ".."
      {
         // Remember which filename to auto-select before return.
         *autoSelect = getFullVirtualPath(false);
         autoSelect->reverse(); // Reverse direction of original directory.
         autoSelect->replaceAll('\\', '/'); // Convert to foreward slashes only.
         int index = autoSelect->indexOf('/');
         if (index >= 0)
            autoSelect->cutTailFrom(index); // Cut the other directories.
         autoSelect->reverse(); // Return to original direction of string.

         // ---
         if (fs.isCurrentDirectoryTheRoot())
         {
            if (isRootOnly())
               return false;
            pop(); // Step out of the virtual file system.
            continue;
         }
      }
      GString path = fs.getCurrentDirectory(false);
      fs.walkPath(path, dirElm);
      if (fs.existDirectory(path))
      {
         if (fs.setCurrentDirectory(path) != GError::Ok)
            return false;
      }
      else
      if (isSupportedZipOrArchiveFile(path))
      {
         // User want to walk into a nested archive file.
         GError err = walkIntoArchiveFile(path);
         if (err != GError::Ok)
            return err;
      }
      else
      {
         // TODO: Other types of VFSs to be supported soon (FTP, etc.)
         return GError::FileNotFound;
      }
   }

   return GError::Ok;
}

bool GVfsStack::isSupportedZipOrArchiveFile ( const GString& path ) const
{
   // It is up to the overriding subclass to implement support for 
   // various archive file formats.
   return false;
}

GError GVfsStack::walkIntoArchiveFile ( const GString& fname )
{
   // It is up to the overriding subclass to implement support for 
   // various archive file formats.
   return GError::NotSupported;
}
