/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/vfs/GFile.h"
#include "glib/GProgram.h"

GHashtable<GInteger, GVfsLocal::OpenFile> GVfsLocal::OpenFiles;

GVfsLocal::GVfsLocal ()
          :GVfs(null)
{
   curDir = GFile::GetCurrentDir();
}

GVfsLocal::~GVfsLocal ()
{
}

GVfsLocal::OpenFile::OpenFile ( const GString& path, GVfs::FileHandle sysFileHandle )
                    :path(path),
                     sysFileHandle(sysFileHandle)
{
}

GVfsLocal::OpenFile::~OpenFile ()
{
}

bool GVfsLocal::findFirstMightNeedLongTime () const 
{ 
   return false; 
}

int GVfsLocal::findFirst ( GFileItem& fitem, 
                           const GString& path_,
                           bool* /*cancelSem*/, // Not used on this kind of file system.
                           int* /*preLoadCounter*/ ) const // Ditto
{
   GString path;
   if (path_ == "")
   {
      path = curDir;
      GFile::Slash(path);
      path += "*.*";
   }
   else
   {
      path = path_;
      if (GFile::IsSlashed(path))
         path += "*.*";
   }
   ULONG fcount = 1; // Don't read more than one filename at once.
   HDIR hdir = HDIR(HDIR_CREATE);
   if (GSystem::GetPlatformID() >= GSystem::Platform_OS2_4_5)
   {
      FILEFINDBUF3L ffb; // File Info including size of EA's.
      APIRET rc = ::DosFindFirst(path, &hdir, GVfs::FAttrAll, &ffb, sizeof(ffb), &fcount, FIL_STANDARDL);
      if (rc != NO_ERROR)
         return 0;
      fitem = ffb;
   }
   else
   {
      FILEFINDBUF3 ffb; // File Info including size of EA's.
      APIRET rc = ::DosFindFirst(path, &hdir, GVfs::FAttrAll, &ffb, sizeof(ffb), &fcount, FIL_STANDARD);
      if (rc != NO_ERROR)
         return 0;
      fitem = ffb;
   }
   return hdir;
}

bool GVfsLocal::findNext ( int hdir, GFileItem& fitem ) const
{
   ULONG fcount = 1; // Don't read more than one filename at once.
   if (GSystem::GetPlatformID() >= GSystem::Platform_OS2_4_5)
   {
      FILEFINDBUF3L ffb; // File Info including size of EA's.
      APIRET rc = ::DosFindNext(hdir, &ffb, sizeof(ffb), &fcount);
      if (rc != NO_ERROR)
         return false;
      fitem = ffb;
   }
   else
   {
      FILEFINDBUF3 ffb; // File Info including size of EA's.
      APIRET rc = ::DosFindNext(hdir, &ffb, sizeof(ffb), &fcount);
      if (rc != NO_ERROR)
         return false;
      fitem = ffb;
   }
   return true;
}

void GVfsLocal::findClose ( int hdir ) const
{
   ::DosFindClose(hdir);
}

GString GVfsLocal::getRootDir () const
{
   GString root = GFile::GetDrive(curDir);
   GFile::Slash(root);
   return root;
}

bool GVfsLocal::isCurrentDirectoryTheRoot () const
{
   GString root = getRootDir();
   return root.equalsIgnoreCase(curDir);
}

GString GVfsLocal::getCurrentDirectory ( bool slash ) const 
{ 
   GString ret(curDir.length() + 1);
   ret = curDir;
   if (slash)
      GFile::Slash(ret);
   return ret; 
}

GError GVfsLocal::setCurrentDirectory ( const GString& dir )
{
   // Walk.
   GString prevDir = curDir;
   APIRET rc = GFile::SetCurrentDir(dir);
   if (rc != NO_ERROR)
   {
      curDir = prevDir;
      return rc;
   }

   // Make sure curDir contains the actual current directory after
   // the walk operation. This also makes sure that all parts of the
   // directory are correctly represented with respect to character cases.
   curDir = GFile::GetCurrentDir();

   // Always update "driveInfo.drive" synchronously, because the rest of the
   // drive information gets updated no more than once per second. But e.g. 
   // {@link #getCurrentDriveLetter} must always return the correct drive.
   char driveChr = curDir.firstChar();
   driveInfo.drive = GCharacter::ToUpperCase(driveChr) - 'A' + 1;

   // ---
   return GError::Ok;
}

GString GVfsLocal::getLogicalSelfName () const 
{ 
   return GString::Empty; 
}

GString GVfsLocal::getPhysicalSelfName () const 
{ 
   return GString::Empty; 
}

GString& GVfsLocal::slash ( GString& dir ) const
{
   return GFile::Slash(dir);
}

bool GVfsLocal::isSlash ( char chr ) const
{
   return GFile::IsSlash(chr);
}

const GString& GVfsLocal::getSlashStr () const
{
   return GVfsLocal::SlashStr;
}

bool GVfsLocal::containsAnySlash ( const GString& path ) const
{
   return path.indexOf('/') >= 0 || path.indexOf('\\') >= 0;
}

GVfs::File* GVfsLocal::preparePhysicalFile ( const GString& fileName, 
                                             WorkStatus& /*stat*/,
                                             const GString& /*prefix*/ )
{
   GString path;
   if (containsAnySlash(fileName))
      path = fileName; // File name is already somewhat qualified.
   else
      path = getCurrentDirectory(true) + fileName; // Make fully qualified filename.
   translateSlashes(path);
   return new File(fileName, path, false, false);
}

void GVfsLocal::updateInternalDriveInfo ( bool force ) const
{
   ulonglong timeNow = GSystem::CurrentTimeMillis();
   ulonglong timeDiffMillis = timeNow - lastDriveInfoUpdateTimeMillis;
   if (!force && timeDiffMillis < 1000) // Don't update too often.
      return;
   int drive;
   if (GFile::ContainsDrive(curDir)) // Should always be true, but in case.
      drive = GCharacter::ToUpperCase(curDir[0]) - 'A' + 1;
   else
      drive = 0; // Current drive.
   driveInfo.update(drive);
   lastDriveInfoUpdateTimeMillis = timeNow;
   // GProgram::GetProgram().printF("GVfsLocal::updateInternalDriveInfo(\"%s\"[0])", GVArgs(curDir));
}

const GDriveInfo& GVfsLocal::getCurrentDriveInfo () const
{
   updateInternalDriveInfo();
   return driveInfo;
}

longlong GVfsLocal::getFreeDriveSpace () const
{
   const GDriveInfo& driveInfo = getCurrentDriveInfo();
   return driveInfo.diskFree;
}

char GVfsLocal::getCurrentDriveLetter () const
{
   const GDriveInfo& di = getCurrentDriveInfo();
   return char(di.drive + 'A' - 1);
}

GString GVfsLocal::getCurrentDriveName () const
{
   char dl = getCurrentDriveLetter();
   return GString("%c:", GVArgs(dl));
}

const GString& GVfsLocal::getFileSystemName () const
{
   const GDriveInfo& di = getCurrentDriveInfo();
   return di.fsName;
}

bool GVfsLocal::isFat16 () const
{
   const GString& fileSysName = getFileSystemName();
   return fileSysName == "FAT"; 
}

bool GVfsLocal::isFilePreparationPossiblyLengthy () const
{
   return false;
}

bool GVfsLocal::supportsChangeFileNameCase () const 
{
   return !isFat16();
}

bool GVfsLocal::isFileNameCasePreserved () const 
{ 
   return !isFat16();
}

bool GVfsLocal::isFileNameCaseSensitive () const
{
   return false;
}

void GVfsLocal::fillList ( GVfs::List& list,
                           const GString& pattern,
                           bool inclFiles,
                           bool inclDirs,
                           bool inclHidden,
                           bool inclSys ) const
{
   list.clear();

   const int MAX_FCOUNT = 100;
   HDIR hdir = HDIR(HDIR_CREATE);
   ULONG fcount = MAX_FCOUNT;
   if (GSystem::GetPlatformID() >= GSystem::Platform_OS2_4_5)
   {
      FILEFINDBUF3L* ffb = new FILEFINDBUF3L[MAX_FCOUNT];
      APIRET rc = ::DosFindFirst(pattern, &hdir, GVfs::FAttrAll, ffb, sizeof(FILEFINDBUF3L)*MAX_FCOUNT, &fcount, FIL_STANDARDL);
      while (rc == NO_ERROR && fcount >= 1)
      {
         FILEFINDBUF3L* next = ffb;
         for (unsigned i=0; i<fcount; i++)
         {
            bool incl = true;
            const char* fname = next->achName;
            if ((next->attrFile & FILE_DIRECTORY) != 0)
            {
               if (!inclDirs || GFile::IsThisDir(fname) || GFile::IsUpDir(fname))
                  incl = false;
            }
            else
            {
               if (!inclFiles)
                  incl = false;
            }
            if (!inclHidden && (next->attrFile & FILE_HIDDEN) != 0)
               incl = false;
            if (!inclSys && (next->attrFile & FILE_SYSTEM) != 0)
               incl = false;
            if (incl)
            {
               List::Item* item = new List::Item(fname, next->attrFile, next->cbFile, next->cbFileAlloc);
               list.items.add(item);
            }
            next = (FILEFINDBUF3L*) (((char*) next) + next->oNextEntryOffset);
         }
         if (fcount < MAX_FCOUNT)
            break;
         rc = ::DosFindNext(hdir, ffb, sizeof(FILEFINDBUF3L)*MAX_FCOUNT, &fcount);
         if (rc != NO_ERROR)
         {
            rc = NO_ERROR; // Since we have found at least one item; return OK.
            break;
         }
      }
      delete [] ffb;
      ::DosFindClose(hdir);
   }
   else
   {
      FILEFINDBUF3* ffb = new FILEFINDBUF3[MAX_FCOUNT];
      APIRET rc = ::DosFindFirst(pattern, &hdir, GVfs::FAttrAll, ffb, sizeof(FILEFINDBUF3)*MAX_FCOUNT, &fcount, FIL_STANDARD);
      while (rc == NO_ERROR && fcount >= 1)
      {
         FILEFINDBUF3* next = ffb;
         for (unsigned i=0; i<fcount; i++)
         {
            bool incl = true;
            const char* fname = next->achName;
            if ((next->attrFile & FILE_DIRECTORY) != 0)
            {
               if (!inclDirs || GFile::IsThisDir(fname) || GFile::IsUpDir(fname))
                  incl = false;
            }
            else
            {
               if (!inclFiles)
                  incl = false;
            }
            if (!inclHidden && (next->attrFile & FILE_HIDDEN) != 0)
               incl = false;
            if (!inclSys && (next->attrFile & FILE_SYSTEM) != 0)
               incl = false;
            if (incl)
            {
               List::Item* item = new List::Item(fname, next->attrFile, next->cbFile, next->cbFileAlloc);
               list.items.add(item);
            }
            next = (FILEFINDBUF3*) (((char*) next) + next->oNextEntryOffset);
         }
         if (fcount < MAX_FCOUNT)
            break;
         rc = ::DosFindNext(hdir, ffb, sizeof(FILEFINDBUF3)*MAX_FCOUNT, &fcount);
         if (rc != NO_ERROR)
         {
            rc = NO_ERROR; // Since we have found at least one item; return OK.
            break;
         }
      }
      delete [] ffb;
      ::DosFindClose(hdir);
   }
}

GVfs::FileHandle GVfsLocal::openFile ( const GString& path,
                                       GError* errorCode,
                                       OF_Mode modeOpt,
                                       OF_Create createOpt,
                                       OF_Share shareOpt,
                                       int flagsOpt,
                                       OF_ActionTaken* actionTaken )
{
   // Make sure that "errorCode" actually points to a valid object.
   GError localErrorCode;
   if (errorCode == null)
      errorCode = &localErrorCode;
   *errorCode = GError::Ok;

   // No action done yet.
   if (actionTaken != null)
      *actionTaken = GVfs::Action_None;

   // ---
   bool append = ((flagsOpt & GVfs::OF_FLAG_APPEND) != 0);
   bool dontOpenIfExist ((flagsOpt & GVfs::OF_FLAG_DONT_OPEN_IF_EXIST) != 0);
   bool randomAccess = ((flagsOpt & GVfs::OF_FLAG_RANDOM_ACCESS) != 0);
   bool shareDenyRead = (shareOpt == GVfs::Share_DenyRead || shareOpt == GVfs::Share_DenyAll);
   bool shareDenyWrite = (shareOpt == GVfs::Share_DenyWrite || shareOpt == GVfs::Share_DenyAll);
   bool replaceIfExist = (createOpt == GVfs::Create_IfExist || createOpt == GVfs::Create_Always);
   bool createIfNew = (createOpt == GVfs::Create_IfNew || createOpt == GVfs::Create_Always);
   bool neverCreate = (createOpt == GVfs::Create_Never);
   bool readOnly = (modeOpt == GVfs::Mode_ReadOnly);

   // Check argument concistensy.
   if ((append && readOnly) || // Cannot "append" when opening for read only.
       (append && randomAccess) || // Cannot "append" when opening for random access.
       (readOnly && !neverCreate)) // Can never "create" when opening for read only.
   {
      *errorCode = GError::InvalidParameter;
      return null;
   }

   // --
   if (dontOpenIfExist && existFile(path))
   {
      if (actionTaken != null)
         *actionTaken = GVfs::Action_None;
      return null;
   }

   ULONG openFlags = 0;
   if (readOnly || neverCreate)
      openFlags |= FILE_OPEN | OPEN_ACTION_OPEN_IF_EXISTS;
   else
   if (replaceIfExist)
      openFlags |= FILE_CREATE | OPEN_ACTION_REPLACE_IF_EXISTS;
   else
   if (createIfNew)
      openFlags |= FILE_CREATE | OPEN_ACTION_CREATE_IF_NEW;
   ULONG openMode = OPEN_FLAGS_FAIL_ON_ERROR | OPEN_FLAGS_NOINHERIT;
   if (readOnly)
      openMode |= OPEN_ACCESS_READONLY;
   else
      openMode |= OPEN_ACCESS_READWRITE;
   if (randomAccess)
      openMode |= OPEN_FLAGS_RANDOM;
   else
      openMode |= OPEN_FLAGS_SEQUENTIAL;
   if (shareDenyRead && shareDenyWrite)
      openMode |= OPEN_SHARE_DENYREADWRITE;
   else
   if (shareDenyRead)
      openMode |= OPEN_SHARE_DENYREAD;
   else
   if (shareDenyWrite)
      openMode |= OPEN_SHARE_DENYWRITE;
   else // if (!shareDenyRead && !shareDenyWrite)
      openMode |= OPEN_SHARE_DENYNONE;
   // ---
   ULONG os2ActionTaken = 0; // FILE_EXISTED, FILE_CREATED or FILE_TRUNCATED.
   GSysFileHandle hfile = null;
   // Use DosOpenL instead of DosOpen, in order to support files larger than 2GB.
   // APIRET rc = ::DosOpen(path, &hfile, &os2ActionTaken, 0, FILE_NORMAL, openFlags, openMode, null);
   APIRET rc = ::DosOpenL(path, &hfile, &os2ActionTaken, 0, FILE_NORMAL, openFlags, openMode, null);
   if (hfile == null)
   {
      *errorCode = rc;
   }
   else
   {
      if (append)
         setFileSeekPosFromEnd(hfile, 0);
   }
   if (actionTaken != null)
   {
      if (hfile == null)
         *actionTaken = GVfs::Action_None;
      else
      if (os2ActionTaken == FILE_CREATED)
         *actionTaken = GVfs::Action_Created;
      else
      if (os2ActionTaken == FILE_TRUNCATED)
         *actionTaken = GVfs::Action_Truncated;
      else
         *actionTaken = GVfs::Action_Opened;
   }
   GVfs::FileHandle ret = hfile;

   // Find the next free file handle and create a new corresponding object 
   // to represent the open file in our hashtable of open files.
   OpenFile* of = new OpenFile(path, ret);
   synchronized (OpenFiles) 
   {
      GInteger* key = new GInteger(int(ret));
      OpenFiles.put(key, of, true, true);
   } synchronized_end;

   return ret;
}

GError GVfsLocal::closeFile ( GVfs::FileHandle hfile )
{
   synchronized (OpenFiles)
   {
      OpenFile* ofile = getOpenFile(hfile);
      if (ofile == null)
         return GError::InvalidHandle;
      GInteger key(static_cast<int>(hfile));
      OpenFiles.remove(key);
   } synchronized_end;
   return ::DosClose(GSysFileHandle(hfile));
}

GVfsLocal::OpenFile* GVfsLocal::getOpenFile ( GVfs::FileHandle hfile )
{
   synchronized (OpenFiles) 
   {
      GInteger key(static_cast<int>(hfile));
      return OpenFiles.get(key);
   } synchronized_end;
}

GVfs::DeletionHandle GVfsLocal::openDeletion ()
{
   // Return a fixed zero, since the local file system performs 
   // deletions directly anyway.
   return 0;
}

GError GVfsLocal::closeDeletion ( GVfs::DeletionHandle /*hdel*/ )
{
   // Noop, since the local file system performs deletions directly anyway.
   return GError::Ok;
}

GError GVfsLocal::performDeletion ( DeletionHandle hdel )
{
   // Noop, since the local file system performs deletions directly anyway.
   return GError::Ok;
}

GError GVfsLocal::readFromFile ( GVfs::FileHandle hfile,
                                 void* buff,
                                 int numBytesToRead,
                                 int* numBytesActuallyRead )
{
   ULONG read = 0;
   APIRET rc = ::DosRead(GSysFileHandle(hfile), buff, numBytesToRead, &read);
   if (numBytesActuallyRead != null)
      *numBytesActuallyRead = int(read);
   return rc;
}

GError GVfsLocal::writeToFile ( GVfs::FileHandle hfile,
                                const void* buff,
                                int numBytesToWrite,
                                int* numBytesActuallyWritten )
{
   ULONG written = 0;
   APIRET rc = ::DosWrite(GSysFileHandle(hfile), (void*) buff, numBytesToWrite, &written);
   if (numBytesActuallyWritten != null)
      *numBytesActuallyWritten = int(written);
   return rc;
}

int GVfsLocal::getFileAttributes ( const GString& path, GError* errorCode )
{
   if (errorCode != null)
      *errorCode = GError::Ok;
   FILESTATUS3 fs;
   memset(&fs, 0, sizeof(fs));
   fs.attrFile = FILE_NORMAL;
   APIRET rc = ::DosQueryPathInfo(path, FIL_STANDARD, &fs, sizeof(fs));
   if (errorCode != null)
      *errorCode = rc;
   if (rc == NO_ERROR)
      return fs.attrFile;
   return GVfs::FAttrError;
}

GError GVfsLocal::setFileAttributes ( const GString& path, int attr )
{
   FILESTATUS3 fs;
   memset(&fs, 0, sizeof(fs));
   fs.attrFile = attr;
   return ::DosSetPathInfo(path, FIL_STANDARD, &fs, sizeof(fs), 0);
}

GError GVfsLocal::setFileSeekPosFromStart ( GVfs::FileHandle hfile, longlong distanceToMove )
{
   // Use DosSetFilePtrL instead of DosSetFilePtr, in order to support files larger than 2GB.
   // ULONG dummy = 0;
   // return ::DosSetFilePtr(GSysFileHandle(hfile), distanceToMove, FILE_BEGIN, &dummy);
   longlong dummy = 0;
   return ::DosSetFilePtrL(GSysFileHandle(hfile), distanceToMove, FILE_BEGIN, &dummy);
}

GError GVfsLocal::setFileSeekPosFromCurrent ( GVfs::FileHandle hfile, longlong distanceToMove )
{
   if (distanceToMove == 0)
      return GError::Ok;
   // Use DosSetFilePtrL instead of DosSetFilePtr, in order to support files larger than 2GB.
   // ULONG dummy = 0;
   // return ::DosSetFilePtr(GSysFileHandle(hfile), distanceToMove, FILE_CURRENT, &dummy);
   longlong dummy = 0;
   return ::DosSetFilePtrL(GSysFileHandle(hfile), distanceToMove, FILE_CURRENT, &dummy);
}

GError GVfsLocal::setFileSeekPosFromEnd ( GVfs::FileHandle hfile, longlong distanceToMove )
{
   // Use DosSetFilePtrL instead of DosSetFilePtr, in order to support files larger than 2GB.
   // ULONG dummy = 0;
   // return ::DosSetFilePtr(GSysFileHandle(hfile), distanceToMove, FILE_END, &dummy);
   longlong dummy = 0;
   return ::DosSetFilePtrL(GSysFileHandle(hfile), distanceToMove, FILE_END, &dummy);
}

longlong GVfsLocal::getFileSeekPos ( GVfs::FileHandle hfile, GError* rc )
{
   GError localRc;
   if (rc == null)
      rc = &localRc;
   *rc = GError::Ok; // Until the opposite has been proven.
   // Use DosSetFilePtrL instead of DosSetFilePtr, in order to support files larger than 2GB.
   // ULONG pos = 0;
   // *rc = ::DosSetFilePtr(GSysFileHandle(hfile), 0, FILE_CURRENT, &pos);
   longlong pos = 0;
   *rc = ::DosSetFilePtrL(GSysFileHandle(hfile), 0, FILE_CURRENT, &pos);
   if (*rc != GError::Ok)
      return -1;
   return pos;
}

GError GVfsLocal::setFileSize ( GVfs::FileHandle hfile, longlong size )
{
   // Use DosSetFileSizeL instead of DosSetFileSize, in order to support files larger than 2GB.
   // return ::DosSetFileSize(GSysFileHandle(hfile), size);
   return ::DosSetFileSizeL(GSysFileHandle(hfile), size);
}

longlong GVfsLocal::getFileSize ( GVfs::FileHandle hfile, GError* rc )
{
   GError localRc;
   if (rc == null)
      rc = &localRc;
   *rc = GError::Ok; // Until the opposite has been proven.
   FILESTATUS3L fi; // Supports file sizes larger than 2GB.
   memset(&fi, 0, sizeof(fi));
   *rc = ::DosQueryFileInfo(GSysFileHandle(hfile), FIL_STANDARDL, &fi, sizeof(fi));
   if (*rc != GError::Ok)
      return -1;
   return fi.cbFile;
}

GError GVfsLocal::removeDirectory ( DeletionHandle /*hdel*/, 
                                    const GString& path, 
                                    bool trashCan, 
                                    HWND ownerWin )
{
   // Remove trailing slash, if any.
   GString cleanDir = path;
   if (GFile::IsSlash(cleanDir.lastChar()))
      cleanDir.removeLastChar();

   // Make sure the specified path is in fact a directory.
   GFileItem fi(*this, cleanDir);
   if (!fi.isDirectory())
      return GError::FileNotFound;

   // ---
   if (trashCan)
   {
      // Attempt to delete via WPS.
      // This code probably needs Warp 4, because the MENUITEMSELECTED 
      // WPS Parameter was new in Warp 4.
      HOBJECT hobj = ::WinQueryObject(cleanDir);
      if (hobj != null)
         if (::WinSetObjectData(hobj, "MENUITEMSELECTED=109")) // 109=WPMENUID_DELETE
            return GError::Ok;
      // The WPS call failed, so continue and retry 
      // using the more "robust" API ::DosDelete().
   }
   return ::DosDeleteDir(cleanDir);
}

GError GVfsLocal::removeFile ( DeletionHandle /*hdel*/, 
                               const GString& path, 
                               bool trashCan )
{
   // Make sure the specified path is in fact not a directory.
   GFileItem fi(*this, path);
   if (fi.isDirectory())
      return GError::FileNotFound;

   // ---
   if (trashCan)
   {
      // Attempt to delete the file using the WPS. But only if the 
      // WPS is in fact present. This code probably needs Warp 4, because 
      // the MENUITEMSELECTED WPS Parameter was new in Warp 4.
      HOBJECT hobj = ::WinQueryObject(path);
      if (hobj != null)
         if (::WinSetObjectData(hobj, "MENUITEMSELECTED=109")) // 109=WPMENUID_DELETE
            return GError::Ok;
      // The WPS call failed, so continue and retry 
      // using the more "robust" API ::DosDelete().
   }
   return ::DosDelete(path);
}

void GVfsLocal::translateSlashes ( GString& path ) const
{
   path.replaceAll('/', '\\');
}

ulonglong GVfsLocal::CalcFileSizeAlloc ( ulonglong fsize, int bytesPerSector )
{
   if (bytesPerSector <= 0 || fsize <= 0)
      return fsize;
   ulonglong mod = fsize % bytesPerSector;
   if (mod <= 0)
      return fsize;
   return fsize + (bytesPerSector - mod);
}

GError GVfsLocal::loadFileInfo ( GVfs::FileHandle hfile, GFileItem& info )
{
   OpenFile* ofile = getOpenFile(hfile);
   if (ofile == null)
      return GError::InvalidHandle;
   FILESTATUS3L fi; // Supports file sizes larger than 2GB.
   memset(&fi, 0, sizeof(fi));
   APIRET rc = ::DosQueryFileInfo(GSysFileHandle(hfile), FIL_STANDARDL, &fi, sizeof(fi));
   if (rc != NO_ERROR)
      return rc;
   info.attr = fi.attrFile;
   info.timeCreate.set(fi.fdateCreation, fi.ftimeCreation);
   info.timeWrite.set(fi.fdateLastWrite, fi.ftimeLastWrite);
   info.timeAccess.set(fi.fdateLastAccess, fi.ftimeLastAccess);
   info.fileSize = fi.cbFile;
   info.fileSizeAlloc = fi.cbFileAlloc;
   info.path = ofile->path;
   return GError::Ok;
}

GError GVfsLocal::writeAttrAndTimes ( GVfs::FileHandle hfile, 
                                      const GFileItem& info,
                                      const GString& path )
{
   FILESTATUS3 fi;
   memset(&fi, 0, sizeof(fi));
   fi.attrFile = info.attr;
   fi.fdateCreation = info.timeCreate.getDate().getOS2FDate();
   fi.ftimeCreation = info.timeCreate.getTime().getOS2FTime();
   fi.fdateLastWrite = info.timeWrite.getDate().getOS2FDate();
   fi.ftimeLastWrite = info.timeWrite.getTime().getOS2FTime();
   fi.fdateLastAccess = info.timeAccess.getDate().getOS2FDate();
   fi.ftimeLastAccess = info.timeAccess.getTime().getOS2FTime();
   if (hfile != null)
      return ::DosSetFileInfo(GSysFileHandle(hfile), FIL_STANDARD, &fi, sizeof(fi));
   else
      return ::DosSetPathInfo(path, FIL_STANDARD, &fi, sizeof(fi), 0);
}

GError GVfsLocal::moveOrRenameFile ( const GString& existingName, 
                                     const GString& newName,
                                     bool allowCopy )
{
   if (!allowCopy)
   {
      // TODO: This code is probably not 100% safe!
      GFile srcFile(existingName);
      GFile dstFile(newName);
      if (srcFile.containsDrive() && dstFile.containsDrive())
         if (srcFile.getDriveNum() != dstFile.getDriveNum())
            return ERROR_NOT_SAME_DEVICE;
   }
   return ::DosMove(existingName, newName);
}

GError GVfsLocal::createDirectory ( const GString& dir )
{
   return ::DosCreateDir(dir, null);
}

GString GVfsLocal::createTemporaryFile ( const GString& prefix, 
                                         const GString& dir )
{
   // Get the directory of where to create the temporary file.
   GString tempDir = dir;
   if (tempDir == "")
   {
      GProgram& prg = GProgram::GetProgram();
      tempDir = prg.getEnvironmentVar("TMP");
      if (tempDir == "")
         tempDir = prg.getEnvironmentVar("TEMP");
      if (tempDir == "")
         gthrow_(GIOException("Could not create temporary file. Environment variable TMP or TEMP is not set."));
      slash(tempDir);
   }

   // Use only the first four letters of the specified prefix.
   GString pre = prefix;
   if (pre.length() > 4)
      pre.cutTailFrom(4);

   // The total length of the generated file name should not exceed eight
   // characters (due to limitations on some file systems). So use a 
   // sequence number stripper flag that makes this true.
   const int numDigits = 8 - pre.length();
   ulonglong stripFlag = 0x01;
   stripFlag <<= (numDigits * 4);
   stripFlag -= 1;

   // Find an unique filename, and create it.
   GString formatStr = "%s%s%0" + GInteger::ToString(numDigits) + "x.tmp";
   for (;;)
   {
      int counter = (TempFileCounter++ & int(stripFlag));
      GString path(formatStr, GVArgs(tempDir).add(pre).add(counter));
      GVfs::OF_ActionTaken actionTaken = GVfs::Action_Created;
      int flags = GVfs::OF_FLAG_SEQUENTIAL_ACCESS | GVfs::OF_FLAG_DONT_OPEN_IF_EXIST;
      GVfs::FileHandle fh = openFile(path, null,
                                     GVfs::Mode_ReadWrite, 
                                     GVfs::Create_IfNew, 
                                     GVfs::Share_DenyWrite, 
                                     flags,
                                     &actionTaken);
      if (fh != null)
      {
         closeFile(fh);
         if (actionTaken == GVfs::Action_Created)
            return path;
      }
   }
}
