/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_VFSARCHIVEFILE
#define __GLIB_VFSARCHIVEFILE

#include "glib/vfs/GVfs.h"
#include "glib/vfs/GFileItem.h"
#include "glib/util/GKeyBag.h"
#include "glib/util/GHashtable.h"
#include "glib/primitives/GInteger.h"

/**
 *
 * @author  Leif Erik Larsen
 * @since   2002.07.28
 */
class GVfsArchiveFile : public GVfs
{
   public:

      /** 
       * Each file entry item that is contained in the archive file
       * is represented as an instance of this class.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.17
       */
      class Item : public GFileItem
      {
         public:

            /** Position within archive file where the file cell header starts. */
            longlong headPos;

            /** Size of the (possibly-) compressed file data, as containted in the archive file. */
            longlong fileDataSize;

         public:

            Item ( longlong headPos, longlong fileDataSize );
            virtual ~Item ();
      };

   protected:

      /** 
       * Class used to make sure that the archive file is closed 
       * on scope exit. 
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       * @see     #openArchiveFile
       * @see     #closeArchiveFile
       */
      class AutoArchiveFileCloser
      {
         private:

            const GVfsArchiveFile& vfs;

         public:

            AutoArchiveFileCloser ( const GVfsArchiveFile& vfs ) : vfs(vfs) {}
            ~AutoArchiveFileCloser () { vfs.closeArchiveFile(); }
      };

      friend class AutoArchiveFileCloser;

   private:

      class HDir : public GObject
      {
         friend class GVfsArchiveFile;
         int pos;
         int counter;
         GString dir;
         GString findPattern;
         GKeyBag<GString>* returnedNames;
         HDir ( const GString& dir, const GString& findPattern );
         virtual ~HDir ();
      };

      /** 
       * The opened physical archive-file.
       * This will be != null only while at least one archive-entry is 
       * currently open for read-/write operations, or temporarily during 
       * reading of the archive-file entries or other archive-file content.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       * @see     #openArchiveFile
       * @see     #closeArchiveFile
       */
      mutable class GRandomAccessFile* archiveFile;
      mutable int archiveFileOpenCounter;

      GString curDir;
      GVfs::File* vfile;
      mutable GKeyBag<HDir> hdirs;
      bool _supportsChangeFileNameCase;
      bool _isFileNameCasePreserved;
      bool _isFileNameCaseSensitive;

      /** The name of the file system (e.g. ZIP, ARJ, etc.). */
      GString fileSystemName;

      /** 
       * The path to the executable archive file tool program (e.g. ZIP.EXE). 
       * <p>
       * The resason why this is declared as a reference to a string is in 
       * order for the string to be dynamically updated when the user 
       * application code changes the content of the string. E.g. in case 
       * the user adjust the strings in something like an options dialog.
       */
      const GString& toolPath;

      /** 
       * The tool parameter for deleting file and/or directories. 
       * See {@link #toolPath} for a reason why this is a reference.
       */
      const GString& paramsDel;

      // ---
      static int OpenDeletionCounter;
      static GHashtable<GInteger, GArray<GString> > OpenDeletions;

   public:

      /**
       * @author  Leif Erik Larsen
       * @since   2006.02.02
       * @see     #getSlashStr
       */
      static const GString SlashStr;

   protected:

      class GVfsLocal& localVfs;
      mutable GKeyBag<Item> items;
      mutable bool passwordProtected;
      mutable bool hasCalledLoadItems;

      /** The default slash (directory separator) character for this VFS. */
      const char slashChar;

      /**
       * The fully qualified path to the physical archive file on the 
       * local VFS. 
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      const GString physicalArchiveFilePath;

      /**
       * All code using the {@link #archiveFile} should always 
       * synchronize of this lock object before access.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      GObject archiveFileAccessLock;

      GFileItem archiveFileItem;

   protected:

      /**
       * @author  Leif Erik Larsen
       * @since   2002.07.28
       * @param   localVfs  We need the local VFS in order to read the 
       *                    archive-file from a VFS that supports all 
       *                    file seek operations for 100% sure.
       * @param   parentVfs The parent file system, as of where the 
       *                    specified archive file is contained.
       * @param   vfile     The archive file for which this new Archive VFS 
       *                    is to be represented. We will receive the 
       *                    ownership of this object.
       * @param   fsName    The name of the file system, e.g. "ZIP" or "ARJ".
       * @param   toolPath  Path of the archive tool program (e.g. ZIP.EXE).
       * @param   paramsDel The arguments to use for performing delete 
       *                    operations via the archive tool command. Must 
       *                    contain the following substring tags:<br>
       *                    %1=Will be replaced by the path of the archive-file.<br>
       *                    %2=Will be replaced by the path of the temporary 
       *                       file containing the list of which files to 
       *                       delete from the archive file.<br>
       * @throws  GFileNotFoundException if the specified file can not
       *                                 be found.
       */
      GVfsArchiveFile ( class GVfsLocal& localVfs, 
                        class GVfs& parentVfs, GVfs::File* vfile,
                        const GString& fsName,
                        char slashChar,
                        bool supportsChangeFileNameCase,
                        bool isFileNameCasePreserved,
                        bool isFileNameCaseSensitive,
                        const GString& toolPath,
                        const GString& paramsDel );

   public:

      virtual ~GVfsArchiveFile ();

   protected:

      /** Disable the copy constructor. */
      GVfsArchiveFile ( const GVfsArchiveFile& src ) : GVfs(src.parentVfs), toolPath(src.toolPath), paramsDel(src.paramsDel), localVfs(src.localVfs), slashChar(src.slashChar) {}

   private:

      /** Disable the assignment operator. */
      GVfsArchiveFile& operator= ( const GVfsArchiveFile& ) { return *this; }

      /** 
       * Get a pointer to the array of which archive file entries are 
       * requested for deletion by the specified deletion handle,
       * or null if the specified deletion handle is invalid.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      GArray<GString>* getOpenDeletion ( GVfs::DeletionHandle hdel );

      /**
       * Common code for {@link #removeFile} and {@link #removeDirectory}.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @param   path   Path of the file or directory of which to remove.
       * @param   isdir  True if the path is to a directory, 
       *                 or else false if it is to a file.
       */
      GError removeFileOrDirectory ( DeletionHandle hdel, 
                                     const GString& path, 
                                     bool isdir );

   public:

      /**
       * Close the deletion handle recently returned by {@link #openDeletion}.
       * See the declaring {@link GVfs#closeDeletion} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      virtual GError closeDeletion ( DeletionHandle hdel );

      /** 
       * Return true if the path contains at least one slash character. 
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual bool containsAnySlash ( const GString& path ) const;

      virtual int findFirst ( GFileItem& fitem, const GString& path = GString::Empty, bool* cancelSem = null, int* preLoadCounter = null ) const;
      virtual bool findFirstMightNeedLongTime () const;
      virtual bool findNext ( int hdir, GFileItem& fitem ) const;
      virtual void findClose ( int hdir ) const;

      /**
       * Get a reference to the current directory within archive (without
       * prefix slash).
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.30
       * @param   slash   True if we shall trail the returned string with 
       *                  a system dependent slash character, but that is 
       *                  only if the current directory is not the root.
       *                  If the current directory is the root directory 
       *                  then we will always return an empty string, 
       *                  regardless of this parameter.
       */
      virtual GString getCurrentDirectory ( bool slash ) const;

      /**
       * Get the attributes (readonly, hidden, etc.) of the specified file.
       * See the declaring {@link GVfs#getFileAttributes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual int getFileAttributes ( const GString& path, GError* errorCode = null );

      /**
       * Get the name of the archive file system (e.g. "ZIP", "ARJ", etc.).
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual const GString& getFileSystemName () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2007.02.05
       */
      virtual longlong getFreeDriveSpace () const;

      /**
       * Get the root directory specification string (an empty string).
       *
       * @author  Leif Erik Larsen
       * @since   2002.07.30
       */
      virtual GString getRootDir () const;

      /**
       * Return the name of the archive file exactly as specified to the
       * constructor.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual GString getPhysicalSelfName () const;

      /**
       * Return the filename part of the fully qualified archive path
       * that was specified to the constructor.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual GString getLogicalSelfName () const;

      /**
       * Most archive file formats uses a single forward slash to separate
       * directories. Therefore, this default implementation returns 
       * a single character string with a forward slash.
       *
       * @author  Leif Erik Larsen
       * @since  2006.02.02
       */
      virtual const GString& getSlashStr () const;

      /**
       * Overrides super in order to handle root directory 
       * correctly for archive files.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      virtual GError getWalkedPath ( const GString& srcDir, GString& walkedDir ) const;

      /**
       * Return true if and only if it is the root directory of the 
       * file system that is the current directory. For this class,
       * this is true if the current directory is empty.
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.04
       */
      virtual bool isCurrentDirectoryTheRoot () const;

      /**
       * Always return true, because calllling any of the file
       * preparation methods can possibly take some rather long time.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.20
       */
      virtual bool isFilePreparationPossiblyLengthy () const;

      /**
       * Return true if the specified character is a slash 
       * (directory seprataor).
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual bool isSlash ( char chr ) const;

      /**
       * Open a new deletion handle to be used with 
       * {@link #removeFile} and/or {@link #removeDirectory}.
       * See the declaring {@link GVfs#openDeletion} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       * @see     #closeDeletion
       */
      virtual DeletionHandle openDeletion ();

      /**
       * Perform all file and/or directory deletion operations requested
       * on the specified deletion handle.
       * See the declaring {@link GVfs#performDeletion} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.08.31
       */
      virtual GError performDeletion ( DeletionHandle hdel );

      /**
       * See the declaring {@link GVfs#removeDirectory} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.22
       */
      virtual GError removeDirectory ( DeletionHandle hdel, 
                                       const GString& path, 
                                       bool trashCan,
                                       HWND ownerWin = null );

      /**
       * See the declaring {@link GVfs#removeFile} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.03.22
       */
      virtual GError removeFile ( DeletionHandle hdel, 
                                  const GString& path, 
                                  bool trashCan );

      /**
       * @author  Leif Erik Larsen
       * @since   2002.08.04
       */
      virtual GError setCurrentDirectory ( const GString& dir );

      /**
       * Make sure the specified dir has a trailing slash, if not empty.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual GString& slash ( GString& dir ) const;

      /**
       * Convert backslashes (if any) to slashes for this VFS, or vice verca.
       * See the declaring {@link GVfs#translateSlashes} for details.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       */
      virtual void translateSlashes ( GString& path ) const;

      virtual bool supportsChangeFileNameCase () const;
      virtual bool isFileNameCasePreserved () const;
      virtual bool isFileNameCaseSensitive () const;

   protected:

      /**
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       * @see     #openArchiveFile
       */
      void closeArchiveFile () const;

      /**
       * @author  Leif Erik Larsen
       * @since   2005.03.23
       */
      Item* getItem ( const GString& path );

      /**
       * This method is automatically called from {@link #findFirst} but
       * only the very first time that method is called.
       *
       * Subclasses typically overrides this method in order to initially
       * load the list of all filenames in the archive file (e.g. from
       * the certral directory list of ZIP-files) if needed.
       *
       * The default implementation of this method does nothing
       * but return.
       *
       * @author  Leif Erik Larsen
       * @since   2002.08.22
       * @param   cancelSem       See {@link GVfs#findFirst}.
       * @param   preLoadCounter  See {@link GVfs#findFirst}.
       * @throws  GIOException    If an I/O error occurs during reading of
       *                          of the archive file entries. Any subclasses
       *                          of <i>GIOException</i> can also be thrown.
       */
      virtual void loadItems ( bool* cancelSem, int* preLoadCounter ) const;

      /**
       * Make sure the archive file is open, and get a reference to it.
       * It will stay open until the archive file reference counter 
       * is back to zero. That is, when {@link #closeArchiveFile} is called
       * equally many times as which this method has been called.
       *
       * @author  Leif Erik Larsen
       * @since   2005.09.01
       * @throws  GIOException in case of any error opening the file.
       * @return  A reference to the archive file.
       * @see     #closeArchiveFile
       */
      class GRandomAccessFile& openArchiveFile () const;
};

#endif
