/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_EAS
#define __GLIB_EAS

#include "glib/vfs/GFile.h"
#include "glib/util/GKeyBag.h"

#define EA_NAME_LONGNAME ".LONGNAME"

/**
 * This class is used to hold the Extended Attributes (EAs) of a single file.
 *
 * Using this class makes it extremely easy to manipulate and query
 * the EAs of a file.
 *
 * <b>Mark</b> that this class is completely dummy in all systems but OS/2.
 * E.g. in Windows none of the methods will do anything at all, but return
 * some dummy success-value.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.04
 */
class GExtendedAttributes : public GObject
{
   public:

      enum EAType
      {
         TYPE_UNKNOWN = 0,
         TYPE_STRING
      };

      /**
       * The DeleteList is used to hold the names of EAs that need to be
       * explicitly deleted before the current EAs are written.
       *
       * This is necessary because there does not exist a way to
       * automatically delete all existing EAs associated with a file.
       * The next field allows the structures to be linked.
       */
      typedef struct _DeleteList
      {
         char* EAName;
         struct _DeleteList* next;
      }
         DELETELIST;

      /**
       * The EA code revolves around the HOLDFEA structure.
       *
       * This structure is a linked list which contains one EA per structure.
       * Each HOLDFEA structure is dynamically allocated as are the two string
       * pointer members of the structure, szName and aValue. This is done to make
       * handling the EAs easier and more flexible.
       *
       * The HOLDFEA is used to hold individual EAs. The member names correspond
       * directly to those of the FEA structure. Note however, that both szName
       * and aValue are pointers to the values. An additional field, next, is
       * used to link the HOLDFEA's together to form a linked list.
       */
      typedef struct _HoldFEA
      {
         BYTE fEA;
         BYTE cbName;
         USHORT cbValue;
         char* szName;
         char* aValue;
         struct _HoldFEA* next;
      }
         HOLDFEA;

   private:

      HOLDFEA* eas;

      /** Bag of which EA's to delete. */
      GKeyBag<GString>* dlist;

   public:

      GExtendedAttributes ();
      explicit GExtendedAttributes ( GSysFileHandle hfile );
      explicit GExtendedAttributes ( const GString& fname );
      explicit GExtendedAttributes ( const GFile& fname );
      virtual ~GExtendedAttributes ();

   private:

      void discardRemovedEAs ( bool isPath, void* file );

   public:

      bool operator== ( const GExtendedAttributes& eas2 ) const;
      bool operator!= ( const GExtendedAttributes& eas2 ) const;

      int getEACount () const;
      HOLDFEA* getEAByName ( const GString& name ) const;
      HOLDFEA* getIndexedEA ( int index ) const;
      GString getIndexedEAName ( int index ) const;
      EAType getIndexedEAType ( int index ) const;

      /**
       * If the specified EA Does not exist, or if it is not a string
       * type, then we will return an empty string.
       */
      GString getEAString ( const GString& name ) const;

      bool isEADefined ( const GString& name ) const;
      void removeEA ( const GString& name );
      void setEAString ( const GString& name, const GString& value );
      APIRET writeEAs ( GSysFileHandle hfile );
      APIRET writeEAs ( const GString& fname );

   public:

      static bool DiscardEA ( GSysFileHandle hfile, const GString& eaname );
      static bool DiscardEA ( const GString& fname, const GString& eaname );

      /**
       * Query a file's EA names and values.
       *
       * <b>Note1:</b> Routine does NOT do full memory error trapping.
       *
       * <b>Note2:</b> This routine does NOT prevent other processes
       * from accessing the file's EAs while it is reading them in,
       * or while the program is editing them.
       *
       * @param   isPath  Value to determine wether fname is a pointer to
       *                  a file path (char *) or a pointer to a file
       *                  handle (GSysFileHandle). Legal values are:
       *                  True if fname is a file path (char *).
       *                  False if fname is a file handle (GSysFileHandle).
       * @param   fname   Dependent of what value is specified in isPath.
       * @return  A linked list of the EAs for the specified file
       *          if it successfully reads in the EAs. In case of any
       *          error we will return null. We will return null also if
       *          there was no EAs on the specified source file. The
       *          returned linked list (if not null) must be freed
       *          using {@link #freeEAs}.
       */
      static HOLDFEA* QueryEAs ( bool isPath, void* fname );

      /**
       * This routine updates the EAs on disk to reflect their current
       * condition in memory.
       *
       * First, all EAs in the delete list are removed from the disk,
       * then all EAs in the pHoldFEA list are written to disk.
       *
       * This routine is not bulletproof as it does not get exclusive
       * access to the file EAs, nor does it handle out of disk space sort
       * of errors. Also, memory fetches are not fully error trapped.
       *
       * @param   pHFEA   Buffer as returned from {@link #queryEAs}.
       * @param   pDL     Linked list of name of which EAs to delete.
       * @param   isPath  Value to determine wether fname is a pointer
       *                  to a file path (char *) or a pointer to a file
       *                  handle (GSysFileHandle). Legal values are:
       *                  True if fname is a file path (char *).
       *                  False if fname is a file handle (GSysFileHandle).
       * @param   fname   Dependent of what value is specified in isPath.
       * @return  The OS/2 API Return Code. NO_ERROR on success.
       */
      static APIRET WriteEAs ( HOLDFEA* pHFEA, DELETELIST* pDL, bool isPath, void* fname );

      /**
       * Free all the EAs in the specified linked list of EAs.
       */
      static void FreeEAs ( HOLDFEA* pFEA );

   private:

      /**
       * Create a new and empty EA list buffer.
       *
       * The returned buffer must be freed with freeEAs() when
       * no longer needed.
       */
      static HOLDFEA* GenMakeEA ();

      /**
       * Free the specified single EA only.
       */
      static void GenFreeEA ( HOLDFEA* pHFEA );

      /**
       * Remove the named EA from the specified file or directory.
       *
       * @param  isPath  True if fname is a path, or else false if it
       *                 is a pointer to a GSysFileHandle.
       * @param  fname   Depending of isPath this is either the GSysFileHandle or
       *                 the path of the file or directory of which to
       *                 discard EA.
       * @param  eaname  Name of which EA to discard.
       * @return True on success, or else false on any error.
       */
      static bool GenDiscardEA ( bool isPath, void* fname, const GString& eaname );

      /**
       * Set the value buffer of the specified EA.
       *
       * If the EA already has a value
       * then free it before setting the new value. The new EA value will
       * be of type EAT_ASCII.
       */
      static void SetAsciiEAValue ( HOLDFEA* pHFEA, const GString& eavalue );

      /**
       * If the specified EA Does not exist, or if it is not a
       * string type, then we will return an empty string.
       */
      static GString GenGetEAStringValue ( HOLDFEA* pHFEA, const GString& eaname );

      /**
       * Return true if and only if the specified EA is
       * defined in the list.
       */
      static bool GenIsEADefined ( HOLDFEA* pHFEA, const GString& eaname );

      static void GenSetEA ( HOLDFEA* pHFEA, const GString& eaname, const GString& eavalue );
};

#endif
