/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_TOKEN
#define __GLIB_TOKEN

#include "glib/util/GAbstractToken.h"

/**
 * This class is used as the return type of {@link GTokenizer#getNextToken}.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.15
 */
class GToken : public GAbstractToken
{
   friend class GTokenizer;

   private:

      GString tokenStr;
      char quoteChar;
      bool quoted;

   public:

      /**
       * Create a new token that is initially empty.
       */
      GToken ();

      /**
       * Create a new token that initially contains the specified string.
       */
      explicit GToken ( const GString& token );

      /**
       * Create a new token that is initially equal to the specified token.
       */
      GToken ( const GToken& src );

      virtual ~GToken ();

   public:

      GToken& operator= ( const GToken& src );

      /**
       * Return true if the current token equals the specified String.
       * We will return true if, and only if, the specified string is 100%
       * equal to the current token.
       */
      virtual bool operator== ( const GString& str ) const;

      virtual bool operator!= ( const GString& str ) const;

      /**
       * @see GAbstractToken#isQuoted
       */
      virtual bool isQuoted () const;

      /**
       * @see GAbstractToken#getQuoteCharacter
       */
      virtual char getQuoteCharacter () const;

      /**
       * @see GAbstractToken#getCharacter
       */
      virtual char getCharacter ( int idx ) const;

      /**
       * @see GAbstractToken#isEmpty
       */
      virtual bool isEmpty () const;

      /**
       * @see GAbstractToken#getLength
       */
      virtual int getLength () const;

      /**
       * Return true if and only if this token currently contains a
       * whitespace token. That is if all characters of the token are
       * whitespace characters only.
       *
       * An empty token is not considered a whitespace token,
       * and will therefore not return true in that case.
       *
       * This method will return true only if the tokenizer is set
       * to tokenize whitespace. See {@link GTokenizer#isTokenizeWhiteSpace}.
       */
      bool isWhiteSpace ( bool ) const;

      /**
       * Implements {@link GAbstractToken#toString}.
       *
       * @see #toString(bool)
       */
      virtual GString toString () const;

      /**
       * Get the current token as a string, with or without enclosing
       * pair of quotes if needed.
       *
       * @author  Leif Erik Larsen
       * @since   2000.07.29
       * @param   inclQuotes True if we shall automatically include the
       *                     enclosing pair of quotes, but only if the
       *                     current token was originally enclosed in a
       *                     pair of quotes. If false is specified then we
       *                     will never quote the returned string anyway.
       * @see     #toString()
       */
      GString toString ( bool inclQuotes ) const;

      /**
       * Clear the current token.
       */
      void clear ();

      /**
       * Append the specified character to the token.
       */
      void append ( char chr );
};

#endif

