/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_SECTIONBAG
#define __GLIB_SECTIONBAG

#include <stdio.h>
#include "glib/util/GKeyBag.h"
#include "glib/gui/GColor.h"
#include "glib/io/GOutputStream.h"

/**
 * This is a Container Class of where to store entries/items within
 * named sections.
 *
 * A typical example of what this class can be used to is a profile (INI-file)
 * manager. Such a profile manager can automatically parse and read all
 * sections and items through this class. The sections and items will be
 * physically stored in a file, which content might look like the below
 * example:
 *
 * <pre>
 *
 * entry1=Value1
 * entry2=Value2
 *
 * [section1]
 * entry1=Value1
 * entry2=Value2
 *
 * [section2]
 * entry1=Value1
 * entry2=Value2
 *
 * </pre>
 *
 * This example will cause three section bags to be managed by this class. The
 * first section will be an unnamed section (the default section, while the
 * two next sections will be named <code>section1</code> and
 * <code>section2</code> respectively. All the three sections will contain two
 * entries, named <code>entry1</code> and <code>entry2</code>.
 *
 * The section and entry names are always case insensitive.
 *
 * @author  Leif Erik Larsen
 * @since   1997.09.04
 */
class GSectionBag : public GObject
{
   private:

      /**
       * Path to the actual ini-file that was used to load the sectionbag.
       */
      GString path;

      /**
       * True if buffer has been changed and is "dirty".
       * @see #isDirty
       */
      bool changed;

      /**
       * The bag of where to store each of the sections.
       *
       * The section name
       * will be used as the key value and the section data will be a
       * reference to a <code>HBag</code> object. Each section bag object
       * can hold an unlimited number of entries.
       */
      GKeyBag< GKeyBag<GString> > sectionsBag;

      /**
       * True if, and only if, parameters was successfully loaded from
       * specified path.
       *
       * If this is false then we can assume that either
       * 1) No file path was specified to the constructor (path is empty),
       * or 2) the file did not exist.
       */
      bool loadedOK;

      /**
       * True if the section names and the key names are case sensitive.
       */
      bool caseSen;

   public:

      /**
       * Create a new section bag to maintain the entries in the
       * specified INI-file.
       *
       * @author  Leif Erik Larsen
       * @param   path Path of file of where to initially load sections,
       *               or an empty string to start with an empty bag.
       */
      explicit GSectionBag ( const GString& path = GString::Empty, 
                             bool caseSen = true );

   public:

      /**
       * Write all the current profiles from this profile manager object
       * back to the file path of where the profiles was initially read.
       *
       * If the profile set hasn't been changed at all since it was
       * initially read then we will do nothing but return.
       * Most user programs will not use this method directly, since it
       * is usually called automatically by {@link GProgram} to store 
       * the user profile (ini-file) upon program exit.
       *
       * @author  Leif Erik Larsen
       * @return  True on success, or else false on any error.
       * @throws  GIOException in case of any error.
       */
      void write ();

      /**
       * Write all the current profiles from this profile manager object
       * to the specified file path.
       *
       * @author  Leif Erik Larsen
       * @return  True on success, or else false on any error.
       * @throws  GIOException in case of any error.
       */
      void write ( const GString& path );

      /**
       * Write all the current profiles from this profile manager object
       * to specified output stream.
       *
       * @author  Leif Erik Larsen
       * @param   os   The output stream of where to write the profiles.
       * @return  Will always return true.
       * @throws  GIOException in case of any error.
       */
      void write ( const GOutputStream& os );

      GKeyBag<GString>& getIndexedSectionBag ( int index );

      GString getIndexedSectionName ( int index );

      /**
       * Get a reference to the specified section.
       * If the specified section doesn't exist then we will return NULL.
       *
       * @author  Leif Erik Larsen
       */
      GKeyBag<GString>* getSectionBag ( const GString& section );

      int getNumberOfSections () const;

      /**
       * Remove the specified section completely from the Section GKeyBag.
       * All of its items will, of course, be deleted too.
       *
       * @author  Leif Erik Larsen
       * @return  True on success, or else false on
       *          any error (e.g. the specified item didn't
       *          exist in the bag).
       */
      bool deleteSection ( const GString& section );

      /**
       * Remove the specified item from the specified section.
       *
       * @author  Leif Erik Larsen
       * @return  True on success, or else false on
       *          any error (e.g. the specified item didn't
       *          exist in the bag).
       */
      bool deleteItem ( const GString& section, const GString& item );

      /**
       * Put the specified item data in the specified section, but only
       * if the <i>onlyIf</i> parameter is true or else we will do
       * nothing but return.
       *
       * If the specified section doesn't already contain the
       * specified key then this method will automatically add the
       * value as a new entry of the section.
       *
       * @author  Leif Erik Larsen
       */
      void putString ( const GString& section, 
                       const GString& item, 
                       const GString& value, 
                       bool onlyIf = true );

      /**
       * Put the specified item data in the specified section, but only
       * if the <i>onlyIf</i> parameter is true or else we will do
       * nothing but return.
       *
       * If the specified section doesn't already contain the
       * specified key then this method will automatically add the
       * value as a new entry of the section.
       *
       * @author  Leif Erik Larsen
       */
      void putBool ( const GString& section, 
                     const GString& item, 
                     bool value, 
                     bool onlyIf = true );

      /**
       * Put the specified item data in the specified section, but only
       * if the <i>onlyIf</i> parameter is true or else we will do
       * nothing but return.
       *
       * If the specified section doesn't already contain the
       * specified key then this method will automatically add the
       * value as a new entry of the section.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.07
       */
      void putChar ( const GString& section, 
                     const GString& item, 
                     char value, 
                     bool onlyIf = true );

      /**
       * Put the specified item data in the specified section, but only
       * if the <i>onlyIf</i> parameter is true or else we will do
       * nothing but return.
       *
       * If the specified section doesn't already contain the
       * specified key then this method will automatically add the
       * value as a new entry of the section.
       *
       * @author  Leif Erik Larsen
       */
      void putInt ( const GString& section, 
                    const GString& item, 
                    int value, 
                    bool onlyIf = true );

      /**
       * Put the specified item data in the specified section, but only
       * if the <i>onlyIf</i> parameter is true or else we will do
       * nothing but return.
       *
       * If the specified section doesn't already contain the
       * specified key then this method will automatically add the
       * value as a new entry of the section.
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.03
       */
      void putLong ( const GString& section, 
                     const GString& item, 
                     longlong value, 
                     bool onlyIf = true );

      /**
       * Put the specified item data in the specified section, but only
       * if the <i>onlyIf</i> parameter is true or else we will do
       * nothing but return.
       *
       * If the specified section doesn't already contain the
       * specified key then this method will automatically add the
       * value as a new entry of the section.
       *
       * @author  Leif Erik Larsen
       */
      void putColor ( const GString& section, 
                      const GString& item, 
                      const GColor& color, 
                      bool onlyIf = true );

      /**
       * Get the string value of the specified item in the
       * specified section.
       *
       * If the specified section or item within section doesn't exist
       * then this method will return the specified default string.
       *
       * @author  Leif Erik Larsen
       */
      GString getString ( const GString& section, 
                          const GString& item, 
                          const GString& def = GString::Empty );

      /**
       * Get the integer value of the specified item in the
       * specified section.
       *
       * If the specified section or item within section doesn't exist
       * then this method will return the specified default integer value.
       *
       * @author  Leif Erik Larsen
       */
      int getInt ( const GString& section, 
                   const GString& item, 
                   int def = 0 );

      /**
       * Get the integer value of the specified item in the
       * specified section.
       *
       * We will ensure that the returned integer doesn't exceed the
       * specified range. If the specified section or item within section
       * doesn't exist then this method will return the specified default
       * integer value.
       *
       * @author  Leif Erik Larsen
       */
      int getInt ( const GString& section, 
                   const GString& item, 
                   int min, 
                   int max, 
                   int def );

      /**
       * Get the long value of the specified item in the
       * specified section.
       *
       * If the specified section or item within section doesn't exist
       * then this method will return the specified default long value.
       *
       * @author  Leif Erik Larsen
       * @since   2007.02.03
       */
      longlong getLong ( const GString& section, 
                         const GString& item, 
                         longlong def = 0L );

      /**
       * Get the boolean value of the specified item in the
       * specified section.
       *
       * If the specified section or item within section doesn't exist
       * then this method will return the specified default boolean value.
       *
       * @author  Leif Erik Larsen
       */
      bool getBool ( const GString& section, 
                     const GString& item, 
                     bool def = false );

      /**
       * Get the character value of the specified item in the
       * specified section.
       *
       * If the specified section or item within section doesn't exist
       * then this method will return the specified default character value.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.07
       */
      char getChar ( const GString& section, 
                     const GString& item, 
                     char def = '\0' );

      /**
       * Get the color value of the specified item in the
       * specified section.
       *
       * If the specified section or item within section doesn't exist
       * then this method will return the specified default color value.
       *
       * @author  Leif Erik Larsen
       */
      GColor getColor ( const GString& section, 
                        const GString& item, 
                        const GColor& def );

      /**
       * Return true if and only if the content of the section bag has been
       * changed by any means since it was last saved with {@link #write}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.10.22
       */
      bool isDirty () const;

      bool wasLoadedOK () const;

      const GString& getPath () const;
};

#endif
