/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include "glib/util/GProcessLauncher.h"
#include "glib/vfs/GFile.h"
#include "glib/sys/GSystem.h"
#include "glib/exceptions/GIllegalStateException.h"

const GString GProcessLauncher::COMSPEC = "COMSPEC";
const GString GProcessLauncher::DEFCOMSPEC = "CMD.EXE";

GULong GProcessLauncher::PipeCounter;
const GString GProcessLauncher::SysShellPrefixArgs = "/S /C ";

GObject GProcessLauncher::execSynch;

GProcessLauncher::GProcessLauncher ( const GString& workingDir,
                                     const GString& prgName,
                                     const GString& paramStr,
                                     bool forceNewSession,
                                     bool closeOnExit,
                                     bool startConFullScreen,
                                     bool startDosFullScreen )
                 :GThread("Process-Launcher-Thread", 32768),
                  prgName(prgName),
                  paramStr(paramStr),
                  workingDir(workingDir),
                  forceNewSession(forceNewSession),
                  closeOnExit(closeOnExit),
                  startedAsChild(false),
                  startConFullScreen(startConFullScreen),
                  startDosFullScreen(startDosFullScreen),
                  errorCode(EC_NoError), // Until the opposite has been proven.
                  exitCodeFromProcess(-1),
                  startTime(), // Initialize with the current time.
                  childStdIn(null),
                  childStdOut(null),
                  childStdErr(null),
                  childPidToWaitFor(0)
{
   memset(&childRes, 0, sizeof(childRes));
   pipeReaderMediator = null;
   this->prgName.trim();
   this->paramStr.trim();
   this->workingDir.trim();
   if (this->workingDir == "")
      this->workingDir = GFile::GetCurrentDir();
}

GProcessLauncher::~GProcessLauncher ()
{
   delete childStdIn;
   delete childStdOut;
   delete childStdErr;
}

void GProcessLauncher::programPathHasBeenValidated ( const GString& prgName,
                                                     const GString& paramStr )
{
}

void GProcessLauncher::processHasBeenLaunched ( bool asChild )
{
}

void GProcessLauncher::processHasFinished ( bool asChild,
                                            int result,
                                            bool normalExit )
{
}

GString GProcessLauncher::getWorkingDir ( const GString& defaultDir ) const
{
   return defaultDir;
}

const GString& GProcessLauncher::getProgramName () const
{
   return prgName;
}

const GString& GProcessLauncher::getParamString () const
{
   return paramStr;
}

const GTime& GProcessLauncher::getStartTime () const
{
   return startTime;
}

bool GProcessLauncher::isRunningChild () const
{
   if (!isRunning())
      return false;
   PID dummy = 0;
   PID waitPid = PID(childRes.codeTerminate);
   if (waitPid == 0)
      return false;
   // We don't want to overwrite our "childRes" until the process has
   // actually finished, because we use this variable to store the
   // PID of the running child. Thus, use a temporary record of where
   // to get the exit information of the child process, in case the
   // child has not yet finsihed.
   RESULTCODES res = childRes;
   // Check if the child process has finished.
   APIRET rc = ::DosWaitChild(DCWA_PROCESS, DCWW_NOWAIT, &res, &dummy, waitPid);
   if (rc == ERROR_CHILD_NOT_COMPLETE)
      return true; // Still running!
   // Ok, the child process has finished. So copy its exit information
   // to our mutable "childRes" variable so the process launcher can
   // get access to the exit information of the child process.
   childRes = res;
   return false;
}

bool GProcessLauncher::breakChildProg ( GSystem::BreakType bt, bool breakTree )
{
   // Both the TID and the PID must be valid.
   if (!isRunning() || !isRunningChild())
      return false;

   PID pid = PID(childRes.codeTerminate);
   return GSystem::BreakProcess(pid, bt, breakTree);
}

GProcessLauncher::ErrorCode GProcessLauncher::prepareTheWorkingDir ()
{
   if (workingDir == "")
      return EC_NoError;

   workingDir = getWorkingDir(workingDir);
   if (workingDir == "")
      return EC_NoError;

   APIRET rc = GFile::SetCurrentDir(workingDir);
   if (rc != NO_ERROR)
   {
      systemErrorCode = rc;
      return EC_ErrActDirX; // GString("Error activating directory: '%s'\n", GVArgs(workingDir));
   }

   return EC_NoError;
}

void GProcessLauncher::waitForTheChildProcessToFinish ()
{
   PID dummy = 0;
   PID pid = PID(childRes.codeTerminate);
   ::DosWaitChild(DCWA_PROCESS, DCWW_WAIT, &childRes, &dummy, pid);
}

void GProcessLauncher::addTextToChildStdInQueue ( const GString& txt )
{
   if (txt == GString::Empty)
      return;
   GObject::Synchronizer synch(stdInData);
   stdInData.add(new GString(txt));
   // TODO: Wait for support for GObject::wait() & notify().
   // TODO: stdInData.notifyAll();
   stdInData.notifyAll();
}

void GProcessLauncher::waitForSomeDataOnStdIn ()
{
   // Wait for {@link #addTextToChildStdInQueue} to notify us
   // when data has arrived.
   // GObject::Synchronizer synch(stdInData);
   // TODO: Is this robust by now?
   stdInData.wait();
}

GString GProcessLauncher::getNextChildStdInText ()
{
   GObject::Synchronizer synch(stdInData);
   if (stdInData.getCount() <= 0)
      return GString::Empty;
   GString ret = stdInData.get(0);
   stdInData.remove(0);
   return ret;
}

GFileOutputStream& GProcessLauncher::getChildStdIn ()
{
   if (childStdIn == null)
      gthrow_(GIllegalStateException("Child process is not running!"));
   return *childStdIn;
}

GPipeInputStream& GProcessLauncher::getChildStdOut ()
{
   if (childStdOut == null)
      gthrow_(GIllegalStateException("Child process is not running!"));
   return *childStdOut;
}

int GProcessLauncher::getPidOfRunningChildProcess () const
{
   return childPidToWaitFor;
}

bool GProcessLauncher::isChildWaitingForDataOnItsStdIn () const
{
   // Always return false if the child program is no longer running.
   // Because when the child has finished the pipe semaphore might
   // be signaled even if there is actually nooone attemting to read.
   if (!isRunningChild())
      return false;
   // Bug in OS/2 4.0? DosQueryNPipeSemState returns 111 (ERROR_BUFFER_OVERFLOW)
   // if buffer is just an instance of PIPESEMSTATE. Therfore we allocates
   // a double size buffer.
   char buff[2*sizeof(PIPESEMSTATE)];
   APIRET rc = ::DosQueryNPipeSemState(HSEM(pipeReaderMediator), (PIPESEMSTATE*) &buff[0], sizeof(buff));
   if (rc == NO_ERROR)
   {
      PIPESEMSTATE *state = (PIPESEMSTATE*) &buff[0];
      if (state->fFlag & NPSS_WAIT)
         return true;
   }
   return false;
}

void GProcessLauncher::createPipe ( GSysFileHandle* hRead,
                                    GSysFileHandle* hWrite )
{
   *hRead = null;
   *hWrite = null;

   ULONG buffSize = 1024;
   ulonglong id = PipeCounter.increment();
   GString fname("\\pipe\\pid%d\\%08x.pip", GVArgs(getpid()).add(id));
   ::DosCreateNPipe(fname, hRead, NP_INHERIT | NP_ACCESS_INBOUND, NP_NOWAIT | NP_TYPE_BYTE | NP_READMODE_BYTE | 1, buffSize, buffSize, 0);
   ::DosConnectNPipe(*hRead);
   ::DosSetNPHState(*hRead, NP_WAIT | NP_READMODE_BYTE);
   ULONG actionTaken = 0;
   ::DosOpen(fname, hWrite, &actionTaken, 0, FILE_NORMAL,
             OPEN_ACTION_OPEN_IF_EXISTS | OPEN_ACTION_FAIL_IF_NEW,
             OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYREADWRITE, null);
}

static void MakeSureStdStreamsAreNotInheritable ()
{
   ULONG state = 0;
   HFILE stdIn = 0, stdOut = 1, stdErr = 2;
   ::DosQueryFHState(stdIn, &state);
   if ((state & OPEN_FLAGS_NOINHERIT) == 0)
      ::DosSetFHState(stdIn, state & 0xFF00 | OPEN_FLAGS_NOINHERIT);
   ::DosQueryFHState(stdOut, &state);
   if ((state & OPEN_FLAGS_NOINHERIT) == 0)
      ::DosSetFHState(stdOut, state & 0xFF00 | OPEN_FLAGS_NOINHERIT);
   ::DosQueryFHState(stdErr, &state);
   if ((state & OPEN_FLAGS_NOINHERIT) == 0)
      ::DosSetFHState(stdErr, state & 0xFF00 | OPEN_FLAGS_NOINHERIT);
}

static void ActivateSystemDefaultStdStreams ()
{
   HFILE stdIn = 0, stdOut = 1, stdErr = 2;
   HFILE hread = null, hwrite = null;
   ULONG actionTaken = 0;
   ::DosOpen("CON", &hread, &actionTaken, 0, FILE_NORMAL, FILE_OPEN, OPEN_ACCESS_READONLY | OPEN_SHARE_DENYNONE, null);
   ::DosOpen("CON", &hwrite, &actionTaken, 0, FILE_NORMAL, FILE_OPEN, OPEN_ACCESS_WRITEONLY | OPEN_SHARE_DENYNONE, null);
   ::DosDupHandle(hread, &stdIn);
   ::DosDupHandle(hwrite, &stdOut);
   ::DosDupHandle(hwrite, &stdErr);
   ::DosClose(hread);
   ::DosClose(hwrite);
}

static void RestoreSavedStdStreams ( HFILE savedIn, HFILE savedOut, HFILE savedErr )
{
   HFILE stdIn = 0, stdOut = 1, stdErr = 2;
   ::DosDupHandle(savedIn, &stdIn);
   ::DosDupHandle(savedOut, &stdOut);
   ::DosDupHandle(savedErr, &stdErr);
   ::DosClose(savedIn);
   ::DosClose(savedOut);
   ::DosClose(savedErr);
}

GProcessLauncher::ErrorCode GProcessLauncher::getErrorCode () const
{
   return errorCode;
}

int GProcessLauncher::getExitCodeFromprocess () const
{
   return exitCodeFromProcess;
}

void GProcessLauncher::run ()
{
   errorCode = EC_NoError; // Until the opposite has been proven.
   startedAsChild = false;
   childStdIn = null;
   childStdOut = null;
   childStdErr = null;
   childPidToWaitFor = 0;

   if (GFile::ContainsExtension(prgName) &&
      !GFile::IsProgramFileName(prgName))
   {
      errorCode = EC_ErrNotAProgX; // GString("Not a program file: '%s'\n", GVArgs(prgName));
      return;
   }

   GString cmd = prgName;
   GProgram& prg = GProgram::GetProgram();
   const GEnvironment& env = prg.getEnvironmentVars();
   if ((prgName = env.which(cmd)) == "")
   {
      prgName = cmd;
      errorCode = EC_ErrPrgNotFoundX; // GString("Program file not found: '%s'\n", GVArgs(cmd));
      return;
   }

   // Let the sub-class get a chance to set "forceNewSession = true" in
   // case the program path or parameter must force it to be run in
   // a new session.
   programPathHasBeenValidated(prgName, paramStr);

   // ---
   GString comSpec = prg.getEnvironmentVar(COMSPEC, DEFCOMSPEC);
   if (GFile::IsShellScriptFileName(prgName))
   {
      // Use COMSPEC to execute Command/Batch files.
      paramStr = SysShellPrefixArgs + "\"\"" + prgName + "\" " + paramStr + "\"";
      prgName = env.which(comSpec);
   }

   // Activate the working directory of which is to be inherited by
   // the new process. Use the current directory if nothing else is
   // specified.
   errorCode = prepareTheWorkingDir();
   if (errorCode != EC_NoError)
      return;

   GVfsLocal localVfs;
   APIRET rc = NO_ERROR;
   char objBuf[GFile::MAXPATH] = "";

   ULONG appTypeFlags;
   rc = ::DosQueryAppType(prgName, &appTypeFlags);
   if (rc != NO_ERROR)
   {
      errorCode = EC_ErrNotAProgX; // GString("Not a program file: '%s'\n", GVArgs(prgName));
      return;
   }

   // Make sure only one thread enters here at a time.
   execSynch.enterSynchronizationLock();

   // ---
   int appType = (appTypeFlags & 0x7);
   if (forceNewSession)
      startedAsChild = false;
   else
   if (appType == FAPPTYP_WINDOWCOMPAT)
      startedAsChild = true;
   else
      startedAsChild = false; // PM, fullscreen VIO, DOS or Windows.

   // Child process doesn't need a copy of our own standard stream handles,
   // so make sure they are not inheritable.
   MakeSureStdStreamsAreNotInheritable();

   // -1 indicates that OS/2 should allocate a handle.
   HFILE stdIn = 0, stdOut = 1, stdErr = 2;
   HFILE savedStdIn = HFILE(-1), savedStdOut = HFILE(-1), savedStdErr = HFILE(-1);

   // The only way to assign standard streams to a new process in OS/2 is by
   // let it inherit the standrad streams of the process that starts it.
   // Thus, we must temporarily change our own standard streams so that the
   // new process inherits streams that it can use. But before doing that we
   // are better making a backup copy of our own standard stream handles so
   // that we can restore them as soon as the new process has been launched.
   ::DosDupHandle(stdIn, &savedStdIn);
   ::DosDupHandle(stdOut, &savedStdOut);
   ::DosDupHandle(stdErr, &savedStdErr);

   // Start the process (if we did not fail on activate
   // the "active directory").
   if (startedAsChild) // If we should start it as a child process.
   {
      // Build the environment variable buffer, which is to be
      // inherited by the program of which to launch.
      GString envStr = env.makeString('=', '\0', '\0', '\0');

      // Make up the program parameter list in the format
      // ===> "PROGNAME.EXE\0 param1 param2 param3\0\0"
      GString pgm = prgName + '\0' + (paramStr!=""?" ":"") + paramStr + '\0' + '\0';

      // Create the pipes. Note that on OS/2 these are "named pipes".
      HPIPE childsStdInPipe_read = null, childsStdInPipe_write = null;
      HPIPE childsStdOutPipe_read = null, childsStdOutPipe_write = null;
      HPIPE childsStdErrPipe_read = null, childsStdErrPipe_write = null;
      createPipe(&childsStdInPipe_read, &childsStdInPipe_write);
      createPipe(&childsStdOutPipe_read, &childsStdOutPipe_write);
      createPipe(&childsStdErrPipe_read, &childsStdErrPipe_write);

      // Create wrapper streams for the redirected STDIN, STDOUT & STDERR,
      // so that the subclass can read/write from/to the pipes in an easy way.
      childStdIn = new GFileOutputStream(localVfs, childsStdInPipe_write, true, false);
      childStdOut = new GPipeInputStream(childsStdOutPipe_read, true);
      childStdErr = new GPipeInputStream(childsStdErrPipe_read, true);

      // Create and set up the pipeReaderMediator semaphore.
      // This is used to check when child reads from its STDIN.
      ::DosCreateEventSem(null, &pipeReaderMediator, DC_SEM_SHARED, 0);
      ::DosSetNPipeSem(childsStdInPipe_write, HSEM(pipeReaderMediator), (ULONG) 1);

      // Redirect our own STDIN, SATDOUT, STDERR so that child process will inherit them.
      ::DosDupHandle(childsStdInPipe_read, &stdIn);
      ::DosDupHandle(childsStdOutPipe_write, &stdOut);
      ::DosDupHandle(childsStdErrPipe_write, &stdErr);

      // Start the child process, with redirected STDIN, STDOUT and STDERR.
      memset(&childRes, 0, sizeof(childRes));
      if (GLog::Filter(GLog::TEST))
         GLog::Log(this, "Start child program using DosExecPgm(): %s %s", GVArgs(prgName).add(paramStr));
      rc = ::DosExecPgm(objBuf, sizeof(objBuf), EXEC_ASYNCRESULT, pgm, envStr, &childRes, prgName);

      // Child is launched. Close the parents copy of those pipe handles that
      // only the child should have open. Make sure that no handles to the
      // write end of the STDOUT/STDERR pipes or to the read end of the STDIN
      // pipe are maintained in this process, or else the pipes will not
      // close when the child process exits and DosRead() will hang.
      ::DosClose(stdIn);
      ::DosClose(stdOut);
      ::DosClose(stdErr);

      // Restore the original (saved-) standard handles.
      RestoreSavedStdStreams(savedStdIn, savedStdOut, savedStdErr);

      // Exit the single thread protected area.
      execSynch.exitSynchronizationLock();

      // ---
      if (rc == NO_ERROR);
      {
         // Call virtual method so that sub-class can get to know
         // that we have launched a child process.
         childPidToWaitFor = childRes.codeTerminate; // The PID of the child.
         processHasBeenLaunched(true);

         // ---
         waitForTheChildProcessToFinish();
         ULONG whyExit = childRes.codeTerminate;
         ULONG codeResult = childRes.codeResult;
         bool normalExit = (whyExit == TC_EXIT); // True if the exit was normal.
         memset(&childRes, 0, sizeof(childRes)); // Just to clear the PID of the Child.

         // Call virtual method so that subclass can get to know
         // that the child process has finished.
         exitCodeFromProcess = codeResult;
         processHasFinished(true, codeResult, normalExit);
         childPidToWaitFor = 0;
      }

      // Close the "pipe reader mediator".
      if (pipeReaderMediator != null)
      {
         ::DosCloseEventSem(pipeReaderMediator);
         pipeReaderMediator = null;
      }

      // Close "other end" (our end) of the redirected pipes.
      ::DosClose(childsStdInPipe_read);
      ::DosClose(childsStdOutPipe_write);
      ::DosClose(childsStdErrPipe_write);
   }
   else
   {
      // Start the program as a new session.
      USHORT stype; // Session type.
      if (appTypeFlags & (FAPPTYP_WINDOWSREAL | FAPPTYP_WINDOWSPROT | FAPPTYP_WINDOWSPROT31))
         stype = SSF_TYPE_WINDOWEDVDM; // Win16
      else
      if (appTypeFlags & FAPPTYP_DOS)
         stype = SSF_TYPE_WINDOWEDVDM; // DOS
      else
      if (appType == FAPPTYP_WINDOWCOMPAT)
         stype = SSF_TYPE_WINDOWABLEVIO; // OS2 windowed VIO
      else
      if (appType == FAPPTYP_NOTWINDOWCOMPAT)
         stype = SSF_TYPE_FULLSCREEN; // OS2 fullscreen VIO
      else
      if (appType == FAPPTYP_WINDOWAPI)
         stype = SSF_TYPE_PM; // OS2 PM
      else
         stype = SSF_TYPE_DEFAULT; // Unknown. Will this ever happen?

      // Temporarily set our own standard streams to "CON" (the system
      // console) so that the new process inherits handles to the system
      // standard streams and not to the standard streams of ours.
      ActivateSystemDefaultStdStreams();

      // Start the application, inheriting our current working directory
      // and standard stream handles.
      bool pm = (stype == SSF_TYPE_PM);
      GString title = prgName + " " + paramStr;
      USHORT pgmCtrl = SSF_CONTROL_VISIBLE;
      if (!closeOnExit)
         pgmCtrl |= SSF_CONTROL_NOAUTOCLOSE;
      if (GLog::Filter(GLog::TEST))
         GLog::Log(this, "Start new session: %s", GVArgs(title));
      STARTDATA sd;
      memset(&sd, 0, sizeof(sd));
      sd.Length        = sizeof(sd);
      sd.PgmTitle      = pm ? null : PSZ(title.cstring());
      sd.PgmName       = PSZ(prgName.cstring());
      sd.PgmInputs     = PBYTE(paramStr.cstring());
      sd.Environment   = null; // Inherit environment from the shell
      sd.InheritOpt    = (USHORT) SSF_INHERTOPT_PARENT;
      sd.SessionType   = (USHORT) stype;
      sd.FgBg          = (USHORT) SSF_FGBG_FORE;
      sd.Related       = (USHORT) SSF_RELATED_INDEPENDENT;
      sd.TraceOpt      = (USHORT) SSF_TRACEOPT_NONE;
      sd.PgmControl    = pgmCtrl;
      sd.ObjectBuffer  = objBuf;
      sd.ObjectBuffLen = sizeof(objBuf);
      PID pid = 0; // Dummy, will not be set because we launch an "independent" new session
      ULONG sessionID = 0; // Ditto
      rc = ::DosStartSession(&sd, &sessionID, &pid);
      if (rc == ERROR_SMG_START_IN_BACKGROUND) // Not really an error in this case!
         rc == NO_ERROR;

      // Restore the original (saved-) standard handles.
      RestoreSavedStdStreams(savedStdIn, savedStdOut, savedStdErr);

      // Exit the single thread protected area.
      execSynch.exitSynchronizationLock();

      // Notify subclass that we have launched a new session.
      if (rc == NO_ERROR)
         processHasBeenLaunched(false);
   }

   // ---
   delete childStdIn;
   childStdIn = null;
   delete childStdOut;
   childStdOut = null;
   delete childStdErr;
   childStdErr = null;

   // ---
   if (rc != NO_ERROR)
   {
      nameOfMissingDll = objBuf;
      systemErrorCode = rc;
      if (objBuf[0])
         errorCode = EC_ErrDllXReqByPrgY; // GString("Error loading module '%s' required by program '%s'.\n", GVArgs(objBuf).add(prgName));
      else
         errorCode = EC_ErrLauchCmdXErrCodeY; // GString("Could not launch command '%s'. Error code: %d\n", GVArgs(prgName).add(rc));
   }
}
