/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/util/GKeyBag.h"
#include "glib/exceptions/GIllegalArgumentException.h"

GKeyBagImpl::Entry::Entry ( const GString& key, GObject* object, bool autoDestroyObj )
                   :key(key),
                    object(object),
                    autoDestroyObj(autoDestroyObj)
{
}

GKeyBagImpl::Entry::~Entry ()
{
   if (autoDestroyObj)
      delete object;
}

GKeyBagImpl::GKeyBagImpl ( int initial, int incremental, bool ignoreCase )
            :objTable(initial, incremental),
             ignoreCase(ignoreCase)
{
}

GKeyBagImpl::~GKeyBagImpl ()
{
   // Our super destructor will destroy all contained items.
}

bool GKeyBagImpl::isIgnoreCase () const 
{ 
   return ignoreCase; 
}

void GKeyBagImpl::enlarge () 
{ 
   objTable.enlarge(); 
}

int GKeyBagImpl::getCount () const 
{ 
   return objTable.getCount(); 
}

const GString& GKeyBagImpl::getKey ( int idx ) const 
{ 
   return objTable.get(idx).key; 
}

GObject* GKeyBagImpl::get ( const GString& key ) const
{
   int idx = findEntry(key, null);
   if (idx < 0)
      return null;
   else
      return objTable[idx].object;
}

bool GKeyBagImpl::put ( const GString& key, GObject* obj, bool autoDestroyObj )
{
   if (obj == null)
      gthrow_(GIllegalArgumentException("Null-objects not supported!"));
   int closestIdx;
   int idx = findEntry(key, &closestIdx);
   if (idx >= 0) // If the key is already contained in the bag
      return false; // All objects in the bag _must_ have its own unique key!

   // Store the new bag entry object
   idx = closestIdx;
   Entry* be = new Entry(key, obj, autoDestroyObj);
   objTable.insert(be, idx);
   return true;
}

void GKeyBagImpl::update ( int idx, GObject* obj, bool autoDestroyObj )
{
   if (obj == null)
      gthrow_(GIllegalArgumentException("Null-objects not supported!"));
   Entry& be = objTable[idx];
   if (be.autoDestroyObj)
      delete be.object;
   be.object = obj;
   be.autoDestroyObj = autoDestroyObj;
}

bool GKeyBagImpl::update ( const GString& key, GObject* obj, bool autoDestroyObj )
{
   if (obj == null)
      gthrow_(GIllegalArgumentException("Null-objects not supported!"));
   int idx = findEntry(key);
   if (idx < 0) // If the key isn't contained in the bag
      return false;
   update(idx, obj, autoDestroyObj);
   return true;
}

void GKeyBagImpl::removeIndexedItem ( int idx, bool destroy )
{
   objTable.remove(idx, 1, destroy);
}

bool GKeyBagImpl::remove ( const GString& key )
{
   int index = findEntry(key);
   if (index <= -1)
      return false;
   removeIndexedItem(index);
   return true;
}

void GKeyBagImpl::removeAll ()
{
   int num = getCount();
   objTable.remove(0, num, true);
}

bool GKeyBagImpl::containsKey ( const GString& key ) const 
{ 
   return findEntry(key) >= 0; 
}

int GKeyBagImpl::findEntry ( const GString& key, int* closestIdx ) const
{
   int count = objTable.getCount();
   int cmpRes = -1;
   int min = 0;
   int max = (count - 1);
   int cur = (count - 1) / 2;

   while (min <= max)
   {
      if (cur <= (count - 1))
      {
         Entry& entry = objTable[cur];
         if (ignoreCase)
            cmpRes = key.compareIgnoreCase(entry.key);
         else
            cmpRes = key.compare(entry.key);
      }
      else
         cmpRes = 1;

      if (cmpRes < 0)
      {
         max = cur - 1;
         cur = min + ((max - min) / 2);
      }
      else
      if (cmpRes > 0)
      {
         min = cur + 1;
         cur = max - ((max - min) / 2);
      }
      else
      {
         if (closestIdx != null)
            *closestIdx = cur;

         return cur;
      }
   }

   if (closestIdx != null)
   {
      if (cmpRes < 0)
         *closestIdx = cur;
      else
         *closestIdx = cur + 1;
   }

   return -1;
}
