/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/primitives/GInteger.h"
#include "glib/primitives/GDouble.h"
#include "glib/primitives/GCharacter.h"
#include "glib/primitives/GBoolean.h"
#include "glib/primitives/GVArgs.h"
#include "glib/util/GExpressionParser.h"
#include "glib/util/GLog.h"
#include "glib/util/GMath.h"
#include "glib/util/GTokenizer.h"
#include "glib/exceptions/GSyntaxErrorException.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/io/GIOException.h"
#include "glib/GThread.h"

const GString GExpressionParser::Token_lpar("(");
const GString GExpressionParser::Token_rpar(")");
const GString GExpressionParser::Token_plus("+");
const GString GExpressionParser::Token_minus("-");
const GString GExpressionParser::Token_percent("%");
const GString GExpressionParser::Token_pi("pi");
const GString GExpressionParser::Token_true("true");
const GString GExpressionParser::Token_false("false");

GExpressionParser::Operator::Operator ( const GAbstractTokenizer& tokenizer )
                            :idPrecedence(OP_NOOP),
                             numArgs(0),
                             lineNr(tokenizer.getCurModuleLineNr()),
                             clmnPos(tokenizer.getCurModuleColumn())
{
}

GExpressionParser::Operator::Operator ( const GAbstractTokenizer& tokenizer, OPERATOR_ID idPrecedence, int numArgs )
                            :idPrecedence(idPrecedence),
                             numArgs(numArgs),
                             lineNr(tokenizer.getCurModuleLineNr()),
                             clmnPos(tokenizer.getCurModuleColumn())
{
}

GExpressionParser::Operator::Operator ( const Operator& src )
                            :idPrecedence(src.idPrecedence),
                             numArgs(src.numArgs),
                             lineNr(src.lineNr),
                             clmnPos(src.clmnPos)
{
}

GExpressionParser::Operator::~Operator ()
{
}

int GExpressionParser::Operator::getLineNr () const 
{ 
   return lineNr; 
}

int GExpressionParser::Operator::getColumnPos () const 
{ 
   return clmnPos; 
}

GExpressionParser::Operator::OPERATOR_ID GExpressionParser::Operator::getIDPrecedence () const 
{ 
   return idPrecedence; 
}

int GExpressionParser::Operator::getNumArgs () const 
{ 
   return numArgs; 
}

void GExpressionParser::Operator::set ( OPERATOR_ID idPrecedence, int numArgs ) 
{ 
   this->idPrecedence = idPrecedence; 
   this->numArgs = numArgs; 
}

GString GExpressionParser::Operator::toString () const
{
   switch (idPrecedence)
   {
      case OP_ASSIGN: return "=";
      case OP_BOOLINVERSE: return "!";
      case OP_BITINVERSE: return "~";
      case OP_POWER: return "^";
      case OP_MULTIPLY: return "*";
      case OP_DIVIDE: return "/";
      case OP_BITAND: return "&";
      case OP_ADD: return "+";
      case OP_SUBTRACT: return "-";
      case OP_BITOR: return "|";
      case OP_EQUALS: return "==";
      case OP_UNLIKE: return "!=";
      case OP_LESSORLIKE: return "<=";
      case OP_LARGERORLIKE: return ">=";
      case OP_LESS: return "<";
      case OP_LARGER: return ">";
      case OP_BOOLAND: return "&&";
      case OP_BOOLOR: return "||";
      case OP_NOOP: return "";
      default: gthrow_(GIllegalStateException(GString("Unknown operator ID: %d", GVArgs(idPrecedence))));
   }
}

static int RvalTot = 0;
static int RvalRemain = 0;

GExpressionParser::RValue::RValue ( const GAbstractTokenizer& tokenizer )
                          :lineNr(tokenizer.getCurModuleLineNr()),
                           clmnPos(tokenizer.getCurModuleColumn()),
                           tokenizer(tokenizer)
{
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "New %s: Total=%d, Remaining=%d", GVArgs(typeid(*this).name()).add(RvalTot++).add(++RvalRemain));
}

GExpressionParser::RValue::~RValue ()
{
   if (GLog::Filter(GLog::DEBUG))
      GLog::Log(this, "Destroy %s: Remaining=%d", GVArgs(typeid(*this).name()).add(--RvalRemain));
}

int GExpressionParser::RValue::getLineNr () const 
{ 
   return lineNr; 
}

int GExpressionParser::RValue::getColumnPos () const 
{ 
   return clmnPos; 
}

GString GExpressionParser::RValue::toString () const 
{ 
   return getString(); 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValue::operate ( const Operator& op ) const
{
   switch (op.getIDPrecedence())
   {
      case Operator::OP_BOOLINVERSE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, !isTrue()));

      case Operator::OP_NOOP: 
         return aptr<GExpressionParser::RValue>(clone());

      default: 
         gthrow_(GSyntaxErrorException(GString("Operator \"%s\" not supported for type \"%s\".", GVArgs(op.toString()).add(getTypeName())), op.getColumnPos() - 1, op.getLineNr()));
   }
}

aptr<GExpressionParser::RValue> GExpressionParser::RValue::operate ( const Operator& op, const GExpressionParser::RValue& arg ) const
{
   switch (op.getIDPrecedence())
   {
      case Operator::OP_BOOLAND: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, isTrue() && arg.isTrue()));

      case Operator::OP_BOOLOR: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, isTrue() || arg.isTrue()));

      default: 
         gthrow_(GSyntaxErrorException(GString("Operator \"%s\" not supported for type \"%s\".", GVArgs(op.toString()).add(getTypeName())), op.getColumnPos() - 1, op.getLineNr()));
   }
}

aptr<GExpressionParser::RValue> GExpressionParser::RValue::allegate () const
{
   gthrow_(GSyntaxErrorException("Allegation not supported for type " + getTypeName(), getColumnPos() - 1, getLineNr()));
}

aptr<GExpressionParser::RValue> GExpressionParser::RValue::negate () const
{
   gthrow_(GSyntaxErrorException("Negation not supported for type " + getTypeName(), getColumnPos() - 1, getLineNr()));
}

GExpressionParser::RValueInteger::RValueInteger ( const GAbstractTokenizer& tokenizer, int val )
                                 :GExpressionParser::RValue(tokenizer),
                                  val(val)
{
}

GExpressionParser::RValueInteger::RValueInteger ( const GAbstractTokenizer& tokenizer, double val )
                                 :GExpressionParser::RValue(tokenizer),
                                  val(int(val))
{
}

GExpressionParser::RValueInteger::~RValueInteger ()
{
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueInteger::operate ( const Operator& op ) const
{
   switch (op.getIDPrecedence())
   {
      case Operator::OP_BITINVERSE: 
         return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, ~val));

      default: 
         return GExpressionParser::RValue::operate(op);
   }
}

GExpressionParser::RValue* GExpressionParser::RValueInteger::clone () const 
{ 
   return new RValueInteger(tokenizer, val); 
}

bool GExpressionParser::RValueInteger::isDouble () const 
{ 
   return false; 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueInteger::allegate () const 
{ 
   RValueInteger* i = new RValueInteger(tokenizer, +val);
   return aptr<RValue>(i); 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueInteger::negate () const 
{ 
   RValueInteger* i = new RValueInteger(tokenizer, -val);
   return aptr<RValue>(i); 
}

GString GExpressionParser::RValueInteger::getTypeName () const 
{ 
   return "INTEGER"; 
}

bool GExpressionParser::RValueInteger::isTrue () const 
{ 
   return val != 0; 
}

GString GExpressionParser::RValueInteger::getString () const 
{ 
   return GInteger::ToString(val); 
}

int GExpressionParser::RValueInteger::getInt () const 
{ 
   return val; 
}

double GExpressionParser::RValueInteger::getDouble () const 
{ 
   return val; 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueInteger::operate ( const Operator& op, const GExpressionParser::RValue& arg ) const
{
   if (arg.isDouble())
   {
      RValueDouble dval1 = RValueDouble(tokenizer, this->getDouble());
      RValueDouble dval2 = RValueDouble(tokenizer, arg.getDouble());
      return dval1.operate(op, dval2);
   }
   else
   {
      switch (op.getIDPrecedence())
      {
         case Operator::OP_EQUALS: 
            return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val == arg.getInt()));

         case Operator::OP_UNLIKE: 
            return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val != arg.getInt()));

         case Operator::OP_LESSORLIKE: 
            return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val <= arg.getInt()));

         case Operator::OP_LARGERORLIKE: 
            return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val >= arg.getInt()));

         case Operator::OP_LESS: 
            return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val < arg.getInt()));

         case Operator::OP_LARGER: 
            return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val > arg.getInt()));

         case Operator::OP_MULTIPLY: 
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, val * arg.getInt()));

         case Operator::OP_DIVIDE: 
         {
            double argVal = arg.getDouble();
            if (argVal == 0.0)
               gthrow_(GSyntaxErrorException("Divide by zero.", op.getColumnPos() - 1, op.getLineNr()));
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, getDouble() / argVal));
         }

         case Operator::OP_ADD: 
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, val + arg.getInt()));

         case Operator::OP_SUBTRACT: 
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, val - arg.getInt()));

         case Operator::OP_POWER: 
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Pow(val, arg.getInt())));

         default: 
            return GExpressionParser::RValue::operate(op, arg);
      }
   }
}

GExpressionParser::RValueDouble::RValueDouble ( const GAbstractTokenizer& tokenizer, double val )
                                :GExpressionParser::RValue(tokenizer),
                                 val(val)
{
}

GExpressionParser::RValueDouble::~RValueDouble ()
{
}

GExpressionParser::RValue* GExpressionParser::RValueDouble::clone () const 
{ 
   return new RValueDouble(tokenizer, val); 
}

bool GExpressionParser::RValueDouble::isDouble () const 
{ 
   return true; 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueDouble::allegate () const 
{ 
   RValueDouble* d = new RValueDouble(tokenizer, +val);
   return aptr<RValue>(d); 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueDouble::negate () const 
{ 
   RValueDouble* d = new RValueDouble(tokenizer, -val);
   return aptr<RValue>(d); 
}

GString GExpressionParser::RValueDouble::getTypeName () const 
{ 
   return "FLOAT"; 
}

bool GExpressionParser::RValueDouble::isTrue () const 
{ 
   return val != 0.0; 
}

GString GExpressionParser::RValueDouble::getString () const 
{ 
   return GDouble::ToString(val); 
}

int GExpressionParser::RValueDouble::getInt () const 
{ 
   return int(val); 
}

double GExpressionParser::RValueDouble::getDouble () const 
{ 
   return val; 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueDouble::operate ( const Operator& op ) const
{
   return GExpressionParser::RValue::operate(op);
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueDouble::operate ( const Operator& op, const GExpressionParser::RValue& arg ) const
{
   switch (op.getIDPrecedence())
   {
      case Operator::OP_EQUALS: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val == arg.getDouble()));

      case Operator::OP_UNLIKE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val != arg.getDouble()));

      case Operator::OP_LESSORLIKE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val <= arg.getDouble()));

      case Operator::OP_LARGERORLIKE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val >= arg.getDouble()));

      case Operator::OP_LESS: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val < arg.getDouble()));

      case Operator::OP_LARGER: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val > arg.getDouble()));

      case Operator::OP_MULTIPLY: 
         return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, val * arg.getDouble()));

      case Operator::OP_DIVIDE: 
      {
         double argVal = arg.getDouble();
         if (argVal == 0.0)
            gthrow_(GSyntaxErrorException("Divide by zero.", op.getColumnPos() - 1, op.getLineNr()));
         return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, val / argVal));
      }

      case Operator::OP_ADD: 
         return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, val + arg.getDouble()));

      case Operator::OP_SUBTRACT: 
         return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, val - arg.getDouble()));

      case Operator::OP_POWER: 
         return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Pow(val, arg.getDouble())));

      default: 
         return GExpressionParser::RValue::operate(op, arg);
   }
}

GExpressionParser::RValueBoolean::RValueBoolean ( const GAbstractTokenizer& tokenizer, bool val )
                                 :GExpressionParser::RValue(tokenizer),
                                  val(val)
{
}

GExpressionParser::RValueBoolean::~RValueBoolean ()
{
}

GExpressionParser::RValue* GExpressionParser::RValueBoolean::clone () const 
{ 
   return new RValueBoolean(tokenizer, val); 
}

bool GExpressionParser::RValueBoolean::isDouble () const 
{ 
   return false; 
}

GString GExpressionParser::RValueBoolean::getTypeName () const 
{ 
   return "BOOLEAN"; 
}

bool GExpressionParser::RValueBoolean::isTrue () const 
{ 
   return val; 
}

GString GExpressionParser::RValueBoolean::getString () const 
{ 
   return val ? GBoolean::TrueStr : GBoolean::FalseStr; 
}

int GExpressionParser::RValueBoolean::getInt () const 
{ 
   return val ? 1 : 0; 
}

double GExpressionParser::RValueBoolean::getDouble () const 
{ 
   return val ? 1.0 : 0.0; 
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueBoolean::operate ( const Operator& op ) const
{
   return GExpressionParser::RValue::operate(op);
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueBoolean::operate ( const Operator& op, const GExpressionParser::RValue& arg ) const
{
   switch (op.getIDPrecedence())
   {
      case Operator::OP_EQUALS: 
         return aptr<GExpressionParser::RValue> (new RValueBoolean(tokenizer, val == arg.isTrue()));

      case Operator::OP_UNLIKE: 
         return aptr<GExpressionParser::RValue> (new RValueBoolean(tokenizer, val != arg.isTrue()));

      default: 
         return GExpressionParser::RValue::operate(op, arg);
   }
}

GExpressionParser::RValueString::RValueString ( const GAbstractTokenizer& tokenizer, const GString& val )
                                :GExpressionParser::RValue(tokenizer),
                                 val(val)
{
}

GExpressionParser::RValueString::~RValueString ()
{
}

GExpressionParser::RValue* GExpressionParser::RValueString::clone () const 
{ 
   return new RValueString(tokenizer, val); 
}

GString GExpressionParser::RValueString::getTypeName () const 
{ 
   return "STRING"; 
}

bool GExpressionParser::RValueString::isTrue () const 
{ 
   return val != ""; 
}

GString GExpressionParser::RValueString::getString () const 
{ 
   return val; 
}

bool GExpressionParser::RValueString::isDouble () const
{
   if (val.indexOf('.') <= -1)
      return false;
   try {
      GDouble::ParseDouble(val);
      return true;
   } catch (GNumberFormatException& /*e*/) {
      return false;
   }
}

int GExpressionParser::RValueString::getInt () const
{
   try {
      return int(GDouble::ParseDouble(val));
   } catch (GNumberFormatException& /*e*/) {
      gthrow_(GSyntaxErrorException(getTypeName() + " value not possible to convert to INTEGER", getColumnPos() - 1, getLineNr()));
   }
}

double GExpressionParser::RValueString::getDouble () const
{
   try {
      return GDouble::ParseDouble(val);
   } catch (GNumberFormatException& /*e*/) {
      gthrow_(GSyntaxErrorException(getTypeName() + " value not possible to convert to FLOAT", getColumnPos() - 1, getLineNr()));
   }
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueString::operate ( const Operator& op ) const
{
   return GExpressionParser::RValue::operate(op);
}

aptr<GExpressionParser::RValue> GExpressionParser::RValueString::operate ( const Operator& op, const GExpressionParser::RValue& arg ) const
{
   switch (op.getIDPrecedence())
   {
      case Operator::OP_EQUALS: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val == arg.getString()));

      case Operator::OP_UNLIKE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val != arg.getString()));

      case Operator::OP_LESSORLIKE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val.compare(arg.getString()) <= 0));

      case Operator::OP_LARGERORLIKE: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val.compare(arg.getString()) >= 0));

      case Operator::OP_LESS: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val.compare(arg.getString()) < 0));

      case Operator::OP_LARGER: 
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, val.compare(arg.getString()) > 0));

      case Operator::OP_ADD:
         // A special one here:
         // "String1" + "String2" = "String1String2"
         return aptr<GExpressionParser::RValue>(new RValueString(tokenizer, val + arg.getString()));

      default: 
         return GExpressionParser::RValue::operate(op, arg);
   }
}

GExpressionParser::GExpressionParser ( GAbstractTokenizer& tokenizer )
                  :tokenizer(tokenizer)
{
}

GExpressionParser::~GExpressionParser ()
{
}

bool GExpressionParser::getOperator ( const GAbstractToken& token, Operator& op ) const
{
   if (token.isQuoted())
      return false;

   const int tokenLen = token.getLength();
   char char1 = (tokenLen >= 1 ? token.getCharacter(0) : ' ');
   char char2 = (tokenLen >= 2 ? token.getCharacter(1) : ' ');

   switch (char1)
   {
      case '=': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_ASSIGN, 1); return true;
                   case '=' : op.set(Operator::OP_EQUALS, 1); return true;
                }
                break;

      case '!': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_BOOLINVERSE, 0); return true;
                   case '=' : op.set(Operator::OP_UNLIKE, 1); return true;
                }
                break;

      case '<': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_LESS, 1); return true;
                   case '=' : op.set(Operator::OP_LESSORLIKE, 1); return true;
                }
                break;

      case '>': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_LARGER, 1); return true;
                   case '=' : op.set(Operator::OP_LARGERORLIKE, 1); return true;
                }
                break;

      case '&': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_BITAND, 1); return true;
                   case '&' : op.set(Operator::OP_BOOLAND, 1); return true;
                }
                break;

      case '|': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_BITOR, 1); return true;
                   case '|' : op.set(Operator::OP_BOOLOR, 1); return true;
                }
                break;

      case '+': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_ADD, 1); return true;
                }
                break;

      case '-': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_SUBTRACT, 1); return true;
                }
                break;

      case '*': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_MULTIPLY, 1); return true;
                }
                break;

      case '/': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_DIVIDE, 1); return true;
                }
                break;

      case '^': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_POWER, 1); return true;
                }
                break;

      case '~': switch (char2)
                {
                   case ' ' : op.set(Operator::OP_BITINVERSE, 0); return true;
                }
                break;
   }

   return false;
}

aptr<GExpressionParser::RValue> GExpressionParser::getRValue ()
{
   const GAbstractToken* token = tokenizer.getNextAbstractToken();
   if (token->isEmpty())
      gthrow_(GSyntaxErrorException("Unexpected end of stream.", tokenizer.getCurModuleColumn(), tokenizer.getCurModuleLineNr()));

   Operator op(tokenizer);
   if (getOperator(*token, op))
   {
      if (op.getNumArgs() == 0)
         return getRValue()->operate(op);
   }
   else
   {
      // Not an operator. That's OK.
   }

   // ---
   if (token->isQuoted())
      return aptr<GExpressionParser::RValue>(new RValueString(tokenizer, token->toString()));

   if (*token == Token_lpar)
      return parseSubExpression();

   if (*token == Token_plus)
      return getRValue()->allegate();

   if (*token == Token_minus)
      return getRValue()->negate();

   // At this point we can safely assume that the token is
   // either a symbol or a literal number.
   char firstChar = token->getCharacter(0);
   if (GCharacter::IsDigit(firstChar) || firstChar == '.')
   {
      // The token is a literal number. Supported formats are:
      // (Mark that all non-digit characters are case-insensitive.)
      //   11     ==> Decimal integer 11
      //   11.0   ==> Decimal float 11.0
      //   11.    ==> Decimal float 11.0
      //     .11  ==> Decimal float 0.11
      // 0h11     ==> Hex 11 (decimal 17)
      // 0x11     ==> Hex 11 (decimal 17)
      // 0o11     ==> Octal 11 (decimal 9)
      // 0b11     ==> Binary 11 (decimal 3)
      //   11t    ==> Decimal integer 11*1000
      //   11m    ==> Decimal integer 11*1000*1000
      //   11b    ==> Decimal integer 11*1000*1000*1000
      //   11.0t  ==> Decimal float 11.0*1000.0
      //   11.0m  ==> Decimal float 11.0*1000.0*1000.0
      //   11.0b  ==> Decimal float 11.0*1000.0*1000.0*1000.0
      //   11e10  ==> Decimal float 11.0*10^10
      GString tokenStr = token->toString();
      const int tokenLen = token->getLength();
      try {
         if (firstChar == '0' && tokenLen > 2 && token->getCharacter(1) != '.')
         {
            char secondChar = GCharacter::ToUpperCase(token->getCharacter(1));
            if (!GCharacter::IsDigit(secondChar))
            {
               switch (secondChar)
               {
                  case 'X': case 'H': return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GInteger::ParseInt(tokenStr.substring(2), 16)));
                  case 'O': return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GInteger::ParseInt(tokenStr.substring(2), 8)));
                  case 'B': return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GInteger::ParseInt(tokenStr.substring(2), 2)));
                  default: gthrow_(GNumberFormatException());
               }
            }
         }
         char lastChar = token->getCharacter(tokenLen - 1);
         int factor = 1;
         if (lastChar != '.' && !GCharacter::IsDigit(lastChar))
         {
            tokenStr = tokenStr.substring(0, tokenLen - 1); // Remove last character
            switch (GCharacter::ToUpperCase(lastChar))
            {
               case 'T': factor = 1000; break;
               case 'M': factor = 1000 * 1000; break;
               case 'B': factor = 1000 * 1000 * 1000; break;
               default: gthrow_(GNumberFormatException());
            }
         }
         bool isFloat = (tokenStr.indexOf('.') >= 0 || tokenStr.indexOf('E') >= 0 || tokenStr.indexOf('e') >= 0);
         if (isFloat) // If this is a literal of type DOUBLE
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GDouble::ParseDouble(tokenStr) * factor));
         else
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GInteger::ParseInt(tokenStr) * factor));
      } catch (GNumberFormatException& /*e*/) {
         gthrow_(GSyntaxErrorException("Not a number: " + tokenStr, tokenizer.getCurModuleColumn() - tokenLen, tokenizer.getCurModuleLineNr()));
      }
   }
   else
   {
      // The token is a symbol.
      if (*token == Token_true)
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, true));
      else
      if (*token == Token_false)
         return aptr<GExpressionParser::RValue>(new RValueBoolean(tokenizer, false));
      else
      if (*token == Token_pi)
         return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::PI));
      else
      {
         // If the next token is a "(" then we can assume that
         // this symbol is a function.
         int functionPosClmn = tokenizer.getCurModuleColumn() - token->getLength();
         int functionPosLine = tokenizer.getCurModuleLineNr();
         GString functionName = token->toString();
         token = tokenizer.getNextAbstractToken();
         if (*token != Token_lpar)
            gthrow_(GSyntaxErrorException(GString("Unknown symbol: %s", GVArgs(functionName)), functionPosClmn, functionPosLine));

         // ---
         aptr<GExpressionParser::RValue> farg = parseSubExpression();
         if (functionName == "abs")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Abs(farg->getDouble())));
         else
         if (functionName == "acos")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Acos(farg->getDouble())));
         else
         if (functionName == "asin")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Asin(farg->getDouble())));
         else
         if (functionName == "atan")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Atan(farg->getDouble())));
         else
         if (functionName == "ceil") // Integer >= Argument.
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GMath::Ceil(farg->getDouble())));
         else
         if (functionName == "cos")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Cos(farg->getDouble())));
         else
         if (functionName == "exp")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Exp(farg->getDouble())));
         else
         if (functionName == "floor") // Integer <= Argument.
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GMath::Floor(farg->getDouble())));
         else
         if (functionName == "log")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Log(farg->getDouble())));
         else
         if (functionName == "rand") // Random number in range 0...Argument-1
            return aptr<GExpressionParser::RValue>(new RValueInteger(tokenizer, GMath::Rand() % farg->getInt()));
         else
         if (functionName == "sin")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Sin(farg->getDouble())));
         else
         if (functionName == "sqrt")
         {
            double arg = farg->getDouble();
            if (arg >= 0.0)
               return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Sqrt(arg)));
            else
               gthrow_(GSyntaxErrorException("Argument of sqrt() must be >= 0.0", functionPosClmn + functionName.length(), functionPosLine));
         }
         else
         if (functionName == "tan")
            return aptr<GExpressionParser::RValue>(new RValueDouble(tokenizer, GMath::Tan(farg->getDouble())));
         else
            gthrow_(GSyntaxErrorException("Unknown function: " + functionName, functionPosClmn, functionPosLine));
      }
   }
}

aptr<GExpressionParser::RValue> GExpressionParser::parseSubExpression ()
{
   GArray<GExpressionParser::RValue> values(16);
   GArray<Operator> operators(16);

   // Scan the sub-expression until we reach its end.
   // The end is reached when there are no more tokens, or when
   // we find an ending pharantesis.
   for (;;)
   {
      aptr<GExpressionParser::RValue> rval = getRValue();
      RValue* value = rval->clone();
      values.add(value);
      // Get the next token, which can be either an operator or
      // a token to mark the end of the sub-expression.
      bool hasReachedPercent = false;
      const GAbstractToken* token = null;
      for (;;)
      {
         token = tokenizer.getNextAbstractToken();
         if (token->isEmpty())
            break;
         if (*token == Token_percent)
         {
            int num = values.getCount();
            if (hasReachedPercent || num <= 1)
               gthrow_(GSyntaxErrorException("Wrong usage of percent", tokenizer.getCurModuleColumn() - token->getLength(), tokenizer.getCurModuleLineNr()));
            hasReachedPercent = true;
            const RValue& prev = values[num-2];
            double prevVal = prev.getDouble();
            double percent = value->getDouble();
            double newValue = prevVal * percent / 100;
            value = new RValueDouble(tokenizer, newValue);
            values.set(num-1, value);
         }
         else
            break;
      }
      if (token->isEmpty() || *token == Token_rpar)
         break;
      Operator op(tokenizer);
      if (!getOperator(*token, op))
         gthrow_(GSyntaxErrorException(GString("Unknown operator: %s", GVArgs(token->toString())), op.getColumnPos() - token->getLength(), op.getLineNr()));
      if (op.getNumArgs() != 1)
         gthrow_(GSyntaxErrorException("Syntax error", tokenizer.getCurModuleColumn() - token->getLength(), tokenizer.getCurModuleLineNr()));
      operators.add(new Operator(op));
   }

   if (values.getCount() <= 0)
      gthrow_(GSyntaxErrorException("No expression", tokenizer.getCurModuleColumn(), tokenizer.getCurModuleLineNr()));

   // Analyze the values and the operators,
   // respecting operator precedence.
   for (;;)
   {
      const int numOperators = operators.getCount();
      if (numOperators <= 0)
      {
         GExpressionParser::RValue* ret = &values[0];
         values.remove(0, 1, false); // Remove the rvalue from the array, without destroying it.
         return aptr<GExpressionParser::RValue>(ret);
      }

      int heighestPrecedence = 0;
      int idxOfHeighestPrecedence = -1;
      for (int i=0; i<numOperators; i++)
      {
         const Operator& op = operators[i];
         if (op.getIDPrecedence() > heighestPrecedence)
         {
            heighestPrecedence = op.getIDPrecedence();
            idxOfHeighestPrecedence = i;
         }
      }

      try {
         const Operator& op = operators[idxOfHeighestPrecedence];
         const GExpressionParser::RValue& val1 = values[idxOfHeighestPrecedence];
         const GExpressionParser::RValue& val2 = values[idxOfHeighestPrecedence+1];
         aptr<GExpressionParser::RValue> res = val1.operate(op, val2);
         operators.remove(idxOfHeighestPrecedence);
         values.remove(idxOfHeighestPrecedence+1);
         values.set(idxOfHeighestPrecedence, res.release());
      } catch (GArrayIndexOutOfBoundsException& /*e*/) {
         gthrow_(GSyntaxErrorException("Missing RValue", tokenizer.getCurModuleColumn(), tokenizer.getCurModuleLineNr()));
      }
   }
}

aptr<GExpressionParser::RValue> GExpressionParser::getParsedExpression ()
{
   // Do the calculation in a block that is protected by a system
   // specific exception handler. That is in case of any floating
   // point exception (such as float-overflow or divide-by-zero)
   // during the calculation, which should not cause the program
   // to crash.
   GTRY(e, true) {
      return parseSubExpression();
   } GCATCH(e) {
      gthrow_(GSyntaxErrorException("Runtime exception. Probably a floating point error, such as value-overflow or divide-by-zero.", tokenizer.getCurModuleColumn(), tokenizer.getCurModuleLineNr()));
   } GENDCATCH();
}

aptr<GExpressionParser::RValue> GExpressionParser::GetParsedExpression ( const GString& expr )
{
   GTokenizer tokenizer(expr, "+-*/^%=!()[]<>&|~", false);
   GArray<GString> symbols(6);
   symbols.add(new GString("&&"));
   symbols.add(new GString("||"));
   symbols.add(new GString("=="));
   symbols.add(new GString("!="));
   symbols.add(new GString("<="));
   symbols.add(new GString(">="));
   tokenizer.setSpecialSymbols(symbols);
   GExpressionParser parser(tokenizer);
   try {
      return parser.getParsedExpression();
   } catch (GIOException& e) {
      // This exception will probably never occur here because the
      // string tokenizer is an in-memory input stream that should
      // never fail except at end-of-stream. But in case...
      gthrow_(GSyntaxErrorException(e.toString()));
   }
}
