/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <stdio.h>
#include <ctype.h>
#include "glib/primitives/GCharacter.h"
#include "glib/util/GDriveInfo.h"
#include "glib/util/GMath.h"
#include "glib/util/GLog.h"
#include "glib/vfs/GFile.h"

GDriveInfo::GDriveInfo ()
{
   clear();
}

void GDriveInfo::clear ()
{
   drive = 0;
   isAvailable = false;
   type = DT_UNKNOWN;
   volumeName = "";
   fsName = "";
   serialNo = 0;
   bytesPerCluster = 0;
   bytesPerSector = 0;
   numSectorsPerUnit = 0;
   numUnits = 0;
   numAvailUnits = 0;
   diskSize = 0;
   diskFree = 0;
}

APIRET GDriveInfo::update ( int drive )
{
   clear();

   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET& e) {
         return e;
      }
   }

   this->drive = drive;

   DosError(FERR_DISABLEHARDERR);

   APIRET ret = NO_ERROR;
   ULONG ulLen = 2048;
   const char *buf = new char[ulLen];
   FSQBUFFER2* bufPtr = (FSQBUFFER2*) buf;
   GString deviceName("%c:", GVArgs(char(drive + 'A' - 1)));
   APIRET apirc = DosQueryFSAttach(deviceName, 0, FSAIL_QUERYNAME, bufPtr, &ulLen);
   if (apirc == NO_ERROR)
   {
      FSQBUFFER2 *p2 = (FSQBUFFER2 *) buf;
      char fileSystemName[64];
      memcpy(fileSystemName, p2->szName + p2->cbName + 1, GMath::Min(int(p2->cbFSDName), int(sizeof(fileSystemName))));
      fileSystemName[GMath::Min(int(p2->cbFSDName), int(sizeof(fileSystemName)-1))] = '\0';
      this->fsName = fileSystemName;

      switch (p2->iType)
      {
         case FSAT_CHARDEV:   // Resident character device
              this->type = DT_FLOPPY;
              break;

         case FSAT_PSEUDODEV: // Pusedu-character device
              if (IsMediaRemoveable(drive))
                 this->type = DT_REMOVABLE;
              else
                 this->type = DT_RAMDRIVE;
              break;

         case FSAT_LOCALDRV:  // Local drive
              if (this->fsName == "CDFS")
                 this->type = DT_CDROM;
              else
              if (IsMediaRemoveable(drive))
                 this->type = DT_FLOPPY;
              else
                 this->type = DT_HARDDISK;
              break;

         case FSAT_REMOTEDRV: // Remote drive attached to FSD
              if (this->fsName == "RAMFS")
                 this->type = DT_RAMDRIVE;
              else
                 this->type = DT_NETWORK;
              break;
      }

      FSINFO FSInfo;
      DosError(FERR_DISABLEHARDERR);
      apirc = DosQueryFSInfo(drive, FSIL_VOLSER, &FSInfo, sizeof(FSInfo));
      if (apirc == NO_ERROR)
      {
         ULONG* date = (ULONG*) &FSInfo.fdateCreation;
         ULONG* time = (ULONG*) &FSInfo.ftimeCreation;
         this->serialNo = (*time << 16) | *date;
         this->volumeName = FSInfo.vol.szVolLabel;
         if (this->volumeName == "OS2VDISK")
            this->type = DT_RAMDRIVE;

         FSALLOCATE fsinf;
         DosError(FERR_DISABLEHARDERR);
         apirc = DosQueryFSInfo(drive, FSIL_ALLOC, (char*) &fsinf, sizeof(fsinf));
         if (apirc == NO_ERROR)
         {
            this->isAvailable = true;
            this->bytesPerSector = fsinf.cbSector;
            this->numSectorsPerUnit = fsinf.cSectorUnit;
            this->numUnits = fsinf.cUnit;
            this->numAvailUnits = fsinf.cUnitAvail;
            this->diskSize = (double) this->bytesPerSector * (double) this->numSectorsPerUnit * (double) this->numUnits;
            this->diskFree = (double) this->bytesPerSector * (double) this->numSectorsPerUnit * (double) this->numAvailUnits;

            if (this->fsName == "FAT")
               this->bytesPerCluster = unsigned(this->diskSize / 0xFFFF);
            else
            if (this->fsName == "HPFS")
               this->bytesPerCluster = 512;
            else
               this->bytesPerCluster = 512; // TODO: ???
         }
      }
   }
   else
   if (IsACdRomDrive(drive))
   {
      this->type = DT_CDROM;
      ret = NO_ERROR;
   }
   else
   if (apirc == ERROR_NOT_READY /* || apirc == ERROR_INVALID_DRIVE || apirc == ERROR_BAD_UNIT*/)
   {
      this->type = DT_FLOPPY; // Assume floppy with no drive in it
   }
   else
   {
      ret = apirc;
   }

   delete[] buf;
   return ret;
}

int GDriveInfo::GetBytesPerSector ( int drive )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET& /*e*/) {
         return 0;
      }
   }

   FSALLOCATE fsinf;
   ::DosError(FERR_DISABLEHARDERR);
   if (::DosQueryFSInfo(drive, FSIL_ALLOC, (char*) &fsinf, sizeof(fsinf)) != NO_ERROR)
      return 0;
   return fsinf.cbSector;
}

int GDriveInfo::GetMapOfAvailableDrives ()
{
   ULONG map;
   ULONG curDisk;
   ::DosError(FERR_DISABLEHARDERR);
   ::DosQueryCurrentDisk(&curDisk, &map);
   return int(map);
}

bool GDriveInfo::IsAValidDriveLetter ( char cDrive )
{
   char workDrive = GCharacter::ToUpperCase(cDrive);
   int logDriveMap = GDriveInfo::GetMapOfAvailableDrives();
   for (int i=0; i<25; i++)
   {
      if ((logDriveMap << (31 - i)) >> 31)
      {
         char testDrive = (char) (i + 65);
         if (testDrive == workDrive && testDrive != 'B')
            return true;
      }
   }

   return false;
}

bool GDriveInfo::IsMediaRemoveable ( int drive )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET) {
         return false;
      }
   }

   #pragma pack(1)
   typedef struct _QRYINP
   {
      BYTE CmdInfo;
      BYTE DriveUnit;
   }
      QRYINP, *PQRYINP;
   #pragma pack()

   QRYINP QueryInfo;
   BIOSPARAMETERBLOCK DiskData;
   ULONG ulParm;
   ULONG ulData;
   QueryInfo.CmdInfo = 0;
   QueryInfo.DriveUnit = (BYTE) (drive - 1);
   DosError (FERR_DISABLEHARDERR);
   APIRET apirc = DosDevIOCtl((HFILE)  -1,
                              (ULONG)  IOCTL_DISK,
                              (ULONG)  DSK_GETDEVICEPARAMS,
                              (void*)  &QueryInfo,
                              (ULONG)  sizeof(QRYINP),
                              (PULONG) &ulParm,
                              (void*)  &DiskData,
                              (ULONG)  sizeof(BIOSPARAMETERBLOCK),
                              (PULONG) &ulData);
   if (apirc != NO_ERROR)
      return false;
   else
   if (!(DiskData.fsDeviceAttr & 0x0001))
      return true;
   else
      return false;
}

bool GDriveInfo::IsMediaInDrive ( int drive )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET) {
         return false;
      }
   }

   FSINFO FSInfo;
   DosError (FERR_DISABLEHARDERR);
   APIRET apiReturn = DosQueryFSInfo(drive, FSIL_VOLSER, (char *) &FSInfo, sizeof(FSInfo));
   return apiReturn == NO_ERROR;
}

bool GDriveInfo::IsANetworkDrive ( int drive )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET) {
         return false;
      }
   }

   char szName[32];
   bool ret = false;
   ULONG ulLen = 4096;
   char *buf = new char[ulLen];
   DosError(FERR_DISABLEHARDERR);
   sprintf(szName, "%c:", drive + 'A' - 1);
   APIRET apirc = DosQueryFSAttach(szName, 0, FSAIL_QUERYNAME, (FSQBUFFER2*) buf, &ulLen);
   if (apirc == NO_ERROR)
   {
      FSQBUFFER2 *p2 = (FSQBUFFER2 *) buf;
      ret = (p2->iType == FSAT_REMOTEDRV);
   }
   delete[] buf;
   return ret;
}

bool GDriveInfo::IsACdRomDrive ( int drive )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET) {
         return false;
      }
   }

   #pragma pack(1)
   struct {
       UCHAR ucCommand;
       UCHAR ucDrive;
   } parms;
   #pragma pack()

   parms.ucCommand = 0; // 0 = return standard media
   parms.ucDrive = (UCHAR) (drive-1);

   BIOSPARAMETERBLOCK pb;
   memset(&pb, 0, sizeof(pb));
   ULONG cbParams = sizeof(parms);
   ULONG cbData = sizeof(pb);
   APIRET rc = DosDevIOCtl((HFILE) -1, IOCTL_DISK, DSK_GETDEVICEPARAMS,
                           &parms, cbParams, &cbParams,
                           &pb, cbData, &cbData);

   if (rc != NO_ERROR)
      return false;
   else
   if (pb.bDeviceType == 7 && // "other"
       pb.usBytesPerSector == 2048 &&
       pb.usSectorsPerTrack == (USHORT) -1)
      return true;
   else
      return false;
}

APIRET GDriveInfo::GetDriveLabel ( int drive, GString& label )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET& e) {
         return e;
      }
   }

   FSINFO FSInfo;
   DosError(FERR_DISABLEHARDERR);
   APIRET rc = DosQueryFSInfo(drive, FSIL_VOLSER, &FSInfo, sizeof(FSInfo));
   if (rc != NO_ERROR)
      label = "";
   else
      label = FSInfo.vol.szVolLabel;
   return rc;
}

unsigned int GDriveInfo::GetDriveSerialNo ( int drive )
{
   if (drive == 0)
   {
      try {
         drive = GFile::GetCurrentDrive();
      } catch (APIRET) {
         return 0;
      }
   }

   FSINFO FSInfo;
   DosError(FERR_DISABLEHARDERR);
   APIRET rc = DosQueryFSInfo(drive, FSIL_VOLSER, &FSInfo, sizeof(FSInfo));
   if (rc != NO_ERROR)
      return 0;
   ULONG* date = (ULONG*) &FSInfo.fdateCreation;
   ULONG* time = (ULONG*) &FSInfo.ftimeCreation;
   ULONG ret = (*time << 16) | *date;
   return ret;
}
