/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <memory.h>
#include "glib/util/GArray.h"
#include "glib/primitives/GInteger.h"
#include "glib/exceptions/GIllegalArgumentException.h"

GArrayImpl::Item::Item ( GObject* data, bool autoDelete )
                 :data(data),
                  autoDelete(autoDelete)
{
}

GArrayImpl::Item::~Item ()
{
   if (autoDelete)
      delete data;
}

GArrayImpl::GArrayImpl ( int initial, int incremental )
           :num(0),
            size(initial > 0 ? initial : 1),
            initial(initial),
            incremental(incremental),
            items(null)
{
   items = new Item*[size];
   memset(items, 0, size * sizeof(Item*));
}

GArrayImpl::~GArrayImpl ()
{
   removeAll();
   delete [] items;
}

int GArrayImpl::getAllocatedSize () const
{ 
   return size; 
}

int GArrayImpl::getCount () const 
{ 
   return num; 
}

bool GArrayImpl::isEmpty () const 
{ 
   return num <= 0; 
}

void GArrayImpl::enlarge ()
{
   int newsize;
   if (incremental < 0)
      newsize = size + (size / ((-1) * incremental)) + 1;
   else
   if (incremental == 0)
      newsize = size + 64;
   else
      newsize = size + incremental;
   ensureCapacity(newsize);
}

void GArrayImpl::ensureCapacity ( int count )
{
   if (count > size)
   {
      int oldsize = size;
      size = count;
      Item** tmp = new Item*[size];
      memcpy(tmp, items, oldsize * sizeof(Item*));
      delete [] items;
      items = tmp;
   }
}

int GArrayImpl::indexOf ( const GObject& anItem ) const
{
   for (int i=0; i<num; i++)
      if (items[i]->data == &anItem)
         return i;
   return -1;
}

bool GArrayImpl::isAutoDeleteAt ( int index ) const
{
   if (index < 0 || index >= getCount())
      gthrow_(GArrayIndexOutOfBoundsException("index=" + GInteger::ToString(index)));
   return items[index]->autoDelete;
}

void GArrayImpl::add ( GObject* anItem, bool autoDelete )
{
   if (anItem == null)
      gthrow_(GIllegalArgumentException("Null-objects not supported!"));
   if (num >= size)
      enlarge();
   items[num++] = new Item(anItem, autoDelete);
}

void GArrayImpl::insert ( GObject* anItem, int index, bool autoDelete )
{
   if (anItem == null)
      gthrow_(GIllegalArgumentException("Null-objects not supported!"));
   if (num >= size)
      enlarge();
   if (index < num) // If there are any tail to pull at all
      for (int i=num; i>index; i--)
         items[i] = items[i-1];
   items[index] = new Item(anItem, autoDelete);
   num++;
}

void GArrayImpl::remove ( int index, int count, bool destroy )
{
   if (count <= 0)
      return;

   if (index + count > num)
   {
      count = num - index;
      if (count <= 0)
         return;
   }

   for (int i=count-1; i>=0; i--)
   {
      Item* itm = items[index+i];
      if (!destroy)
         itm->autoDelete = false;
      delete itm;
   }

   // Displace the tail to close the gap from the removed items.
   for (int i=index+count; i<num; i++)
      items[i-count] = items[i];

   int prevNum = num;
   num -= count;

   // Set the removed tail of the items array to null.
   for (int i=num; i<prevNum; i++)
      items[i] = null;
}

void GArrayImpl::removeAll ()
{
   remove(0, num, true);
   num = 0;
}

void GArrayImpl::swap ( int idx1, int idx2 )
{
   Item* tmp = items[idx1];
   items[idx1] = items[idx2];
   items[idx2] = tmp;
}

void GArrayImpl::qsort ( int lo0, 
                         int hi0, 
                         const GComparator& comparator, 
                         bool ascending )
{
   int low = lo0;
   int high = hi0;
   if (hi0 > lo0)
   {
      // Arbitrarily establishing partition element as the midpoint of
      // the array.
      Item* mid = items[(lo0 + hi0) / 2];

      // loop through the array until indices cross
      while (low <= high)
      {
         // find the first element that is greater than or equal to
         // the partition element starting from the left Index.
         while ((low < hi0) && (qsort_compare(comparator, *items[low]->data, *mid->data, ascending) < 0))
            ++low;

         // find an element that is smaller than or equal to
         // the partition element starting from the right Index.
         while ((high > lo0) && (qsort_compare(comparator, *items[high]->data, *mid->data, ascending) > 0))
            --high;

         // if the indexes have not crossed, swap
         if (low <= high)
            swap(low++, high--);
      }

      // If the right index has not reached the left side of array
      // must now sort the left partition.
      if (lo0 < high)
         qsort(lo0, high, comparator, ascending);

      // If the left index has not reached the right side of array
      // must now sort the right partition.
      if (low < hi0)
         qsort(low, hi0, comparator, ascending);
   }
}

void GArrayImpl::sortItems ( const GComparator& comparator, bool ascending ) 
{
   qsort(0, num - 1, comparator, ascending);
}

int GArrayImpl::compare2Objects ( const GObject& obj1, const GObject& obj2 ) const
{
   return obj1.compareObj(obj2);
}

void GArrayImpl::sortItems ( bool ascending ) 
{
   sortItems(*this, ascending);
}

