/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_SYSTEMINFO
#define __GLIB_SYSTEMINFO

#include "glib/sys/GSystem.h"
#include "glib/util/GHashtable.h"
#include "glib/primitives/GInteger.h"

/**
 * Class used to store information about a single process.
 *
 * @author  Leif Erik Larsen
 * @since   2004.07.26
 */
class GProcessInfo : public GObject
{
   public:
      
      const GSystem::ProcessID pid;
      const GSystem::ProcessID ppid;
      
   public:
      
      GProcessInfo ( GSystem::ProcessID pid, 
                     GSystem::ProcessID ppid );

      virtual ~GProcessInfo ();

   public:
      
      bool isDescendant ( GSystem::ProcessID ancestorPid, 
                          GHashtable<GInteger, GProcessInfo>& all );
};

/**
 * Class used to specify an amount of time.
 */
class GTimeAmount : public GObject
{
   public:

      int numDays;
      int numHours;
      int numMinutes;
      int numSeconds;
      int numMillis;

   public:

      GTimeAmount ();
      virtual ~GTimeAmount ();
};

/**
 * Class used to provide various information about the Processes, Threads
 * Modules, Semaphores, Shared Memory, RAM and Uptime that are active on
 * the system.
 */
class GSystemInfo : public GObject
{
   private:

      /**
       * Information about processes that runs on the system.
       * The kays of the hashtable are instances of GInteger that 
       * contains the pid of the corresponding process information.
       *
       * @author  Leif Erik Larsen
       * @since   2004.07.26
       */
      GHashtable<GInteger, GProcessInfo> processes;

      int numThreads;

      int systemStateBufferSize;
      char* systemStateBuffer;

      GTimeAmount uptime; // Time that the system has been up and running
      ulonglong physicalRAM;
      ulonglong freePhysicalRAM;

   public:

      GSystemInfo ();
      virtual ~GSystemInfo ();

   private:

      /** Disable the copy constructor. */
      GSystemInfo ( const GSystemInfo& src ) {}

      /** Disable the assignment operator. */
      GSystemInfo& operator= ( const GSystemInfo& ) { return *this; }

   public:

      void update ();
      int getProcessCount ();
      GHashtable<GInteger, GProcessInfo>& getProcessList ();
      int getThreadCount ();
      const GTimeAmount& getUpTime ();
      ulonglong getPhysicalRAM ();
      ulonglong getFreePhysicalRAM ();
};

#endif
