/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GMenuItemParams.h"
#include "glib/resource/GTextResource.h"
#include "glib/resource/GResourceTable.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"

GMenuItemParams::GMenuItemParams ( const GString& id )
                :GAbstractCommand(id),
                 textIsDef(false),
                 hintIsDef(false),
                 iconIsDef(false),
                 keyIsDef(false),
                 userDataIsDef(false),
                 iHide(false),
                 iLock(false),
                 iType(ITEM)
{
}

GMenuItemParams::GMenuItemParams ()
                :GAbstractCommand(),
                 textIsDef(false),
                 hintIsDef(false),
                 iconIsDef(false),
                 keyIsDef(false),
                 userDataIsDef(false),
                 iHide(false),
                 iLock(false),
                 iType(ITEM)
{
}

GMenuItemParams::~GMenuItemParams ()
{
}

void GMenuItemParams::loadFromScript ( GRcTokenizer& tokenizer, GResourceTable* /*table*/ )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lpar) // Start of menuitem-parameters block.
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);

      if (*token == GRcTokenizer::Token_rpar)
      {
         break;
      }

      else
      if (*token == GRcTokenizer::Token_comma)
      {
         continue;
      }

      else
      if (*token == GRcTokenizer::Token_id)
      {
         if (getIDString() != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRIDARDEF)); // ID already defined!

         token = tokenizer.queryArgValue();

         if (iType == SEPARATOR)
            gthrow_(GRcException(tokenizer, GRcException::ERRIDINSEPRT)); // Can't define an ID to a SEPARATOR!

         setIDString(token->getString());
      }

      else
      if (*token == GRcTokenizer::Token_text)
      {
         if (textIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRTEXTARDEF)); // TEXT already defined!

         token = tokenizer.queryArgValue();
         textID = token->getString();
         textIsDef = true;
      }

      else
      if (*token == GRcTokenizer::Token_hint)
      {
         if (hintIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRHINTARDEF)); // HINT already defined!

         token = tokenizer.queryArgValue();
         hintID = token->getString();
         hintIsDef = true;
      }

      else
      if (*token == GRcTokenizer::Token_icon)
      {
         if (iconIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRICONARDEF)); // ICON already defined!

         token = tokenizer.queryArgValue();
         iconID = token->getString();
         iconIsDef = true;
      }

      else
      if (*token == GRcTokenizer::Token_key)
      {
         if (keyIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRKEYARDEF)); // KEY already defined!

         token = tokenizer.queryArgValue();

         if (iType != ITEM) // Only ITEM can be associated to function keys.
            gthrow_(GRcException(tokenizer, GRcException::ERRKEYINPOPUP)); // Can't define shortcut to POPUP/SEPARATOR!

         accelKeyID = token->getString();
         keyIsDef = true;
      }

      else
      if (*token == GRcTokenizer::Token_userdata1)
      {
         if (userDataIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRUSERDATA1ARDEF)); // Userdata1 already defined!

         token = tokenizer.queryArgValue();
         setUserData1(token->getString());
         userDataIsDef = true;
      }

      else
      if (*token == GRcTokenizer::Token_invisible)
      {
         if (iType == SEPARATOR)
            gthrow_(GRcException(tokenizer, GRcException::ERRINVISBLSEP)); // Can't define a SEPARATOR as INVISIBLE!

         iHide = true;
      }

      else
      if (*token == GRcTokenizer::Token_disabled)
      {
         if (iType == SEPARATOR)
            gthrow_(GRcException(tokenizer, GRcException::ERRDISABLDSEP)); // Can't define a SEPARATOR as DISABLED!

         iLock = true;
      }

      else
         gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(token->getString()))); // Unknown token: %s
   }

   if (iType != SEPARATOR)  // If POPUP or ITEM.
   {
      const GString& id = getIDString();
      if (iType == ITEM && id == "")
         gthrow_(GRcException(tokenizer, GRcException::ERRITEMMISSID)); // No ID specified for this ITEM!

      if (textID == "")
      {
         if (hintID != "" || id == "")
            gthrow_(GRcException(tokenizer, GRcException::ERRITEMMISSTEXT)); // No TEXT specified for this POPUP/ITEM!

         // This item is defined without TEXT and without HINT.
         // Such items should try to fetch TEXT and HINT from the
         // resource table by default.
         textID += '%'; // Make this a reference to the other TEXT.
         textID += id;
      }
   }
}

