/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ICONRESOURCE
#define __GLIB_ICONRESOURCE

#include "glib/resource/GAbstractResource.h"
#include "glib/util/GVector.h"
#include "glib/gui/GColor.h"
#include "glib/gui/GIcon.h"

/**
 * This class is used to represent a single icon resource in memory.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 * @see     #drawTheIcon
 * @see     GGraphics#drawIcon
 */
class GIconResource : public GAbstractResource
{
   private:

      /** Number of horizontal pixels in icon. */
      int width;

      /** Number of vertical pixels in icon. */
      int height;

      /** Number of bits per pixel in icon (4=16 colors, 8=256 colors, 0=True RGB-colors). */
      int colorBits;

      /** The color palette is used in 16 and 256 color icons only. Else it is empty. */
      GVector<int> palette;

      /** Icon pixels in RGB colors. Pixels with value == -1 means "transparent". */
      GVector<int> pixels;

   public:

      GIconResource ();
      explicit GIconResource ( const GString& id );
      explicit GIconResource ( const GIconResource& id );
      virtual ~GIconResource ();

   public:

      GIconResource& operator= ( const GIconResource& src );
      bool operator== ( const GIconResource& src ) const;
      bool operator!= ( const GIconResource& src ) const;

   private:

      /**
       * Parse the specified string and get it as an RGB color value.
       *
       * The specified string must be a six digit hexadecimal representation of
       * the color, in the format "RRGGBB" where RR is the red value (in hex),
       * GG is the green value (in hex) and BB is the blue value (in hex).
       *
       * @author  Leif Erik Larsen
       * @since   2001.03.06
       * @return  The RGB color representation of the specified string, or
       *          -1 in case of any error (unsupported string format).
       */
      static int MakeColorFromString ( const GString& str );

   public:

      int getColorBits () const;
      int getHeight () const;
      int getWidth () const;
      const GVector<int>& getPalette () const;
      const GVector<int>& getPixels () const;
      int getPixelAt ( int x, int y ) const;
      void setPixelAt ( int x, int y, int pixelValue );

      /**
       * Load and compile the following resource definition from the
       * specified tokenizer for the calling resource compiler.
       *
       * Since we assume that the ID-part of the resource has already been
       * parsed the next token (which is the first, from our point of view)
       * returned from the tokenizer must be "(".
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.08
       * @param   tokenizer   The tokenizer of where to get tokens to parse.
       * @throws  GRcException in case of any error.
       */
      virtual void loadFromScript ( class GRcTokenizer& tokenizer, class GResourceTable* table );
};

#endif


