/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GComponentParams.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"
#include "glib/io/GStringStream.h"

GComponentParams::GComponentParams ()
                 :GAbstractResource(""),
                  disabled(false),
                  toggleOn(false),
                  hidden(false),
                  oily(false),
                  extraArgs(true)
{
}

GComponentParams::GComponentParams ( const GString& script )
                 :GAbstractResource(""),
                  disabled(false),
                  toggleOn(false),
                  hidden(false),
                  oily(false),
                  extraArgs(true)
{
   GStringStream stream(script);
   GRcTokenizer tokenizer(stream);
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   typeName = token->getString();
   loadFromScript(tokenizer, null);
}

GComponentParams::GComponentParams ( const GString& typeName, 
                                     const GString& id )
                 :GAbstractResource(id),
                  typeName(typeName),
                  disabled(false),
                  toggleOn(false),
                  hidden(false),
                  oily(false),
                  extraArgs(true)
{
}

GComponentParams::GComponentParams ( const GComponentParams& src )
                 :GAbstractResource(src),
                  typeName(src.typeName),
                  textID(src.textID),
                  hintID(src.hintID),
                  constraints(src.constraints),
                  pos(src.pos),
                  size(src.size),
                  disabled(src.disabled),
                  toggleOn(src.toggleOn),
                  hidden(src.hidden),
                  oily(src.oily),
                  extraArgs(src.extraArgs)
{
}

GComponentParams::~GComponentParams ()
{
}

GComponentParams& GComponentParams::operator= ( const GComponentParams& src )
{
   GAbstractResource::operator=(src);
   this->typeName = src.typeName;
   this->textID = src.textID;
   this->hintID = src.hintID;
   this->constraints = src.constraints;
   this->pos = src.pos;
   this->size = src.size;
   this->disabled = src.disabled;
   this->toggleOn = src.toggleOn;
   this->hidden = src.hidden;
   this->oily = src.oily;
   this->extraArgs = src.extraArgs;
   return *this;
}

GString GComponentParams::toString () const
{
   GString str(256);
   str += "typeName=";
   str += typeName;
   str += ", textID=\"";
   str += textID;
   str += "\", hintID=\"";
   str += hintID;
   str += "\", constraints=";
   str += constraints;
   str += ", pos=";
   str += pos.toString();
   str += ", size=";
   str += size.toString();
   str += ", disabled=";
   str += disabled;
   str += ", toggleOn=";
   str += toggleOn;
   str += ", hidden=";
   str += hidden;
   str += ", oily=";
   str += oily;
   str += ", ";
   str += extraArgs.toString();
   return str;
}

void GComponentParams::loadFromScript ( GRcTokenizer& tokenizer, 
                                        GResourceTable* /*table*/ )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lpar)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

   bool readNextToken = true;

   for (;;)
   {
      if (readNextToken)
         token = tokenizer.nextPreCompiledToken(false);

      readNextToken = true;

      if (*token == GRcTokenizer::Token_rpar)
         break;
      else
      if (*token == GRcTokenizer::Token_comma)
         continue;
      else
      if (*token == GRcTokenizer::Token_id)
      {
         if (getIDString() != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRIDARDEF)); // ID already defined!
         token = tokenizer.queryArgValue();
         setIDString(token->getString());
      }
      else
      if (*token == GRcTokenizer::Token_text)
      {
         token = tokenizer.queryArgValue();
         textID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_hint)
      {
         token = tokenizer.queryArgValue();
         hintID = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_xpos)
      {
         if (pos.x != 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRXPOSARDEF)); // XPOS already defined!
         pos.x = tokenizer.queryArgValueFloat();
      }
      else
      if (*token == GRcTokenizer::Token_ypos)
      {
         if (pos.y != 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRYPOSARDEF)); // YPOS already defined!
         pos.y = tokenizer.queryArgValueFloat();
      }
      else
      if (*token == GRcTokenizer::Token_width)
      {
         if (size.width)
            gthrow_(GRcException(tokenizer, GRcException::ERRWIDTHARDEF)); // WIDTH already defined!
         size.width = tokenizer.queryArgValueFloat();
         if (size.width <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRWIDTHLTHAN0)); // WIDTH must be larger than zero!
      }
      else
      if (*token == GRcTokenizer::Token_height)
      {
         if (size.height)
            gthrow_(GRcException(tokenizer, GRcException::ERRHEIGHTARDEF)); // HEIGHT already defined!
         size.height = tokenizer.queryArgValueFloat();
         if (size.height <= 0)
            gthrow_(GRcException(tokenizer, GRcException::ERRHEIGHTLTHAN0)); // HEIGHT must be larger than zero!
      }
      else
      if (*token == GRcTokenizer::Token_constraints)
      {
         token = tokenizer.queryArgValue();
         constraints = token->getString();
      }
      else
      if (*token == GRcTokenizer::Token_oily)
         oily = true;
      else
      if (*token == GRcTokenizer::Token_disabled)
         disabled = true;
      else
      if (*token == GRcTokenizer::Token_toggleon)
         toggleOn = true;
      else
      if (*token == GRcTokenizer::Token_hidden)
         hidden = true;
      else
      {
         // Next parameter isn't known by us, so add it to the list of extra
         // arguments that may be specific for the current type of component.
         // It is up to each component to use any of these extra arguments.
         GString idStr = token->getString();
         token = tokenizer.nextPreCompiledToken(false);
         if (*token == GRcTokenizer::Token_eq)
         {
            token = tokenizer.nextPreCompiledToken(false);
            const GString& str = token->getString();
            extraArgs.setParameter(idStr, str);
         }
         else
         {
            extraArgs.setParameter(idStr, GString::Empty);
            if (*token == GRcTokenizer::Token_rpar)
               break;
            else
               readNextToken = false;
         }
      }
   }
}

const GString& GComponentParams::getConstraints () const 
{ 
   return constraints; 
}

const GString& GComponentParams::getHintID () const 
{ 
   return hintID; 
}

const GString& GComponentParams::getTextID () const 
{ 
   return textID; 
}

const GString& GComponentParams::getTypeName () const 
{ 
   return typeName; 
}

const GComponentPos& GComponentParams::getPos () const 
{ 
   return pos; 
}

const GComponentSize& GComponentParams::getSize () const 
{ 
   return size; 
}

bool GComponentParams::isDisabled () const 
{ 
   return disabled; 
}

bool GComponentParams::isToggleOn () const 
{ 
   return toggleOn; 
}

bool GComponentParams::isHidden () const 
{ 
   return hidden; 
}

bool GComponentParams::isOily () const 
{ 
   return oily; 
}
