/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/resource/GAccelItem.h"
#include "glib/resource/GRcCompiler.h"
#include "glib/resource/GRcException.h"

GAccelItem::GAccelItem ( const GString& id )
           :GAbstractCommand(id),
            userDataIsDef(false)
{
}

GAccelItem::GAccelItem ( const GString& id, 
                         const GString& keyName, 
                         bool control, 
                         bool alt, 
                         bool shift, 
                         bool gray,
                         bool up )
           :GAbstractCommand(id),
            userDataIsDef(false)
{
   init(keyName, control, alt, shift, gray, up);
}

GAccelItem::~GAccelItem ()
{
}

void GAccelItem::init ( const GString& keyName, 
                        bool control, 
                        bool alt, 
                        bool shift, 
                        bool gray,
                        bool up )
{
   GString humanName(32);
   GString kname = keyName;
   kname.toUpperCase();

   bool anyAdded = false;

   if (control)
   {
      kname += "_C";
      if (anyAdded)
         humanName += "+";
      humanName += "Ctrl";
      anyAdded = true;
   }

   if (alt)
   {
      kname += "_A";
      if (anyAdded)
         humanName += "+";
      humanName += "Alt";
      anyAdded = true;
   }

   if (shift)
   {
      kname += "_S";
      if (anyAdded)
         humanName += "+";
      humanName += "Shift";
      anyAdded = true;
   }

   if (gray)
   {
      kname += "_N";
      if (anyAdded)
         humanName += "+";
      humanName += "Num ";
      anyAdded = true;
   }

   if (up)
   {
      kname += "_U";
   }

   if (anyAdded && !gray)
      humanName += "+";

   const GKeyBag<GString>& availak = GetAvailAccelKeys();
   GString* temp = availak.get(keyName);
   if (temp == null)
      humanName += "???"; // Should never happen, but in case
   else
      humanName += *temp;

   if (up)
   {
      humanName += " (Up)";
   }

   this->keyName = kname;
   this->keyNameForHumans = humanName;
}

const GKeyBag<GString>& GAccelItem::GetAvailAccelKeys ()
{
   static GKeyBag<GString>* Bag = null;

   // Make the bag of the available shortcut keys the first time
   // we are called. The bag is not case sensitive.
   if (Bag == null)
   {
      Bag = new GKeyBag<GString>(64, -3, true);

      const GString AccelChars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ 0123456789+-/*\\.,_?={}[]()<>!|@$\"#%&;:";
      for (int i=0, num=AccelChars.length(); i<num; i++)
      {
         GString key = AccelChars.substring(i, i+1);
         GString* val = new GString(key);
         Bag->put(key, val);
      }

      Bag->put("VK_NULL", new GString(GString::Empty));
      Bag->put("VK_F1", new GString("F1"));
      Bag->put("VK_F2", new GString("F2"));
      Bag->put("VK_F3", new GString("F3"));
      Bag->put("VK_F4", new GString("F4"));
      Bag->put("VK_F5", new GString("F5"));
      Bag->put("VK_F6", new GString("F6"));
      Bag->put("VK_F7", new GString("F7"));
      Bag->put("VK_F8", new GString("F8"));
      Bag->put("VK_F9", new GString("F9"));
      Bag->put("VK_F10", new GString("F10"));
      Bag->put("VK_F11", new GString("F11"));
      Bag->put("VK_F12", new GString("F12"));
      Bag->put("VK_DOWN", new GString("Down"));
      Bag->put("VK_UP", new GString("Up"));
      Bag->put("VK_LEFT", new GString("Left"));
      Bag->put("VK_RIGHT", new GString("Right"));
      Bag->put("VK_PAGEUP", new GString("PgUp"));
      Bag->put("VK_PAGEDOWN", new GString("PgDn"));
      Bag->put("VK_HOME", new GString("Home"));
      Bag->put("VK_END", new GString("End"));
      Bag->put("VK_INSERT", new GString("Ins"));
      Bag->put("VK_DELETE", new GString("Del"));
      Bag->put("VK_TAB", new GString("Tab"));
      Bag->put("VK_ESC", new GString("Esc"));
      Bag->put("VK_BACKSPACE", new GString("Backspace"));
      Bag->put("VK_SPACE", new GString("Space"));
      Bag->put("VK_ENTER", new GString("Enter"));
   }

   return *Bag;
}

const GString& GAccelItem::getKeyName () const
{
   return keyName;
}

const GString& GAccelItem::getKeyNameForHumans () const
{
   return keyNameForHumans;
}

void GAccelItem::loadFromScript ( GRcTokenizer& tokenizer, GResourceTable* table )
{
   const GRcToken* token = tokenizer.nextPreCompiledToken(false);
   if (*token != GRcTokenizer::Token_lpar)
      gthrow_(GRcException(tokenizer, GRcException::ERR_EXPECTED_X_FOUND_Y, GVArgs("(").add(token->getString()))); // Expected '(' but found '%2' in statement!

   GString keyNameBuff(32);
   bool alt = false;
   bool shift = false;
   bool control = false;
   bool gray = false;
   bool up = false;

   for (;;)
   {
      token = tokenizer.nextPreCompiledToken(false);

      if (*token == GRcTokenizer::Token_rpar)
         break;

      if (*token == GRcTokenizer::Token_comma)
         continue;

      if (*token == GRcTokenizer::Token_key)
      {
         if (keyName != "")
            gthrow_(GRcException(tokenizer, GRcException::ERRKEYARDEF)); // KEY already defined!

         token = tokenizer.queryArgValue();

         // Make sure the specified key is one of the legal shortcut keys.
         const GKeyBag<GString>& availAccelKeys = GetAvailAccelKeys();
         if (!availAccelKeys.containsKey(token->getString()))
            gthrow_(GRcException(tokenizer, GRcException::ERRUNKNOWNKEY, GVArgs(token->getString()))); // Unknown shortcut KEY: %s

         // Copy the specified shortcut key string to CB of current menu item.
         keyNameBuff = token->getString();
      }

      else
      if (*token == GRcTokenizer::Token_userdata)
      {
         if (userDataIsDef)
            gthrow_(GRcException(tokenizer, GRcException::ERRUSERDATA1ARDEF)); // Userdata already defined!

         token = tokenizer.queryArgValue();
         setUserData1(token->getString());
         userDataIsDef = true;
      }

      else
      if (*token == GRcTokenizer::Token_alt)
      {
         alt = true;
      }

      else
      if (*token == GRcTokenizer::Token_shift)
      {
         shift = true;
      }

      else
      if (*token == GRcTokenizer::Token_control)
      {
         control = true;
      }

      else
      if (*token == GRcTokenizer::Token_num)
      {
         gray = true;
      }

      else
      if (*token == GRcTokenizer::Token_up)
      {
         up = true;
      }

      else
         gthrow_(GRcException(tokenizer, GRcException::ERRWUNKNOWN, GVArgs(token->getString()))); // Unknown token: %s
   }

   if (keyNameBuff == "")
      gthrow_(GRcException(tokenizer, GRcException::ERRNOKEYDEF)); // No KEY defined!

   init(keyNameBuff, control, alt, shift, gray, up);
}

