/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ABSTRACTRESOURCE
#define __GLIB_ABSTRACTRESOURCE

#include "glib/primitives/GString.h"

/**
 * This is the base class of all resource types, such as menues, icons,
 * toolbars, texts and dialogs.
 *
 * The main purpose of this class is to keep track of the ID string of the
 * resource object, and to declare the pure virtual method of which to be
 * used in order to let the resource object initialize it self from a
 * Resource Script token stream.
 *
 * @author  Leif Erik Larsen
 * @since   1999.09.13
 */
class GAbstractResource : public GObject
{
   private:

      GString idString;

   public:

      explicit GAbstractResource ( const GString& id );
      GAbstractResource ( const GAbstractResource& src );
      virtual ~GAbstractResource ();

   protected:

      GAbstractResource ();

   public:

      GAbstractResource& operator= ( const GAbstractResource& src );
      bool operator== ( const GAbstractResource& src ) const;
      bool operator!= ( const GAbstractResource& src ) const;

   public:

      virtual const GString& getIDString () const;
      virtual void setIDString ( const GString& newID );

      /**
       * Load and compile the following resource definition from the
       * specified tokenizer for the calling resource compiler.
       *
       * Since we assume that the ID-part of the resource has already been
       * parsed the next token (which is the first, from our point of view)
       * returned from the tokenizer must be "(".
       *
       * @author  Leif Erik Larsen
       * @since   2000.09.08
       * @param   tokenizer   The tokenizer of where to get tokens to parse.
       * @param   table       The resource table of where to load referenced
       *                      texts and other referenced resources (if any).
       *                      If this parameter is null then the referenced
       *                      resources should not be inlined.
       * @throws  GRcException in case of any error.
       */
      virtual void loadFromScript ( class GRcTokenizer& tokenizer, 
                                    class GResourceTable* table ) = 0;
};

#endif


