/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/primitives/GVArgs.h"
#include "glib/primitives/GBoolean.h"
#include "glib/primitives/GCharacter.h"
#include "glib/primitives/GShort.h"
#include "glib/primitives/GUShort.h"
#include "glib/primitives/GInteger.h"
#include "glib/primitives/GUInteger.h"
#include "glib/primitives/GLong.h"
#include "glib/primitives/GULong.h"
#include "glib/primitives/GFloat.h"
#include "glib/primitives/GDouble.h"

GVArgs::Arg::Arg ( GObject* obj ) 
            :obj(obj)
{
}

GVArgs::Arg::~Arg ()
{
   delete obj;
}

GVArgs::GVArgs ()
       :args(null)
{
}

GVArgs::GVArgs ( const char* a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( const GString& a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( bool a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( char a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( short a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( unsigned short a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( int a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( unsigned int a )
       :args(null)
{
   add(a);
}

#if !defined(__IBMCPP4__)
GVArgs::GVArgs ( long a )
       :args(null)
{
   add(a);
}
#endif

#if !defined(__IBMCPP4__)
GVArgs::GVArgs ( unsigned long a )
       :args(null)
{
   add(a);
}
#endif

GVArgs::GVArgs ( longlong a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( ulonglong a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( float a )
       :args(null)
{
   add(a);
}

GVArgs::GVArgs ( double a )
       :args(null)
{
   add(a);
}

GVArgs::~GVArgs ()
{
   delete args;
}

GVArgs& GVArgs::add ( const char* a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GString(a)));
   return *this;
}

GVArgs& GVArgs::add ( const GString& a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GString(a)));
   return *this;
}

GVArgs& GVArgs::add ( bool a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GBoolean(a)));
   return *this;
}

GVArgs& GVArgs::add ( char a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GCharacter(a)));
   return *this;
}

GVArgs& GVArgs::add ( short a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GShort(a)));
   return *this;
}

GVArgs& GVArgs::add ( unsigned short a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GUShort(a)));
   return *this;
}

GVArgs& GVArgs::add ( int a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GInteger(a)));
   return *this;
}

GVArgs& GVArgs::add ( unsigned int a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GUInteger(a)));
   return *this;
}

#if !defined(__IBMCPP4__)
GVArgs& GVArgs::add ( long a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GInteger(a)));
   return *this;
}
#endif

#if !defined(__IBMCPP4__)
GVArgs& GVArgs::add ( unsigned long a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GUInteger(a)));
   return *this;
}
#endif

GVArgs& GVArgs::add ( longlong a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GLong(a)));
   return *this;
}

GVArgs& GVArgs::add ( ulonglong a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GULong(a)));
   return *this;
}

GVArgs& GVArgs::add ( float a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GFloat(a)));
   return *this;
}

GVArgs& GVArgs::add ( double a )
{
   if (args == null)
      args = new GArray<GVArgs::Arg>(16);
   args->add(new GVArgs::Arg(new GDouble(a)));
   return *this;
}

const GObject* GVArgs::get ( int idx ) const
{
   if (idx < 0 || idx >= num())
      return null;
   else
      return args->get(idx).obj;
}

const GObject* GVArgs::get_excp ( int idx ) const
{
   if (idx < 0 || idx >= num())
      gthrow_(UndefinedArgumentException(idx));
   return args->get(idx).obj;
}

int GVArgs::num () const 
{ 
   if (args == null)
      return 0;
   else
      return args->getCount();
}

const GString GVArgs::STR_NULL = "[null]";
const GString GVArgs::STR_ERROR = "[error]";

GVArgs::UndefinedArgumentException::UndefinedArgumentException ( int idx )
       :GException("[Error: Undefined argument #" + GInteger::ToString(idx+1) + "]")
{
}

GVArgs::Tag::Tag ( const char* exp, const int startPos, const int explen )
            :width(0),
             precision(-1),
             leadingZeroes(false),
             showPlus(false),
             alternate(false),
             showSpace(false),
             leftAlign(false),
             formatCode(' '),
             codeLength(0)
{
   int pos = startPos;

   // Scanning Phase #1.
   for (; pos<explen; pos++)
   {
      char code = exp[pos];
      switch (code)
      {
         case ' ': showSpace = true; continue;
         case '-': leftAlign = true; continue;
         case '+': showPlus = true; continue;
         case '0': leadingZeroes = true; continue;
         case '#': alternate = true; continue;
      }
      break; // Break the loop.
   }

   // Scanning phase #2.
   for (; pos<explen;)
   {
      char code = exp[pos];
      if ('0' <= code && code <= '9')
      {
         width = width * 10 + code - '0';
         pos++;
      }
      else
      if (code == '.')
      {
         precision = 0;
         pos++;

         // Sub-scanning for phase II:
         for (; pos<explen;)
         {
            code = exp[pos];
            if ('0' <= code && code <= '9')
            {
               precision = precision * 10 + code - '0';
               pos++;
            }
            else
            {
               formatCode = code;
               pos++;
               break;
            }
         }

         break;
      }
      else
      {
         formatCode = code;
         pos++;
         break;
      }
   }

   // ---
   codeLength = pos - startPos;
}

GVArgs::Tag::~Tag ()
{
}

GString GVArgs::Tag::fixedFormatted ( double d )
{
   if (d > GLong::MAX_VALUE)
      return expFormatted(d);

   bool removeTrailing = (formatCode == 'G' ||
                          formatCode == 'g' ||
                          formatCode == 'E' ||
                          formatCode == 'e') && !alternate;

   if (precision == 0)
   {
      const GString& s = GLong::ToString(long(d + 0.5));
      if (removeTrailing)
         return s;
      else
         return s + '.';
   }

   long whole = long(d);
   double fr = d - whole; // Fractional part.
   if (fr >= 1 || fr < 0)
      return expFormatted(d);

   double factor = 1;
   GString leading_zeroes(12);
   for (int i=1; i<=precision && factor <= GLong::MAX_VALUE; i++)
   {
      factor *= 10;
      leading_zeroes += '0';
   }
   long l = long(factor * fr + 0.5);
   if (l >= factor)
   {
      l = 0;
      whole++;
   }

   GString z = leading_zeroes + GLong::ToString(l);
   GString sub = z.substring(z.length() - precision, z.length());
   z = "." + sub;

   if (removeTrailing)
   {
      while (z.lastChar() == '0')
         z.removeLastChar();
      if (z.lastChar() == '.')
         z.removeLastChar();
   }

   return GLong::ToString(whole) + z;
}

GString GVArgs::Tag::expFormatted ( double d )
{
   GString f(16);
   int e = 0;
   double dd = d;
   double factor = 1;
   if (d != 0)
   {
      while (dd > 10)
      {
         e++;
         factor /= 10;
         dd = dd / 10;
      }

      while (dd < 1)
      {
         e--;
         factor *= 10;
         dd = dd * 10;
      }
   }
   if ((formatCode == 'g' || formatCode == 'G') && e >= -4 && e < precision)
      return fixedFormatted(d);

   d = d * factor;
   f += fixedFormatted(d);

   if (formatCode == 'e' || formatCode == 'g')
      f += "e";
   else
      f += "E";

   GString p = "000";
   if (e >= 0)
   {
      f += "+";
      p += GInteger::ToString(e);
   }
   else
   {
      f += "-";
      p += GInteger::ToString(-e);
   }

   GString sub = p.substring(p.length() - 3, p.length());
   return f + sub;
}

GString GVArgs::Tag::packWith ( char c, int n )
{
   if (n <= 0)
      return GString::Empty;
   GString s(n);
   for (int i=0; i<n; i++)
      s += c;
   return s;
}

GString GVArgs::Tag::setIndication ( int parameter, const GString& formattedExp )
{
   GString result(2);
   if (parameter < 0)
   {
      result = "-";
   }
   else
   if (parameter > 0)
   {
      if (showPlus)
         result = "+";
      else
      if (showSpace)
         result = GString::Blank;
   }
   else
   if (alternate)
   {
      if (formatCode == 'o' && formattedExp.length() > 0 && formattedExp[0] != '0')
         result = "0";
      else
      if (formatCode == 'x')
         result = "0x";
      else
      if (formatCode == 'X')
         result = "0X";
   }

   if (leadingZeroes)
      return result + packWith('0', width - result.length() - formattedExp.length()) + formattedExp;
   else
   if ((formatCode == 'd' || formatCode == 'i' || formatCode == 'x' || formatCode == 'X' || formatCode == 'o') && precision > 0)
      return result + packWith('0', precision - result.length() - formattedExp.length()) + formattedExp;
   else
      return result + packWith('0', 0 - result.length() - formattedExp.length()) + formattedExp;
}

GString GVArgs::Tag::formatString ( const GObject* arg )
{
   GString argString = arg->toString();
   if (precision >= 0 && precision < argString.length())
      argString = argString.substring(0, precision);
   GString blancs = packWith(' ', (width - argString.length()));
   if (leftAlign)
      return argString + blancs;
   else
      return blancs + argString;
}

GString GVArgs::Tag::formatCharacter ( const GObject* arg )
{
   char argChar;
   if (dynamic_cast<const GNumber*>(arg) != null)
   {
       argChar = dynamic_cast<const GNumber*>(arg)->charValue();
   }
   else
   {
      const GString& s = arg->toString();
      if (s.length() <= 0)
         argChar = ' ';
      else
         argChar = s[0];
   }

   GString chrstr = GCharacter::ToString(argChar);
   GString blancs = packWith(' ', width - chrstr.length());

   if (leftAlign)
      return chrstr + blancs;
   else  // Right align.
      return blancs + chrstr;
}

GString GVArgs::Tag::formatDouble ( const GNumber* arg )
{
   if (precision < 0)
      precision = 6;

   int sign = 1;
   double argDouble = arg->doubleValue();
   if (argDouble < 0)
   {
      argDouble = -argDouble;
      sign = -1;
   }

   GString formattedStr;
   if (formatCode == 'f')
   {
      const GString& ff = fixedFormatted(argDouble);
      formattedStr = setIndication(sign, ff);
   }
   else
   if (formatCode == 'e' || formatCode == 'E' ||
       formatCode == 'g' || formatCode == 'G')
   {
      const GString& ef = expFormatted(argDouble);
      formattedStr = setIndication(sign, ef);
   }
   else
      return "";

   const GString& blancs = packWith(' ', width - formattedStr.length());
   if (leftAlign)
      return formattedStr + blancs;
   else // Right align.
      return blancs + formattedStr;
}

GString GVArgs::Tag::formatLong ( const GNumber* arg )
{
   longlong argLong = arg->longValue();

   GString res;
   switch (formatCode)
   {
      case 'd':
      case 'i': 
      {
         if (argLong < 0)
         {
            GString s = GLong::ToString(argLong);
            GString sub = s.substring(1);
            res = setIndication(-1, sub);
         }
         else
         {
            GString s = GLong::ToString(argLong);
            res = setIndication(1, s);
         }
         break;
      }

      case 'b': 
      {
         GString s = GLong::ToString(argLong, 2);
         res = setIndication(0, s);
         break; 
      }

      case 'o': 
      {
         GString s = GLong::ToString(argLong, 8);
         res = setIndication(0, s);
         break; 
      }

      case 'x': 
      {
         GString s = GLong::ToString(argLong, 16);
         s.toLowerCase();
         res = setIndication(0, s);
         break; 
      }

      case 'X': 
      {
         GString s = GLong::ToString(argLong, 16);
         s.toUpperCase();
         res = setIndication(0, s);
         break; 
      }

      default:
         return GString::Empty;
   }

   GString blancs = packWith(' ', width - res.length());
   if (leftAlign)
      return res + blancs;
   else // Right align.
      return blancs + res;
}

GString GVArgs::FormatArgs ( const GString& str, const GVArgs& args )
{
   const int len = str.length();
   const int buffsize = len + (16 * args.num());
   GString buff(buffsize);
   FormatArgs(buff, str, len, args);
   return buff;
}

void GVArgs::FormatArgs ( GString& buff, const char* str, const int len, const GVArgs& args )
{
   int argIndex = 0;

   for (int i=0; i<len;)
   {
      char chr = str[i++];
      if (chr == '#')
      {
         if (i >= len)
         {
            buff += STR_ERROR;
            break;
         }

         // Following number references the n'th argument.
         char c = str[i];
         if (c == '#')
         {
            // Append the '#' character it self.
            buff += c;
            i++;
            continue;
         }
         GString b(2);
         for (; i<len; c=str[++i])
         {
            if (!GCharacter::IsDigit(c))
               break;
            b += c;
         }
         try {
            int idx = GInteger::ParseInt(b) - 1;
            const GObject* a = args.get_excp(idx);
            if (a == null)
               buff += STR_NULL;
            else
               buff += a->toString();
         } catch (UndefinedArgumentException& e) {
            buff += e.getMessage();
         } catch (std::exception& /*e*/) {
            buff += STR_ERROR;
         }
         continue;
      }
      else
      if (chr != '%')
      {
         buff += chr;
         continue;
      }
      else
      if (i >= len)
      {
         buff += STR_ERROR;
         break;
      }

      // ---
      char nextChr = str[i];
      if (nextChr == '%')
      {
         // Append the '%' character it self.
         i++;
         buff += '%';
         continue;
      }
      else
      if (GCharacter::IsDigit(nextChr))
      {
         // For backward compatibility reasons, allow syntax forms
         // like "%1" instead of the newer "#1" to access numbered
         // arguments, as long as the digit is only one character
         // long (1..9 that is) and the character next to it is not
         // equal to any supported format characters.
         try {
            char nextNextChr = ' ';
            if (i + 1 < len)
               nextNextChr = str[i+1];
            if (nextNextChr != '.' && !GCharacter::IsLetterOrDigit(nextNextChr))
            {
               i++;
               int idx = int(nextChr - '0') - 1;
               const GObject* a = args.get_excp(idx);
               if (a == null)
                  buff += STR_NULL;
               else
                  buff += a->toString();
               continue;
            }
         } catch (UndefinedArgumentException& e) {
            buff += e.getMessage();
            continue;
         }
      }

      // We have reached another %-tag of which to parse and translate.
      Tag tag(str, i, len);
      i += tag.codeLength;
      char typeChr = tag.formatCode;
      switch (typeChr)
      {
         case 'd': // Signed decimal int/long (Byte, Short, Integer, Long, Float, Double or "Double.parseLong(arg.toString())").
         case 'i': // Ditto.
         case 'b': // Ditto, but binary.
         case 'o': // Ditto, but octal.
         case 'x': // Ditto, but hexadecimal with lowercase a...f.
         case 'X': { // Ditto, but uppercase A...F.
            GString s;
            try {
               const GObject* a = args.get_excp(argIndex++);
               if (a == null)
                  s = STR_NULL;
               else
               if (dynamic_cast<const GNumber*>(a) != null)
                  s = tag.formatLong(dynamic_cast<const GNumber*>(a));
               else
               try {
                  GLong l(GLong::ParseLong(a->toString()));
                  s = tag.formatLong(&l);
               } catch (std::exception& /*e*/) {
                  s = STR_ERROR;
               }
            } catch (UndefinedArgumentException& e) {
               s = e.getMessage();
            }
            buff += s;
            break; }

         case 'f': // Signed floating point (Byte, Short, Integer, Long, Float, Double or "Double.parseDouble(arg.toString())").
         case 'g': // Ditto.
         case 'G': // Ditto.
         case 'e': // Ditto, but in exponential form with lowercase e.
         case 'E': { // Ditto, but uppercase E.
            GString s;
            try {
               const GObject* a = args.get_excp(argIndex++);
               if (a == null)
                  s = STR_NULL;
               else
               if (dynamic_cast<const GNumber*>(a) != null)
                  s = tag.formatDouble(dynamic_cast<const GNumber*>(a));
               else
               try {
                  GDouble d(GDouble::ParseDouble(a->toString()));
                  s = tag.formatDouble(&d);
               } catch (std::exception& /*e*/) {
                  s = STR_ERROR;
               }
            } catch (UndefinedArgumentException& e) {
               s = e.getMessage();
            }
            buff += s;
            break; }

         case 'c': { // Single character (Character or "arg.toString().charAt(0)").
            GString s;
            try {
               const GObject* a = args.get_excp(argIndex++);
               if (a == null)
                  s = STR_NULL;
               else
                  s = tag.formatCharacter(a);
            } catch (UndefinedArgumentException& e) {
               s = e.getMessage();
            }
            buff += s;
            break; }

         case 's': { // Character string (String or "arg.toString()").
            GString s;
            try {
               const GObject* a = args.get_excp(argIndex++);
               if (a == null)
                  s = STR_NULL;
               else
                  s = tag.formatString(a);
            } catch (UndefinedArgumentException& e) {
               s = e.getMessage();
            }
            buff += s;
            break; }

         default: {
            buff += STR_ERROR;
            break; }
      }
   }
}
