/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_ULONG
#define __GLIB_ULONG

#include "glib/primitives/GIntegerNumber.h"

/**
 * A general unsigned long integer representation class.
 *
 * @author  Leif Erik Larsen
 * @since   2000.11.01
 */
class GULong : public GIntegerNumber<ulonglong>
{
   private:

      ulonglong value;

   public:

      static const ulonglong MIN_VALUE;
      static const ulonglong MAX_VALUE;

   public:

      GULong ();
      GULong ( const GULong& val );
      GULong ( ulonglong val );

      virtual ~GULong ();

   public:

      operator ulonglong () const { return value; }

   public:

      virtual bool boolValue () const { return value == 0 ? false : true; }
      virtual char charValue () const { return char(value); }
      virtual short shortValue () const { return short(value); }
      virtual int intValue () const { return int(value); }
      virtual longlong longValue () const { return longlong(value); }
      virtual unsigned short ushortValue () const { return (unsigned short) value; }
      virtual unsigned int uintValue () const { return (unsigned int) value; }
      virtual ulonglong ulongValue () const { return value; }
      virtual float floatValue () const { return float(value); }
      virtual double doubleValue () const { return double(value); }

      virtual bool isUnsigned () const { return true; }

      virtual int hashCode () const;
      virtual bool equals ( const GObject& obj ) const;

   public:

      /** 
       * Increment the contained value, and return the new value, 
       * in a thread safe way. 
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.11
       */
      ulonglong increment ();

      /** 
       * Decrement the contained value, and return the new value, 
       * in a thread safe way. 
       *
       * @author  Leif Erik Larsen
       * @since   2004.05.11
       */
      ulonglong decrement ();

      /**
       * Return a <i>GString</i> representation of the current
       * unsigned long integer value.
       */
      virtual GString toString () const;

      /**
       * Convert the specified unsigned long integer value to a string.
       *
       * @since  2000.10.13
       * @author Leif Erik Larsen
       * @param  value   Unsigned long integer value of which to convert.
       * @param  radix   2, 8, 10 or 16.
       * @param  width   Width of the returned string.
       * @param  padChar If the value it self is not large enough to produce
       *                 a string that is at least <i>width</i> characters
       *                 long then we will insert a number of this character
       *                 to satisfy the requested width.
       */
      static GString ToString ( ulonglong value, short radix = 10, int width = 0, char padChar = '0' );

      /**
       * Get the parsed unsigned long integer value of the specified
       * string, assuming the string is specifying an unsigned long integer
       * with the specified radix.
       *
       * @param   str    String of which to parse.
       * @param   radix  0, 2, 8, 10 or 16. If 0 is specified then we
       *                 will use the default radix as with respect to
       *                 the format of the string; assuming that the
       *                 string is specifying an unsigned integer number
       *                 with radix 10, or radix 16 if the string starts
       *                 with the character sequence "0x".
       * @return  The parsed unsigned long integer value.
       * @throws  GNumberFormatException if the specified string is not
       *                                 possible to parse into an unsigned long.
       */
      static ulonglong ParseULong ( const GString& str, short radix = 0 );
};

#endif


