/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/io/GPipeInputStream.h"
#include "glib/GProgram.h"
#include "glib/util/GMath.h"
#include "glib/util/GLog.h"

GPipeInputStream::GPipeInputStream ( HPipe hpipe, 
                                     bool autoClose )
                 :GFileInputStream(GPipeInputStream::LocalVfs,
                                   GVfs::FileHandle(hpipe), 
                                   autoClose,
                                   false, // textMode.
                                   "[GPipeInputStream]")
{
}

// Disable the copy constructor.
GPipeInputStream::GPipeInputStream ( const GPipeInputStream& src ) 
                 :GFileInputStream(GPipeInputStream::LocalVfs, null, false, false) 
{
}

GPipeInputStream::~GPipeInputStream ()
{
}

unsigned GPipeInputStream::peekBytesAvailable () const
{
   GPipeInputStream* self = const_cast<GPipeInputStream*>(this);
   GSysFileHandle hpipe = GSysFileHandle(self->getFileHandle());
   AVAILDATA avail;
   ULONG buff_ = 0; // Dummy.
   ULONG read_ = 0; // Dummy.
   ULONG state_ = 0; // Dummy.
   APIRET rc = ::DosPeekNPipe(hpipe, &buff_, 0, &read_, &avail, &state_);
   if (rc != NO_ERROR)
      return 0; 
   return avail.cbpipe;
}

void GPipeInputStream::readUnblocked ( GString& text, int maxLength )
{
   char buff[256];
   text.clear();
   for (;;)
   {
      // Test if there is any data available to be read without blocking.
      unsigned countAvail = peekBytesAvailable();
      if (countAvail <= 0)
         break;

      // Read at least some of the buffered bytes, but don't attempt 
      // to read more bytes than we know exist in the buffer.
      // Remember: This method should never block!
      int maxRead = GMath::Min(sizeof(buff) - 1, countAvail);
      if (maxRead > maxLength && maxLength > 0)
         maxRead = maxLength;
      int numRead = read(buff, maxRead);
      buff[numRead] = '\0'; // Just to make the buffer clean for debugging sessions.
      if (numRead <= 0)
         break;
      text.appendNum(buff, numRead);
      if (maxLength > 0)
      {
         maxLength -= numRead;
         if (maxLength <= 0)
            break;
      }
   }
}

