/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FILEOUTPUTSTREAM
#define __GLIB_FILEOUTPUTSTREAM

#include "glib/vfs/GVfs.h"
#include "glib/vfs/GFile.h"
#include "glib/io/GOutputStream.h"
#include "glib/io/GFileNotFoundException.h"
#include "glib/exceptions/GIllegalArgumentException.h"

/**
 * A file output stream is an output stream for writing data to a file.
 * It is meant for writing streams of raw bytes such as image data. 
 * For writing streams of characters, consider using {@link GFileWriter}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.04.15
 */
class GFileOutputStream : public GOutputStream
{
   private:

      class GVfs& vfs;
      GString path;
      GVfs::FileHandle hfile; // File is closed if == null.
      bool autoClose;
      bool textMode;

      // Variables used for buffering.
      static const int BuffSize;
      mutable int buffPos;
      mutable BYTE* myBuff;

   public:

      /**
       * Creates a file stream object for writing data sequentally to the 
       * file with the specified name.
       *
       * @param   vfs            The Virtual File System of where to open 
       *                         the file.
       * @param   name           The vfs-dependent filename of which to open.
       * @param   createIfNew    True if we shall create the file in case it
       *                         doesn't already exist.
       * @param   replaceIfExist True if we shall replace the file in case it
       *                         does already exist.
       * @param   textMode       If this parameter is true the file is 
       *                         opened in "text mode", meaning that any 
       *                         carriage-return sequences of characters are 
       *                         automatically recognized as linefeeds.
       * @throws  GFileNotFoundException    If the file exists but is a
       *                                    directory rather than a regular
       *                                    file, or if file does not
       *                                    exist at all. <b>Note</b> that
       *                                    this class is a subclass of
       *                                    {@link GOpenFileException}.
       * @throws  GOpenFileException        If the file exist but cannot be
       *                                    opened or created for some
       *                                    other reason.
       */
      GFileOutputStream ( class GVfs& vfs,
                          const GString& name, 
                          bool createIfNew, 
                          bool replaceIfExist,
                          bool textMode );

      /**
       * Same as {@link GFileOutputStream(const GString&)},
       * but giving a file object instead of just the file name.
       *
       * @throws  The same as {@link GFileOutputStream(const GString&)}.
       */
      GFileOutputStream ( class GVfs& vfs,
                          const GFile& file, 
                          bool createIfNew, 
                          bool replaceIfExist,
                          bool textMode );

      /**
       * Create a file output stream that wraps the specified system 
       * dependent file handle.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @param   hfile     The system dependent file handle of which to wrap.
       * @param   autoClose True if we shall close the handle automatically.
       * @param   textMode  If this parameter is true the file is 
       *                    opened in "text mode", meaning that any 
       *                    carriage-return sequences of characters are 
       *                    automatically recognized as linefeeds.
       * @param   path      The path of the file. For debug and error 
       *                    messages only.
       */
      GFileOutputStream ( class GVfs& vfs,
                          GVfs::FileHandle hfile, 
                          bool autoClose, 
                          bool textMode,
                          const GString& path = GString::Empty );

      /**
       * The destructor will automatically flush and close the stream 
       * if it is still open.
       */
      virtual ~GFileOutputStream ();

   private:

      /**
       * Common construction code.
       *
       * @throws  The same as {@link GFileOutputStream(const GString&)}.
       */
      void init ( bool createIfNew, bool replaceIfExist );

      /**
       * Prevent usage of the copy constructor.
       */
      GFileOutputStream ( const GFileOutputStream& src ) : vfs(src.vfs) {}

      /**
       * Prevent usage of the assignment operator.
       */
      const GFileOutputStream& operator= ( const GFileOutputStream& src ) const { return *this; }

   public:

      /**
       * Flush and close the stream if it is still open.
       * If it is already closed we will do nothing but return.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.23
       */
      void close ();

      /**
       * Get the low level file handle that is contained by this stream.
       *
       * @author  Leif Erik Larsen
       * @since   2004.11.23
       */
      GVfs::FileHandle getFileHandle ();

      /**
       * Flush the buffered output stream bytes, if any.
       *
       * @author  Leif Erik Larsen
       * @since   2004.12.10
       * @throws  GIOException In case of any error flushing the buffer.
       */
      virtual void flush () const;

      /**
       * Implements {@link GOutputStream#writeByte}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @throws  GIOException  in case of any error writing the byte.
       */
      virtual void writeByte ( int b ) const;

      /**
       * Implements {@link GOutputStream#write}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @throws  GIOException  in case of any error writing the buffer.
       */
      virtual void write ( const void* buff, int size, int count ) const;

      /**
       * Implements {@link GOutputStream#print}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @throws  GIOException  in case of any error writing the string.
       */
      virtual int print ( const GString& str ) const;

      /**
       * Implements {@link GOutputStream#printf}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @throws  GIOException  in case of any error writing the
       *                        formatted string.
       */
      virtual int printf ( const char* str, const GVArgs& args = GVArgs() ) const;
};

#endif
