/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_FILEINPUTSTREAM
#define __GLIB_FILEINPUTSTREAM

#include "glib/vfs/GVfs.h"
#include "glib/vfs/GFile.h"
#include "glib/io/GInputStream.h"
#include "glib/io/GFileNotFoundException.h"
#include "glib/exceptions/GIllegalArgumentException.h"

/**
 * A <code>GFileInputStream</code> obtains input bytes from a file. 
 * It is meant for reading streams of raw bytes such as image data. 
 * For reading streams of characters, consider using {@link GFileReader}.
 *
 * @author  Leif Erik Larsen
 * @since   2004.04.15
 */
class GFileInputStream : public GInputStream
{
   private:

      class GVfs& vfs;
      GString path;
      GVfs::FileHandle hfile;
      bool autoClose;
      bool textMode;

      /** Used only if {@link #textMode} is true. */
      mutable int nextByte;

      // Variables used for buffering.
      static const int BuffSize;
      mutable int buffPos;
      mutable int buffCount; 
      mutable BYTE* buff;

   public:

      /**
       * Creates a file stream object for reading data sequentally from the 
       * file with the specified name.
       *
       * @param   vfs       The Virtual File System of where to open the file.
       * @param   name      The vfs-dependent filename of which to open.
       * @param   textMode  If this parameter is true the file is opened
       *                    in "text mode", meaning that linefeed characters 
       *                    are automatically translated into the platform 
       *                    dependent linefeed sequence (that is; "\r\n" on
       *                    Windows and OS/2, or just "\n" on Linux) 
       *                    on output. Text mode files will also 
       *                    automatically "stop reading" when and end-of-file
       *                    or a null-character is reached. That is, method
       *                    {@link #readByte} will return -1 in that case.
       * @throws  GFileNotFoundException    If the file exists but is a
       *                                    directory rather than a regular
       *                                    file, or if file does not
       *                                    exist at all. <b>Note</b> that
       *                                    this class is a subclass of
       *                                    {@link GOpenFileException}.
       * @throws  GOpenFileException        If the file exist but cannot be
       *                                    opened or created for some
       *                                    other reason.
       */
      GFileInputStream ( class GVfs& vfs,
                         const GString& name, 
                         bool textMode );

      /**
       * Same as {@link GFileInputStream(const GString&)},
       * but giving a file object instead of just the file name.
       *
       * @throws  The same as {@link GFileInputStream(const GString&)}.
       */
      GFileInputStream ( class GVfs& vfs,
                         const GFile& file, 
                         bool textMode );

      /**
       * Create a file input stream that wraps the specified system 
       * dependent file handle.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       * @param   vfs       The Virtual File System of where the file 
       *                    was opened.
       * @param   hfile     The vfs-dependent file handle of which to wrap.
       * @param   autoClose True if we shall close the handle automatically.
       * @param   textMode  If this parameter is true the file is opened
       *                    in "text mode", meaning that linefeed characters 
       *                    are automatically translated into the platform 
       *                    dependent linefeed sequence (that is; "\r\n" on
       *                    Windows and OS/2, or just "\n" on Linux) 
       *                    on output.
       * @param   path      The path of the file. For debug and error 
       *                    messages only.
       */
      GFileInputStream ( class GVfs& vfs,
                         GVfs::FileHandle hfile, 
                         bool autoClose, 
                         bool textMode,
                         const GString& path = GString::Empty );

      /**
       * Close the file.
       */
      virtual ~GFileInputStream ();

   private:

      /**
       * Common construction code.
       *
       * @throws  The same as {@link GFileInputStream(const GString&)}.
       */
      void init ();

      /**
       * Prevent usage of the copy constructor.
       */
      GFileInputStream ( const GFileInputStream& src ) : vfs(src.vfs) {}

      /**
       * Prevent usage of the assignment operator.
       */
      const GFileInputStream& operator= ( const GFileInputStream& src ) const { return *this; }

   public:

      /**
       * Get the low level file handle that is contained by this stream.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.28
       */
      GVfs::FileHandle getFileHandle ();

      /**
       * Implements {@link GInputStream#readByte}.
       *
       * If the file is opened in Text Mode then we will automatically 
       * translate any platform dependent linefeed sequence to a pure 
       * single linefeed character. Text mode files will also return -1
       * from this method when an end-of-file or a null-character is 
       * reached.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       */
      virtual int readByte () const;

      /**
       * Implements {@link GInputStream#readString}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       */
      virtual GString& readString ( GString& str ) const;

      /**
       * Implements {@link GInputStream#read}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       */
      virtual int read ( void* buff, int count ) const;

      /**
       * Implements {@link GInputStream#readExact}.
       *
       * @author  Leif Erik Larsen
       * @since   2004.04.15
       */
      virtual void readExact ( void* buff, int count ) const;
};

#endif
