/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/tree/GMutableTreeNode.h"
#include "glib/gui/GIcon.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/exceptions/GIllegalArgumentException.h"
#include "glib/exceptions/GNoSuchElementException.h"

GMutableTreeNode::GMutableTreeNode ( bool allowsChildren,
                                     GObject* userObject, 
                                     bool autoDeleteUserObject,
                                     const GString& iconClosedID,
                                     const GString& iconOpenedID )
                 :GTreeNode(userObject, autoDeleteUserObject),
                  parent(null),
                  children(null),
                  allowsChildren(allowsChildren),
                  iconClosed(null),
                  iconOpened(null),
                  autoDeleteIconClosed(false),
                  autoDeleteIconOpened(false)
{
   if (iconClosedID != "")
      setIconClosed(iconClosedID);
   if (iconOpenedID != "")
      setIconClosed(iconOpenedID);
}

GMutableTreeNode::~GMutableTreeNode ()
{
   if (autoDeleteIconClosed)
      delete iconClosed;
   if (autoDeleteIconOpened)
      delete iconOpened;
   delete children;
}

int GMutableTreeNode::isAutoDeleteChildAt ( int index ) const
{
   if (children == null)
      gthrow_(GArrayIndexOutOfBoundsException("node has no children"));
   return children->isAutoDeleteAt(index);
}

void GMutableTreeNode::insert ( GMutableTreeNode* child, int index, bool autoDelete )
{
   if (!allowsChildren)
      gthrow_(GIllegalStateException("node does not allow children"));
   if (child == null)
      gthrow_(GIllegalArgumentException("new child is null"));
   if (isNodeAncestor(*child))
      gthrow_(GIllegalArgumentException("new child is an ancestor"));

   GMutableTreeNode* oldParent = dynamic_cast<GMutableTreeNode*>(child->getParent());
   if (oldParent != null)
      oldParent->remove(child, false); // Remove, but never destroy the child object!
   child->setParent(this);
   if (children == null)
      children = new GArray<GMutableTreeNode>(32);
   children->insert(child, index, autoDelete);
}

void GMutableTreeNode::remove ( int index, bool doDestroy ) 
{
   GMutableTreeNode& child = dynamic_cast<GMutableTreeNode&>(getChildAt(index));
   child.setParent(null);
   children->remove(index, 1, doDestroy);
}

void GMutableTreeNode::remove ( GMutableTreeNode* node, bool doDestroy )
{
   if (node == null)
      gthrow_(GIllegalArgumentException("argument is null"));
   if (!isNodeChild(*node))
      gthrow_(GIllegalArgumentException("argument is not a child"));
   int index = getIndex(*node);
   remove(index, doDestroy);
}

void GMutableTreeNode::setParent ( GMutableTreeNode* newParent ) 
{
   this->parent = newParent;
}

GTreeNode* GMutableTreeNode::getParent () 
{
	return parent;
}

GTreeNode& GMutableTreeNode::getChildAt ( int index ) const
{
   if (children == null)
      gthrow_(GArrayIndexOutOfBoundsException("node has no children"));
   return children->get(index);
}

int GMutableTreeNode::getChildCount () const
{
   if (children == null)
      return 0;
   return children->getCount();
}

int GMutableTreeNode::getIndex ( const GTreeNode& node ) const
{
   if (!isNodeChild(node))
      return -1;
   return children->indexOf(node);
}

void GMutableTreeNode::setAllowsChildren ( bool allows ) 
{
   if (allows == allowsChildren) 
      return;
   allowsChildren = allows;
   if (!allowsChildren)
      removeAllChildren();
}

bool GMutableTreeNode::getAllowsChildren () const
{
   return allowsChildren;
}

void GMutableTreeNode::setUserObject ( GObject* newUserObject, bool autoDelete )
{
   if (userObject != userObject)
   {
      if (autoDeleteUserObject)
         delete userObject;
      userObject = newUserObject;
   }
   autoDeleteUserObject = autoDelete;
}

void GMutableTreeNode::removeFromParent ()
{
   GMutableTreeNode* parent = dynamic_cast<GMutableTreeNode*>(getParent());
   if (parent != null)
      parent->remove(this, false); // Remove from parent, but don't destroy our self!
}

void GMutableTreeNode::removeAllChildren () 
{
	for (int i=getChildCount()-1; i>=0; i--)
      remove(i, true);
}

void GMutableTreeNode::add ( GMutableTreeNode* child, bool autoDelete ) 
{
   int index = getChildCount();
   if (child != null && child->getParent() == this)
      insert(child, index - 1, autoDelete);
   else
      insert(child, index, autoDelete);
}

bool GMutableTreeNode::isNodeAncestor ( const GTreeNode& anotherNode ) const
{
   GTreeNode* ancestor = const_cast<GMutableTreeNode*>(this);
   do {
      if (ancestor == &anotherNode)
         return true;
      ancestor = ancestor->getParent();
   } while (ancestor != null);
   return false;
}

bool GMutableTreeNode::isNodeDescendant ( GMutableTreeNode& anotherNode ) const
{
   return anotherNode.isNodeAncestor(*this);
}

GTreeNode* GMutableTreeNode::getSharedAncestor ( GMutableTreeNode& aNode )
{
   if (&aNode == this)
      return this;

   int level1 = getLevel();
   int level2 = aNode.getLevel();
   GTreeNode* node1;
   GTreeNode* node2;
   int diff;
   if (level2 > level1) 
   {
      diff = level2 - level1;
      node1 = &aNode;
      node2 = this;
   } 
   else 
   {
      diff = level1 - level2;
      node1 = this;
      node2 = &aNode;
   }

   // Go up the tree until the nodes are at the same level.
   while (diff > 0) 
   {
      node1 = node1->getParent();
      diff--;
   }

   // Move up the tree until we find a common ancestor. Since we know
   // that both nodes are at the same level, we won't cross paths
   // unknowingly (if there is a common ancestor, both nodes hit it in
   // the same iteration).

   do {
      if (node1 == node2)
         return node1;
      node1 = node1->getParent();
      node2 = node2->getParent();
   } while (node1 != null); // Only need to check one -- they're at the same level so if one is null, the other is so too.

   if (node1 != null || node2 != null)
      gthrow_(GIllegalStateException("nodes should be null"));

   return null;
}

bool GMutableTreeNode::isNodeRelated ( const GMutableTreeNode& aNode ) const 
{
   GTreeNode& selfRoot = const_cast<GMutableTreeNode*>(this)->getRoot();
   GTreeNode& nodeRoot = const_cast<GMutableTreeNode&>(aNode).getRoot();
   return &selfRoot == &nodeRoot;
}

int GMutableTreeNode::getLevel () const 
{
   int levels = 0;
   GTreeNode* ancestor = const_cast<GMutableTreeNode*>(this);
   while ((ancestor = ancestor->getParent()) != null)
      levels++;
   return levels;
}

GTreeNode& GMutableTreeNode::getRoot () 
{
   GTreeNode* ancestor = this;
   GTreeNode* previous;
   do {
      previous = ancestor;
      ancestor = ancestor->getParent();
   } while (ancestor != null);
   return *previous;
}

bool GMutableTreeNode::isRoot () const 
{
   GTreeNode* self = const_cast<GMutableTreeNode*>(this);
   return self->getParent() == null;
}

GMutableTreeNode* GMutableTreeNode::getNextNode () 
{
   if (getChildCount() == 0) 
   {
      // No children, so look for nextSibling.
      GMutableTreeNode* nextSibling = getNextSibling();
      if (nextSibling == null) 
      {
         GMutableTreeNode* aNode = dynamic_cast<GMutableTreeNode*>(getParent());
         do {
            if (aNode == null) 
               return null;
            nextSibling = aNode->getNextSibling();
            if (nextSibling != null) 
               return nextSibling;
            aNode = dynamic_cast<GMutableTreeNode*>(aNode->getParent());
         } while(true);
      } 
      else 
      {
         return nextSibling;
      }
   } 
   else 
   {
      return &dynamic_cast<GMutableTreeNode&>(getChildAt(0));
   }
}

GMutableTreeNode* GMutableTreeNode::getPreviousNode () 
{
   GMutableTreeNode* myParent = dynamic_cast<GMutableTreeNode*>(getParent());
   if (myParent == null)
      return null;

   GMutableTreeNode* previousSibling = getPreviousSibling();
   if (previousSibling != null) 
   {
      if (previousSibling->getChildCount() == 0)
         return previousSibling;
      else
         return &previousSibling->getLastLeaf();
   } 
   else 
   {
      return myParent;
   }
}

GTreeNode& GMutableTreeNode::getFirstChild () 
{
	if (getChildCount() == 0)
	   gthrow_(GNoSuchElementException("node has no children"));
	return getChildAt(0);
}

GTreeNode& GMutableTreeNode::getLastChild () 
{
   if (getChildCount() == 0)
      gthrow_(GNoSuchElementException("node has no children"));
   int index = getChildCount() - 1;
   return getChildAt(index);
}

GTreeNode* GMutableTreeNode::getChildAfter ( GTreeNode& aChild ) 
{
   int index = getIndex(aChild);
   if (index < 0)
      gthrow_(GIllegalArgumentException("node is not a child"));
   if (index >= getChildCount() - 1)
      return null;
   return &getChildAt(index + 1);
}

GTreeNode* GMutableTreeNode::getChildBefore ( GTreeNode& aChild ) 
{
   int index = getIndex(aChild);
   if (index < 0)
      gthrow_(GIllegalArgumentException("argument is not a child"));
   if (index <= 0)
      return null;
   return &getChildAt(index - 1);
}

bool GMutableTreeNode::isNodeSibling ( const GTreeNode& anotherNode ) const
{
   if (&anotherNode == this)
      return true;
   GTreeNode* myParent = const_cast<GMutableTreeNode*>(this)->getParent();
   bool ret = myParent != null && myParent == const_cast<GTreeNode&>(anotherNode).getParent();
   if (ret && !myParent->isNodeChild(anotherNode))
      gthrow_(GIllegalStateException("sibling has different parent"));
   return ret;
}

int GMutableTreeNode::getSiblingCount () const
{
   const GTreeNode* myParent = const_cast<GMutableTreeNode*>(this)->getParent();
   if (myParent == null)
      return 1;
   return myParent->getChildCount();
}

GMutableTreeNode* GMutableTreeNode::getNextSibling () 
{
   GMutableTreeNode* myParent = dynamic_cast<GMutableTreeNode*>(getParent());
   GMutableTreeNode* retval;
   if (myParent == null)
      retval = null;
   else
      retval = dynamic_cast<GMutableTreeNode*>(myParent->getChildAfter(*this));
   if (retval != null && !isNodeSibling(*retval))
      gthrow_(GIllegalStateException("child of parent is not a sibling"));
   return retval;
}

GMutableTreeNode* GMutableTreeNode::getPreviousSibling () 
{
   GMutableTreeNode* myParent = dynamic_cast<GMutableTreeNode*>(getParent());
   GMutableTreeNode* retval;
   if (myParent == null)
      retval = null;
   else
      retval = dynamic_cast<GMutableTreeNode*>(myParent->getChildBefore(*this));
   if (retval != null && !isNodeSibling(*retval))
      gthrow_(GIllegalStateException("child of parent is not a sibling"));
   return retval;
}

GMutableTreeNode& GMutableTreeNode::getFirstLeaf () 
{
   GMutableTreeNode* node = this;
   while (!node->isLeaf())
      node = dynamic_cast<GMutableTreeNode*>(&node->getFirstChild());
   return *node;
}

GMutableTreeNode& GMutableTreeNode::getLastLeaf () 
{
   GMutableTreeNode* node = this;
   while (!node->isLeaf())
      node = dynamic_cast<GMutableTreeNode*>(&node->getLastChild());
   return *node;
}

GMutableTreeNode* GMutableTreeNode::getNextLeaf () 
{
   GMutableTreeNode* myParent = dynamic_cast<GMutableTreeNode*>(getParent());
   if (myParent == null)
      return null;

   GMutableTreeNode* nextSibling = getNextSibling();
   if (nextSibling != null)
      return &nextSibling->getFirstLeaf();

   return myParent->getNextLeaf(); // Tail recursion.
}

GMutableTreeNode* GMutableTreeNode::getPreviousLeaf () 
{
   GMutableTreeNode* myParent = dynamic_cast<GMutableTreeNode*>(getParent());
   if (myParent == null)
      return null;

   GMutableTreeNode* previousSibling = getPreviousSibling();
   if (previousSibling != null)
      return &previousSibling->getLastLeaf();

   return myParent->getPreviousLeaf(); // Tail recursion.
}

GString GMutableTreeNode::toString () const 
{
   if (userObject == null)
      return GString::Empty;
   return userObject->toString();
}

const GIcon* GMutableTreeNode::getIconClosed () const
{
   return iconClosed;
}

const GIcon* GMutableTreeNode::getIconOpened () const
{
   return iconOpened;
}

void GMutableTreeNode::setIconClosed ( const GString& iconID )
{
   const GIcon* icon = GIcon::GetIcon(iconID);
   setIconClosed(const_cast<GIcon*>(icon), false);
}

void GMutableTreeNode::setIconClosed ( class GIcon* icon, bool autoDelete )
{
   if (autoDeleteIconClosed)
      delete iconClosed;
   iconClosed = icon;
   autoDeleteIconClosed = autoDelete;
}

void GMutableTreeNode::setIconOpened ( const GString& iconID )
{
   const GIcon* icon = GIcon::GetIcon(iconID);
   setIconOpened(const_cast<GIcon*>(icon), false);
}

void GMutableTreeNode::setIconOpened ( class GIcon* icon, bool autoDelete )
{
   if (autoDeleteIconOpened)
      delete iconOpened;
   iconOpened = icon;
   autoDeleteIconOpened = autoDelete;
}
