/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_LISTSELECTIONLISTENER
#define __GLIB_LISTSELECTIONLISTENER

#include "glib/gui/GAbstractListBox.h"

/**
 * This interface is used to declare the methods that must be
 * implemented by every class that want to be able to register
 * it self as a list box selection listener.
 *
 * @author  Leif Erik Larsen
 * @since   2001.01.25
 * @see     GListBox#addListBoxSelectionListener
 */
class GListSelectionListener
{
   public:

      /**
       * This method will be called synchronously just <i>after</i> the
       * current selection of the specified list box has been changed,
       * regardless of how and by whom that caused the selection change
       * event.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.25
       */
      virtual void onListBoxSelectionChanged ( GAbstractListBox& listb ) = 0;

      /**
       * This method will be called synchronously just <i>after</i> the
       * default handling of the <i>onKey()</i> event, but only if
       * the key was a character (not arrow keys) and it caused the current 
       * selection to change. It will be called <i>after</i> 
       * {@link #onListBoxSelectionChanged} has been called and returned.
       *
       * Most clients will do nothing but return when this method is called,
       * but it is needed in cases when the client code (the listener) need
       * to take some special actions when a listbox selection is caused by
       * a character being typed. This is true for instance by the drive
       * selection dialog of Larsen Commander. If the user types a drive
       * letter in the drives-listbox then that drive is activated directly
       * without waiting for the user to press the OK-button. This behaviour
       * is easy implemented thanks to this interface method.
       *
       * @author  Leif Erik Larsen
       * @since   2001.04.09
       */
      virtual void onListBoxSelectionChangedByCharKey ( GAbstractListBox& listb ) = 0;

      /**
       * This method will be called when the user has double-clicked on
       * the current selected item of the specified list box.
       *
       * @author  Leif Erik Larsen
       * @since   2001.01.25
       */
      virtual void onListBoxSelectionDblClick ( GAbstractListBox& listb ) = 0;
};

#endif

