/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GWorkerThread.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/event/GDialogMessage.h"
#include "glib/primitives/GBoolean.h"
#include "glib/exceptions/GThreadStartException.h"
#include "glib/exceptions/GIllegalStateException.h"
#include "glib/GProgram.h"

const GString GWorkerThread::PROGRESS_TIMER_ID = "WorkerThreadProgressTimer";
const GString GWorkerThread::STAY_INVISIBLE_TIMER_ID = "WorkerThreadStayInvisibleTimer";
int GWorkerThread::InstanceCounter = 0;

GWorkerThread::GWorkerThread ( const GString& dlgResourceID,
                               int monitorUpdtMillis, 
                               GWorkerThreadAdapter* adapter,
                               bool autoDeleteAdapter,
                               const GString& threadName )
              :timeUsedWaitingForUserAnswersMillis(0),
               timeStartMillis(0),
               updtMillis(monitorUpdtMillis),
               success(false), // Until the opposite has been proven
               stopIsRequested(false),
               executing(false),
               workerIsRunning(false),
               isInMsgBox(false),
               msgBoxHasBeenShownAtLeastOnce(false),
               parentWinWasActiveUponStartWorker(false),
               monitorDlgWasActiveUponEndWorker(false),
               noExceptionMsgBox(false),
               bckThread(threadName, *this),
               mproc(*this),
               adapter(adapter),
               autoDelAdptr(autoDeleteAdapter),
               stayInvisibleTimerMillis(0),
               monitorDlg(null),
               topFrame(null)
{
   GProgram& prg = GProgram::GetProgram();
   monitorFrame = prg.makeDialog(dlgResourceID, &mproc);
}

GWorkerThread::~GWorkerThread ()
{
   if (autoDelAdptr)
      delete adapter;
}

GWorkerThread::BckThread::BckThread ( const GString& threadName, GWorkerThread& owner )
                         :GThread(GString(threadName, GVArgs(++InstanceCounter))),
                          owner(owner)
{
}

GWorkerThread::BckThread::~BckThread ()
{
}

/**
 * This is the low-level entry point of the background thread.
 * From here we will call the user defined entry point.
 */
void GWorkerThread::BckThread::run ()
{
   GDialogMessage um1(*owner.monitorDlg, GDialogMessageHandler::GM_USER, "WORKER_THREAD_HAS_STARTED");
   owner.sendMessageToGUI(um1);

   bool ok = true;

   try {
      owner.timeStartMillis = GSystem::CurrentTimeMillis();
      owner.workerIsRunning = true;
      owner.runTheWorkerThread(owner);
      owner.workerIsRunning = false;
   } catch (GException& e) {
      owner.workerIsRunning = false;
      ok = false;
      owner.workerThreadExceptionMessage = GString("Unhandled exception in worker thread: %s", GVArgs(e.toString()));
      owner.workerThreadExceptionStackTrace = e.getStackTrace(e.toString());
      GLog::Log(this, owner.workerThreadExceptionStackTrace);
      if (!owner.noExceptionMsgBox)
         owner.showWorkerMessageBox(owner.workerThreadExceptionMessage, GMessageBox::TYPE_ERROR, "Od", GString::Empty, false, owner.workerThreadExceptionStackTrace);
   } catch (std::exception& e) {
      owner.workerIsRunning = false;
      ok = false;
      owner.workerThreadExceptionMessage = GString("Unhandled exception in worker thread: %s", GVArgs(e.what()));
      GLog::Log(this, owner.workerThreadExceptionMessage);
      if (!owner.noExceptionMsgBox)
         owner.showWorkerMessageBox(owner.workerThreadExceptionMessage, GMessageBox::TYPE_ERROR);
   } catch (...) {
      owner.workerIsRunning = false;
      ok = false;
      owner.workerThreadExceptionMessage = GString("Unhandled exception of unknown type in worker thread.");
      GLog::Log(this, owner.workerThreadExceptionMessage);
      if (!owner.noExceptionMsgBox)
         owner.showWorkerMessageBox(owner.workerThreadExceptionMessage, GMessageBox::TYPE_ERROR);
   }

   if (!owner.isStopRequested()) // If thread has finished automatically.
      owner.success = ok;
   else
      owner.success = (ok && owner.success);

   // This will dismiss the progress dialog.
   GString successStr = GBoolean::GetBooleanStr(owner.success);
   GDialogMessage um2(*owner.monitorDlg, GDialogMessageHandler::GM_USER, "WORKER_THREAD_HAS_FINISHED", successStr);
   owner.sendMessageToGUI(um2);
}

GWorkerThread::MsgProc::MsgProc ( GWorkerThread& wthread )
                       :wthread(wthread)
{
}

bool GWorkerThread::MsgProc::handleDialogMessage ( GDialogMessage& msg )
{
   return wthread.handleMsgProcMessage(msg);
}

GWorkerThread::MessageBoxInfo::MessageBoxInfo ( const GString& msg,
                                                GMessageBox::Type type,
                                                const GString& flags,
                                                const GString& title,
                                                bool monoFont,
                                                const GString& userText1,
                                                const GString& userText2,
                                                const GString& userText3 )
                              :msg(msg),
                                 type(type),
                                 flags(flags),
                                 title(title),
                                 monoFont(monoFont),
                                 userText1(userText1),
                                 userText2(userText2),
                                 userText3(userText3),
                                 answer(GMessageBox::IDTIMEOUT)
{
}

GWorkerThread::MessageBoxInfo::~MessageBoxInfo ()
{
}

GWorkerThread::UserMsgData::UserMsgData ( const GString& id,
                                          GObject* userData )
                           :id(id),
                            userData(userData)
{
}

GWorkerThread::UserMsgData::~UserMsgData ()
{
}

bool GWorkerThread::workModal ( GWindow& parentWin,
                                int stayInvisibleTimerMillis_,
                                bool noExceptionMsgBox_ )
{
   if (bckThread.isRunning())
      gthrow_(GIllegalStateException("The worker thread is already running!"));

   if (GThread::GetCurrentThreadID() == bckThread.getThreadID())
      gthrow_(GIllegalStateException("GWorkerThread::workModal() must not be called by the GUI Thread only!"));

   GFrameWindow* tf = null;
   
   try {

      executing = true;
      topFrame = null;
      success = false; // Until the opposite has been proven
      noExceptionMsgBox = noExceptionMsgBox_;
      workerThreadExceptionMessage = "";
      workerThreadExceptionStackTrace = "";
      stopIsRequested = false;
      monitorDlg = null;
      msgBoxHasBeenShownAtLeastOnce = false;
      parentWinWasActiveUponStartWorker = parentWin.isActive();
      monitorDlgWasActiveUponEndWorker = false;
      timeUsedWaitingForUserAnswersMillis = 0;
      timeStartMillis = 0;
      stayInvisibleTimerMillis = stayInvisibleTimerMillis_;

      // Try to prevent the "flashing" of the parent window frame, until 
      // we actually shows the progress dialog or some modal message box.
      if (stayInvisibleTimerMillis > 0)
      {
         GWindow& topWin = parentWin.getTopLevelWindow();
         topFrame = dynamic_cast<GFrameWindow*>(&topWin);
         if (topFrame != null)
            topFrame->setKeepCaptionActive(true);
      }

      // Remember the top frame even if we set "topFrame" null somewhere else.
      tf = topFrame; 

      // ---
      monitorFrame->executeModal(&parentWin);
      monitorDlg = null;
      executing = false;

      // Restore the "keep caption active" mode.
      if (tf != null)
      {
         tf->setKeepCaptionActive(false);
         bool force = msgBoxHasBeenShownAtLeastOnce || monitorDlgWasActiveUponEndWorker;
         tf->setActive(force);
         topFrame = null;
      }

      // Wait until the background thread has actually terminated.
      while (bckThread.isRunning())
         GThread::Sleep(25);
   } catch (...) {
      executing = false;
      if (tf != null)
         tf->setKeepCaptionActive(false);
      topFrame = null;
      throw;
   }

   return success;
}

bool GWorkerThread::isExecuting () const
{
   return executing;
}

GDialogPanel* GWorkerThread::getMonitorDialog ()
{
   return monitorDlg;
}

void GWorkerThread::sendMessageToGUI ( GDialogMessage& msg )
{
   if (GThread::GetCurrentThreadID() != bckThread.getThreadID())
      gthrow_(GIllegalStateException("GWorkerThread::sendMessageToGUI() must be called by the Worker Thread only!"));

   bckThread.sendGuiUserMessage(*monitorDlg, msg);
}

void GWorkerThread::sendUserMessageToMonitor ( const GString& id, GObject* userData )
{
   UserMsgData umd(id, userData);
   GDialogMessage um(*monitorDlg, GDialogMessageHandler::GM_USER, "WORKER_THREAD_USER_EVENT", &umd);
   sendMessageToGUI(um);
}

ulonglong GWorkerThread::getTimeMillisSinceStart ( bool exclusiveMsgBox ) const
{
   if (timeStartMillis == 0)
      return 0;
   ulonglong now = GSystem::CurrentTimeMillis();
   ulonglong ret = now - timeStartMillis;
   if (exclusiveMsgBox)
      ret -= timeUsedWaitingForUserAnswersMillis;
   return ret;
}

ulonglong GWorkerThread::getTimeMillisUsedWaitingForUserAnswers () const
{
   return timeUsedWaitingForUserAnswersMillis;
}

void GWorkerThread::requestStop ( bool success )
{
   if (GThread::GetCurrentThreadID() == bckThread.getThreadID())
      gthrow_(GIllegalStateException("GWorkerThread.requestStop() must not be called by the Worker Thread!"));
   if (!stopIsRequested) // Accept only one single stop request!
   {
      this->success = success;
      stopIsRequested = true;
      onWorkerThreadRequestStop(*this, *monitorDlg, success);
   }
}

bool GWorkerThread::isStopRequested () const
{
   return stopIsRequested;
}

bool GWorkerThread::isSuccess () const
{
   return success;
}

const GString& GWorkerThread::getWorkerThreadExceptionMessage () const
{
   return workerThreadExceptionMessage;
}

const GString& GWorkerThread::getWorkerThreadExceptionStackTrace () const
{
   return workerThreadExceptionStackTrace;
}

void GWorkerThread::updateMonitor ()
{
   if (GThread::GetCurrentThreadID() != bckThread.getThreadID())
      gthrow_(GIllegalStateException("GWorkerThread.showWorkerMessageBox() must be called by the Worker Thread only!"));
   GDialogMessage um(*monitorDlg, GDialogMessageHandler::GM_USER, "WORKER_THREAD_UPDATEMONITOR");
   sendMessageToGUI(um);
}

GMessageBox::Answer GWorkerThread::showWorkerMessageBox ( const GString& msg, 
                                                          GMessageBox::Type type, 
                                                          const GString& flags, 
                                                          const GString& title, 
                                                          bool monoFont,
                                                          const GString& userText1,
                                                          const GString& userText2,
                                                          const GString& userText3 )
{
   if (GThread::GetCurrentThreadID() != bckThread.getThreadID())
      gthrow_(GIllegalStateException("GWorkerThread.showMessageBox() must be called by the Worker Thread only!"));
   isInMsgBox = true;
   try {
      ulonglong msStart = GSystem::CurrentTimeMillis();
      MessageBoxInfo mi(msg, type, flags, title, monoFont, userText1, userText2, userText3);
      GDialogMessage um(*monitorDlg, GDialogMessageHandler::GM_USER, "WORKER_THREAD_MESSAGEBOX", &mi);
      sendMessageToGUI(um);
      ulonglong msNow = GSystem::CurrentTimeMillis();
      timeUsedWaitingForUserAnswersMillis += msNow - msStart;
      isInMsgBox = false;
      return mi.answer;
   } catch (...) {
      isInMsgBox = false;
      throw;
   }
}

void GWorkerThread::onWorkerThreadCtrlChanged ( GWorkerThread& worker,
                                                GDialogPanel& monitor,
                                                const GString& compID,
                                                const GString& newValue )
{
   if (adapter != null)
      adapter->onWorkerThreadCtrlChanged(worker, monitor, compID, newValue);
}

void GWorkerThread::onWorkerThreadDismissDialog ( GWorkerThread& worker,
                                                  GDialogPanel& monitor )
{
   if (adapter != null)
      adapter->onWorkerThreadDismissDialog(worker, monitor);
}

void GWorkerThread::onWorkerThreadFocusLoss ( GWorkerThread& worker,
                                              GDialogPanel& monitor,
                                              const GString& focusLossID,
                                              const GString& focusSetID )
{
   if (adapter != null)
      adapter->onWorkerThreadFocusLoss(worker, monitor, focusLossID, focusSetID);
}

void GWorkerThread::onWorkerThreadFocusSet ( GWorkerThread& worker,
                                             GDialogPanel& monitor,
                                             const GString& focusSetID,
                                             const GString& focusLossID )
{
   if (adapter != null)
      adapter->onWorkerThreadFocusSet(worker, monitor, focusSetID, focusLossID);
}

void GWorkerThread::onWorkerThreadInitDialog ( GWorkerThread& worker,
                                               class GDialogPanel& monitor )
{
   if (adapter != null)
      adapter->onWorkerThreadInitDialog(worker, monitor);
}

void GWorkerThread::onWorkerThreadRequestStop ( GWorkerThread& worker,
                                                GDialogPanel& monitor,
                                                bool success )
{
   if (adapter != null)
      adapter->onWorkerThreadRequestStop(worker, monitor, success);
}

void GWorkerThread::onWorkerThreadUpdateMonitor ( GWorkerThread& worker,
                                                  GDialogPanel& monitor )
{
   if (adapter != null)
      adapter->onWorkerThreadUpdateMonitor(worker, monitor);
}

void GWorkerThread::onWorkerThreadCommand ( GWorkerThread& worker,
                                            GDialogPanel& monitor,
                                            const GString& cmd )
{
   if (adapter != null)
      adapter->onWorkerThreadCommand(worker, monitor, cmd);
}

void GWorkerThread::onWorkerThreadHasStarted ( GWorkerThread& worker,
                                               GDialogPanel& monitor )
{
   if (adapter != null)
      adapter->onWorkerThreadHasStarted(worker, monitor);
}

void GWorkerThread::onWorkerThreadHasFinished ( GWorkerThread& worker,
                                                GDialogPanel& monitor,
                                                bool success )
{
   if (adapter != null)
      adapter->onWorkerThreadHasFinished(worker, monitor, success);
}

void GWorkerThread::onWorkerThreadUserEvent ( GWorkerThread& worker,
                                              GDialogPanel& monitor,
                                              const GString& msgID,
                                              GObject* userParam )
{
   if (adapter != null)
      adapter->onWorkerThreadUserEvent(worker, monitor, msgID, userParam);
}

void GWorkerThread::runTheWorkerThread ( GWorkerThread& worker )
{
   if (adapter != null)
      adapter->runTheWorkerThread(worker);
}

void GWorkerThread::showTheProgressDialog ()
{
   if (topFrame == null)
   {
      // The monitor dialog has already been displayed, or we should not
      // display it for some reason. E.g. because a modal message box
      // is currently visible.
      return;
   }

   GFrameWindow* tf = topFrame;
   topFrame = null;
   tf->setKeepCaptionActive(false);
   GDialogFrame& dlgFrame = monitorDlg->getOwnerFrame();
   dlgFrame.setVisible(true);
   bool force = msgBoxHasBeenShownAtLeastOnce || parentWinWasActiveUponStartWorker;
   dlgFrame.setActive(force);

   // Make the title bar of our parent window paint with "deselected" color,
   // since the progress dialog is now active instead.
   tf->sendMessage(WM_ACTIVATE, GWindowMessage::Param1(FALSE), GWindowMessage::Param2(NULL));
}

GMessageBox::Answer GWorkerThread::showGuiMessageBoxImpl ( GWindow& ownerWin, 
                                                           const MessageBoxInfo& mi )
{
   return ownerWin.showMessageBox(mi.msg, mi.type, mi.flags, 
                                  mi.title, mi.monoFont, 
                                  mi.userText1, mi.userText2, mi.userText3);
}

bool GWorkerThread::handleMsgProcMessage ( GDialogMessage& msg )
{
   GDialogPanel& dlg = msg.getDialog();
   switch (msg.getID())
   {
      case GDialogMessageHandler::GM_INITDIALOG:
      {
         onWorkerThreadInitDialog(*this, dlg);
         monitorDlg = &dlg;
         bckThread.start(); // Start the background worker thread.
         if (updtMillis > 0)
            dlg.startTimer(PROGRESS_TIMER_ID, updtMillis);
         if (stayInvisibleTimerMillis > 0)
         {
            GDialogFrame& dlgFrame = dlg.getOwnerFrame();
            dlgFrame.setStayInvisible(true);
            if (GLog::Filter(GLog::TEST))
               GLog::Log(this, "Starting timer: %s, millis=%d", GVArgs(STAY_INVISIBLE_TIMER_ID).add(stayInvisibleTimerMillis));
            dlg.startTimer(STAY_INVISIBLE_TIMER_ID, stayInvisibleTimerMillis);
         }
         return true;
      }

      case GDialogMessageHandler::GM_DISMISSDIALOG:
      {
         // Stop the timer that generates the GM_TIMER message.
         if (updtMillis > 0)
            dlg.stopTimer(PROGRESS_TIMER_ID);
         // Stop the "stay invisible" timer.
         if (stayInvisibleTimerMillis > 0)
            dlg.stopTimer(STAY_INVISIBLE_TIMER_ID);
         onWorkerThreadDismissDialog(*this, dlg);
         return true;
      }

      case GDialogMessageHandler::GM_TIMER:
      {
         GString timerID = msg.getParam1String();
         if (timerID == STAY_INVISIBLE_TIMER_ID)
         {
            if (GLog::Filter(GLog::TEST))
               GLog::Log(this, "Received timer: %s", GVArgs(STAY_INVISIBLE_TIMER_ID));
            dlg.stopTimer(STAY_INVISIBLE_TIMER_ID);
            stayInvisibleTimerMillis = 0;
            showTheProgressDialog();
         }
         else
         if (timerID == PROGRESS_TIMER_ID)
         {
            if (updtMillis > 0 &&
                workerIsRunning && // Ignore all timer events after the worker thread has exited.
                !isInMsgBox) // Don't update progress bar while inside "showMessageBox()".
            {
               onWorkerThreadUpdateMonitor(*this, dlg);
            }
         }
         return true;
      }

      case GDialogMessageHandler::GM_CTRLCHANGED:
      {
         const GString& compID = msg.getParam1String();
         const GString& compVal = msg.getParam2String();
         onWorkerThreadCtrlChanged(*this, dlg, compID, compVal);
         return true;
      }

      case GDialogMessageHandler::GM_FOCUSSET:
      {
         const GString& focusSet = msg.getParam1String();
         const GString& focusLoss = msg.getParam2String();
         onWorkerThreadFocusSet(*this, dlg, focusSet, focusLoss);
         return true;
      }

      case GDialogMessageHandler::GM_FOCUSLOSS:
      {
         const GString& focusLoss = msg.getParam1String();
         const GString& focusSet = msg.getParam2String();
         onWorkerThreadFocusLoss(*this, dlg, focusLoss, focusSet);
         return true;
      }

      case GDialogMessageHandler::GM_COMMAND:
      {
         const GString& cmd = msg.getParam1String();
         onWorkerThreadCommand(*this, dlg, cmd);
         return true;
      }

      case GDialogMessageHandler::GM_USER:
      {
         GString id = msg.getParam1String();
         if (id == "WORKER_THREAD_HAS_STARTED")
         {
            onWorkerThreadHasStarted(*this, dlg);
            return true;
         }
         else
         if (id == "WORKER_THREAD_HAS_FINISHED")
         {
            // Update the progress bar for the last time, just to make sure
            // that it is updated with something like "100%".
            if (updtMillis > 0)
               onWorkerThreadUpdateMonitor(*this, dlg);
            // ---
            monitorDlgWasActiveUponEndWorker = monitorFrame->isVisible() && monitorFrame->isActive();
            // Dismiss the worker thread progress bar.
            bool success = msg.getParam2Bool();
            onWorkerThreadHasFinished(*this, dlg, success);
            dlg.dismiss();
            return true;
         }
         else
         if (id == "WORKER_THREAD_USER_EVENT")
         {
            UserMsgData* umd = (UserMsgData*) msg.getParam2();
            onWorkerThreadUserEvent(*this, dlg, umd->id, umd->userData);
            return true;
         }
         else
         if (id == "WORKER_THREAD_UPDATEMONITOR")
         {
            onWorkerThreadUpdateMonitor(*this, dlg);
            return true;
         }
         else
         if (id == "WORKER_THREAD_MESSAGEBOX")
         {
            // Temporarily set the "topFrame" to null so that the monitor
            // dialog isn't displayed while we waits for the user to 
            // respond to the message box.
            GFrameWindow* tf = topFrame;
            topFrame = null;

            // Make the title bar of the topFrame "inactive" now that we 
            // actually displays a modal sub-dialog. We wanted to keep it 
            // "active" to prevent flashing in case the worker thread finish
            // before the progress dialog is ever displayed.
            if (tf != null)
               tf->setKeepCaptionActive(false); 

            // Temporarily stop the timer while we are showing the 
            // message box. In case the message box is shown before the 
            // progress dialog has been visible.
            GDialogFrame& frame = dlg.getOwnerFrame();
            if (!frame.isVisible() && stayInvisibleTimerMillis > 0)
               dlg.stopTimer(STAY_INVISIBLE_TIMER_ID);

            // Show the message box.
            GWindow* owner;
            if (tf == null || frame.isVisible())
               owner = &dlg;
            else
               owner = tf;
            msgBoxHasBeenShownAtLeastOnce = true;
            MessageBoxInfo* mi = (MessageBoxInfo*) msg.getParam2();
            GMessageBox::Answer answ = showGuiMessageBoxImpl(*owner, *mi);
            mi->answer = answ;

            // Must re-disable the top-frame if the message box was shown 
            // on it, because the modal message loop of the message box
            // has now enabled it. We wants to keep it disabled until 
            // our progress dialog has also been dismissed.
            if (owner == tf)
            {
               if (tf != null)
                  tf->setEnabled(false);
               // Force focus away from the topFrame. This is in order to 
               // prevent user from being able to perform any action on 
               // the disabled topFrame.
               GWindow::GetHiddenDummy().setActive(true);
            }

            // Reset the timer so that there will be a new full time 
            // until the progress dialog is considered to be displayed.
            if (!frame.isVisible() && stayInvisibleTimerMillis > 0)
               dlg.startTimer(STAY_INVISIBLE_TIMER_ID, stayInvisibleTimerMillis);

            // Restore the "topFrame" so that the monitor dialog gets 
            // displayed on the next STAY_INVISIBLE_TIMER_ID timer event.
            topFrame = tf;
            return true;
         }
         return false;
      }

      default:
         return false;
   }
}
