/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#ifndef __GLIB_WINDOW
#define __GLIB_WINDOW

#include "glib/primitives/GPoint.h"
#include "glib/primitives/GDimension.h"
#include "glib/primitives/GRectangle.h"
#include "glib/primitives/GInsets.h"
#include "glib/gui/GWindowClass.h"
#include "glib/gui/GMessageBox.h"
#include "glib/gui/GColor.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/GTooltip.h"
#include "glib/gui/event/GWindowMessage.h"
#include "glib/util/GKeyBag.h"
#include "glib/util/GChangeableValueContainer.h"
#include "glib/util/GSystemYConverter.h"

class GWindow : public GChangeableValueContainer, public GSystemYConverter
{
   friend class GWindowEntry;
   friend class GWindowClass;
   friend class GDialogPanel;
   friend class GDialogFrame;
   friend class GFrameWindow;

   public:
      typedef GWNDPROCENTRY WindowProc;
      typedef GRECT SysRect;

   private:
      bool wrapperOnly;
      GString name;
      mutable GString fullName;
      int nameCounter;
      GString text;
      GString tooltipText;
      class GResourceTable* resources;
      GObject* userData;
      bool autoDeleteUserData;
      GVector<class GMouseListener*>* mouseListeners;
      GVector<class GValueChangeListener*>* valueChangeListeners;
      WindowProc defaultMsgProc;
      const class GWindowClass& winClass;
      class GWindowEntry* winentry;
      HWND hWnd;
      GWindow* parentWin;
      GWindow* ownerWin;
      GTooltip::Position tooltipPos;
      int tooltipDelayShowMillis;
      int tooltipAutoHideMillis;
      GArray<GWindow> ownedModalDialogs;
      bool deleteParentWin;
      bool deleteOwnerWin;
      GKeyBag<GWindow> childrenByName;
      GArray<GWindow> childrenByIndex;
      GKeyBag<class GWindowTimerData>* activeTimersBySysID;
      class GDragDropHandler* dragDropHandler;
      bool autoDeleteDragDropHandler;
      bool caretIsOn;
      int caretXPos;
      int caretYPos;
      int caretWidth;
      int caretHeight;
      bool os2y;
      bool mouseIsCaptured;
      class GMenu* popupMenu;
      class GAccelResource* accelTable;
      bool closeWinOnDestroyObject;
      class GLayoutManager* layoutManager;
      class GBorder* border;
      bool deleteBorder;
      bool autoDeleteLayoutManager;
      GString layoutConstraints;
      GInsets insets;
      bool oily;
      bool focusable;
      bool layoutPosted;
      static bool AboutToHandleKey;
      static GWindow* SysDesktopWindow;
      static GWindow* HiddenDummy;

   protected:
      const GDimension* preferredSize;
      bool autoDeletePreferredSize;
      GDimension* minimumSize;
      GDimension* maximumSize;
      int winStyle2;

   public:
      GColor defaultBackgroundColor;
      GColor defaultForegroundColor;
      GString defaultFontNameSize;

   public:
      enum WS2_STYLES
      {
         WS2_NULL = 0x00000000,
         WS2_DEFAULTPAINT = 0x00000001,
         WS2_IGNORE_COLORS_PROFILE = 0x00000002,
         WS2_IGNORE_FONT_PROFILE = 0x00000004,
         WS2_IGNORE_COLORS_AND_FONT_PROFILE = 0x00000006, // WS2_IGNORE_COLORS_PROFILE | WS2_IGNORE_FONT_PROFILE
         WS2_DONT_LAYOUT_PARENT_ON_FONTNAMESIZE_CHANGED = 0x00000008,
         WS2_AUTO_PAINT_BACKGROUND = 0x00000010,
         WS2_DONT_CREATE_HWND = 0x00000020,
         WS2_OS2Y = 0x00000040,
         WS2_NO_RECURSIVE_KEY_HANDLING = 0x00000080,
         WS2_NOT_STATIC = 0x00000100,
         WS2_DONT_INCLUDE_VALUE_IN_CTRL_CHANGE_MSG = 0x00000200,
         WS2_DONT_SEND_CTRL_CHANGE_MSG = 0x00000400,
         WS2_AUTO_PAINT_TEXT = 0x00000800,
         WS2_AUTO_SHOW_TOOLTIP = 0x00001000,
         WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT = 0x00002000,
         WS2_HIDDEN = 0x00004000
      };

   public:
      enum MessageID
      {
         GM_ABSTRACTCOMMAND = WM_USER,
         GM_PRESPARAMCHANGED,
         GM_USERMESSAGE,
         GM_USERMESSAGE_FROM_THREAD,
         GM_ASYNC_PAINT,
         GM_CREATE_CARET,
         GM_ASYNC_GRAB_FOCUS
      };

      typedef int PresParamID;

   public:
      static const GString FontDimTemplateStr;
      static const GString DESKTOP_WINDOW_NAME;
      static const GColor SysDefBckColor;
      static const GColor SysDefFrgColor;
      static const GString SysDefFontNameSize;
      static const GString SysDefFixedFontNameSize;
      static const GString PrfItemName_BckColor;
      static const GString PrfItemName_FrgColor;
      static const GString PrfItemName_FontNameSize;

   public:
      enum LeaderStyle
      {
         LeaderStyle_DEFAULT,
         LeaderStyle_CLASSIC,
         LeaderStyle_LEFT,
         LeaderStyle_RIGHT,
         LeaderStyle_FANCY1
      };

   private:
      GWindow::LeaderStyle leaderStyle;
      static GWindow::LeaderStyle CommonLeaderStyle;

   public:
      explicit GWindow ( HWND win, bool os2y );

      explicit GWindow ( const GString& name, HWND win, int winStyle2 );

      explicit GWindow ( GWindow* parentWin = null,
                         const GString& name = GString::Empty,
                         const GString& constraints = GString::Empty,
                         int winStyle2 = 0,
                         const GColor& defBckColor = GGraphics::DUMMY_COLOR,
                         const GColor& defFrgColor = GGraphics::DUMMY_COLOR,
                         const GString& defFont = GString::Empty,
                         GWindow* ownerWin = null,
                         long winStyle = WS_VISIBLE,
                         const class GWindowClass& winClass = GWindowClass::GENERIC );

      explicit GWindow ( const GString& name,
                         const GString& constraints,
                         GWindow* parentWin,
                         GWindow* ownerWin = null,
                         long winStyle = WS_VISIBLE,
                         int winStyle2 = 0,
                         const class GWindowClass& winClass = GWindowClass::GENERIC,
                         const GColor& defBckColor = GGraphics::DUMMY_COLOR,
                         const GColor& defFrgColor = GGraphics::DUMMY_COLOR,
                         const GString& defFont = GString::Empty );

      virtual ~GWindow ();
 
   protected:
      GWindow ( const class GWindowClass& winClass,
                const GString& constraints,
                bool os2y );

      void init ( const GString& name = GString::Empty,
                  GWindow* parentWin = null,
                  GWindow* ownerWin = null,
                  long winStyle = WS_VISIBLE,
                  int winStyle2 = 0,
                  const GColor& defBckColor = GGraphics::DUMMY_COLOR,
                  const GColor& defFrgColor = GGraphics::DUMMY_COLOR,
                  const GString& defFont = GString::Empty );

   private:
      GWindow ( const GWindow& ); //!< Disable the copy constructor.
      void operator= ( const GWindow& ); //!< Disable the assignment operator.

   private:
      void addChildWindow ( GWindow& win );
      void clearInstanceVariables ();
      void createTheWindow ( GWindow* parentWin, GWindow* ownerWin, long winStyle );
      bool fireMouseMovedEvent ( const class GMouseEvent& ev );
      bool fireMousePressedEvent ( const class GMouseEvent& ev );
      bool fireMouseReleasedEvent ( const class GMouseEvent& ev );
      bool fireMouseClickedEvent ( const class GMouseEvent& ev );
      bool fireMouseDoubleClickedEvent ( const class GMouseEvent& ev );
      class GWindowTimerData* getTimerData ( const GString& timerID, bool throw_ );
      void initiateTheName ( const GString& name );
      void initiateTheObject ( bool postponeWinCrea, GWindow* parentWin, GWindow* ownerWin, long winStyle, int winStyle2 );
      void removeChildWindow ( GWindow& win );
      void createCaret_ ();
      void destroyCaret_ ();
      void setCaretPos_ ();

   protected:
      virtual void fireValueChangeListeners ();
      static GWindowMessage::Answer EXPENTRY GenericWindowProc ( HWND hWnd, GWindowMessage::IDType ulMsg, GWindowMessage::Param1 mp1, GWindowMessage::Param2 mp2 );
      GWindow* getFirstOwnedModalDialog ();
      class GWindowEntry* getWindowEntry () const;
      virtual void setAccelTableImpl ( GAccelResource* newAccel );
      void setDefaultMsgProc ( WindowProc msgProc );
      void setHWND ( HWND hWin );
      void setOwnerWindow ( GWindow* owner, bool autoDelete );
      virtual void updateKeybarCommands ( GWindow* focusWin );
      virtual bool __executeAbstractCommand ( class GAbstractCommand* /*cmd*/ );

   public:
      void addMouseListener ( class GMouseListener* l );
      void addValueChangeListener ( class GValueChangeListener* l );
      GWindowMessage::Answer callDefaultMsgProc ( GWindowMessage& msg );
      virtual bool captureMouse ( bool flag );
      virtual void changeValue ( const GString& newValue, bool notify = true );
      virtual bool containsFocusedWindow () const;
      virtual bool executeAbstractCommand ( class GAbstractCommand *cmd );
      virtual class GAccelItem* getAccelItemByCommandID ( const GString& commandID ) const;
      virtual class GAccelItem* getAccelItemByKeyName ( const GString& keyName );
      virtual class GAccelResource* getAccelTable () const;
      virtual GColor getBackgroundColor () const;
      virtual int getCaretHeight () const;
      virtual int getCaretWidth () const;
      virtual int getCaretXPos () const;
      virtual int getCaretYPos () const;
      virtual GWindow& getChildWindow ( int index );
      virtual GWindow& getChildWindow ( const GString& name );
      virtual GWindow* getChildWindowByLayoutConstraints ( const GString& constraints );
      virtual int getChildWindowCount () const;
      const GString& getClassName () const;
      virtual GString getDebugInfoString () const;
      virtual GString getDebugName () const;
      virtual GWindow* getFocusedWindow () const;
      virtual int getFontHeight () const;
      virtual GString getFontNameSize () const;
      virtual GColor getForegroundColor () const;
      virtual const GString& getFullName () const;
      int getHeightOfString ( const GString& str ) const;
      int getHeightOfString ( const char* str ) const;
      static GWindow& GetHiddenDummy ();
      HWND getHWND () const;
      virtual const GInsets& getInsets () const;
      virtual const GString& getLayoutConstraints () const;
      virtual class GLayoutManager* getLayoutManager ();
      virtual GWindow::LeaderStyle getLeaderStyle () const;
      virtual int getMaximumHeight () const;
      virtual GDimension getMaximumSize () const;
      virtual int getMaximumWidth () const;
      virtual int getMinimumHeight () const;
      virtual GDimension getMinimumSize () const;
      virtual int getMinimumWidth () const;
      virtual GString getName () const;
      virtual GWindow* getOwnerWindow () const;
      virtual GWindow* getParentWindow () const;
      virtual class GMenu* getPopupMenu ();
      virtual int getPreferredHeight () const;
      virtual GDimension getPreferredSize () const;
      virtual int getPreferredWidth () const;
      virtual class GResourceTable& getResourceTable () const;
      GDimension getStringDimension ( const GString& str ) const;
      GDimension getStringDimension ( const char* str ) const;
      static GWindow& GetSysDesktopWindow ();
      virtual GString getText () const;
      virtual int getTextLength () const;
      virtual int getTooltipAutoHideMillis () const;
      virtual int getTooltipDelayShowMillis () const;
      virtual GTooltip::Position getTooltipPosition () const;
      virtual GWindow& getTopLevelWindow ();
      virtual GObject* getUserData () const;
      virtual int getWidthOfString ( const GString& str ) const;
      virtual int getWindowID () const;
      virtual GRectangle getWindowFaceRect () const;
      static GWindow* GetWindowFromHandle ( HWND hwnd );
      virtual GPoint getWindowPos () const;
      virtual GPoint getWindowPosOnScreen () const;
      virtual GRectangle getWindowRect () const;
      virtual GDimension getWindowSize () const;
      virtual GString getWindowTreeString ( int level = 0 ) const;
      virtual void grabFocus ( bool force = false );
      virtual void grabFocusAsynchronously ();
      virtual bool hasFocus () const;
      virtual void invalidateAll ( bool inclChildren ) const;
      virtual void invalidateRect ( const GRectangle& rect ) const;
      static bool IsAboutToHandleKey ();
      virtual bool isActive () const;
      virtual bool isAlive () const;
      virtual bool isAnyWaitingMessages () const;
      virtual bool isCaretOn () const;
      virtual bool isDescendant ( const GWindow& parent ) const;
      virtual bool isEnabled () const;
      virtual bool isFlatButtonBorders () const;
      virtual bool isFocusCycleRoot () const;
      virtual bool isFocusable () const;
      virtual bool isMaximized () const;
      virtual bool isMinimized () const;
      virtual bool isMouseCapture () const;
      virtual bool isOily () const;
      virtual bool isOS2Y () const;
      virtual bool isParentOf ( HWND win ) const;
      virtual bool isParentOf ( GWindow *win ) const;
      virtual bool isPointInWindowRect ( int xpos, int ypos ) const;
      virtual bool isShowTooltip () const;
      virtual bool isTimerStarted ( const GString& timerID );
      virtual bool isVisible () const;
      virtual bool isVisiblePopupMenu () const;
      virtual void layout ();
      virtual void postCommand ( class GAbstractCommand* cmd );
      virtual void postLayout ();
      void postMessage ( int id, GWindowMessage::Param1 mp1 = 0, GWindowMessage::Param2 mp2 = 0 );
      void postUserMessage ( class GUserMessage* msg, bool autoDeleteMsg );
      void postUserMessage ( const GString& str1 );
      void postUserMessage ( const GString& str1, const GString& str2 );
      void postUserMessage ( const GString& str1, GObject* param2, bool autoDelParam2 );
      virtual void queryProfile ( const GString& sectName );
      virtual GString queryValue () const;
      virtual void removeMouseListener ( class GMouseListener* l );
      virtual void removeOwnedModalDialog ( GWindow& win );
      virtual void removeValueChangeListener ( class GValueChangeListener* l );
      virtual void resetTimer ( const GString& timerID );
      virtual void scrollWindow ( int hscroll, int vscroll, bool invalidate, const class GRectangle* scrollRect = null, const class GRectangle* clipRect = null );
      GWindowMessage::Answer sendMessage ( int id, GWindowMessage::Param1 mp1 = 0, GWindowMessage::Param2 mp2 = 0 );
      bool sendUserMessage ( class GUserMessage& msg );
      bool sendUserMessage ( GObject* param1 = null, GObject* param2 = null );
      bool sendUserMessage ( const GString& str1, GObject* param2 = null );
      bool sendUserMessage ( const GString& str1, const GString& str2 );
      void setAccelTable ( GAccelResource* newAccel );
      void setAccelTable ( const GString& accelID );
      virtual void setActive ( bool force = false );
      virtual void setBackgroundColor ( const GColor& color );
      virtual void setBorder ( GBorder* border, bool autoDelete );
      virtual void setCaretOn ( bool flag );
      virtual void setCaretPos ( int x, int y );
      virtual void setCaretSize ( int width, int height );
      virtual void setCommandEnableState ( const char* cmdID, bool state );
      virtual void setCommandToggleState ( const char* cmdID, bool state );
      static void SetCommonLeaderStyle ( GWindow::LeaderStyle style );
      virtual void setDefaultPaint ( bool defaultPaint );
      void setDragDropHandler ( class GDragDropHandler* handler, bool autoDelete );
      virtual void setEnabled ( bool flag = true, bool repaint = true );
      virtual void setFocusable ( bool b );
      virtual void setFontNameSize ( const GString& fontNameSize );
      virtual void setForegroundColor ( const GColor& color );
      virtual void setInsets ( class GInsets* ins, bool autoDelete );
      virtual void setLayoutConstraints ( const GString& lc, bool doLayout );
      virtual void setLayoutManager ( class GLayoutManager* lm, bool autoDelete );
      virtual void setLeaderStyle ( GWindow::LeaderStyle style );
      virtual void setMaximized ( bool flag = true );
      virtual void setMaximumSize ( const GDimension& size );
      virtual void setMinimized ( bool flag = true );
      virtual void setMinimumSize ( const GDimension& size );
      virtual void setOily ( bool flag );
      void setOS2Y ( bool flag );
      void addOwnedModalDialog ( GWindow& win );
      void setParentWindow ( GWindow* newParent );
      void setPopupMenu ( class GMenuResource* newMenu, bool useFancy = true );
      void setPopupMenu ( const char* menuID, bool useFancy = true );
      virtual void setPreferredSize ( const GDimension* size, bool autoDelete );
      virtual void setText ( const GString& text );
      virtual void setTooltipAutoHideMillis ( int value );
      virtual void setTooltipDelayShowMillis ( int value );
      virtual void setTooltipPosition ( GTooltip::Position pos );
      virtual void setUserData ( GObject* userData, bool autoDelete );
      virtual void setVisible ( bool flag = true );
      virtual void setVisiblePopupMenu ( bool flag, int xpos = 0, int ypos = 0 );
      virtual void setWindowBounds ( int xpos, int ypos, int width, int height );
      virtual void setWindowPos ( int xpos, int ypos );
      virtual void setWindowPosOnScreen ( int xpos, int ypos, bool autoAdjust = false );
      virtual void setWindowSize ( int width, int height );
      virtual void showCaret ( bool flag );
      GMessageBox::Answer showMessageBox ( const GString& msg, GMessageBox::Type type = GMessageBox::TYPE_NONE, const GString& flags = GString::Empty, const GString& title = GString::Empty, bool monoFont = false, const GString& userText1 = GString::Empty, const GString& userText2 = GString::Empty, const GString& userText3 = GString::Empty );
      virtual void startTimer ( const GString& timerID, int millis, GObject* userData = null, bool autoDeleteUserData = false );
      virtual void stopTimer ( const GString& timerID );
      static GRectangle MakeLibRect ( const SysRect& r, const GSystemYConverter& conv );
      static SysRect MakeSysRect ( const GRectangle& r, const GSystemYConverter& conv, bool incl );
      virtual int sysY ( int y ) const;
      static WindowProc SubclassWindow ( HWND hwnd, WindowProc newProc );
      virtual int trackHSplit ( int minY, int maxY, int startY, int cancel );
      virtual int trackVSplit ( int minX, int maxX, int startX, int cancel );
      virtual void updateHintString ( const GString& cmdID );
      virtual void updateWindow ();
      virtual GString userChooseFile ( const GString& titleText, const GString& initialPath = GString::Empty );
      virtual GString userChooseFont ( const GString& titleText, bool setFont, GWindow* parentWin = null, const GString& initialFont = GString::Empty, bool fixedOnly = false );
      virtual void validateAll ( bool inclChildren ) const;
      virtual void validateRect ( const GRectangle& rect ) const;
      virtual void writeProfile ( const GString& sectName, bool force = false );

   protected:
      bool callDefaultKeyHandler ( const class GKeyMessage& key, bool lowLevel, bool transAccel );
      virtual GWindowMessage::Answer handleWindowMessage ( GWindowMessage& msg );
      virtual bool onActivate ();
      virtual bool onBackgroundColorChanged ( const GColor& color );
      virtual bool onButton1Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton1Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2Click ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onButton2Up ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onClose ();
      virtual bool onCommand ( int cmd );
      virtual bool onDeactivate ();
      virtual void onFocus ( bool set );
      virtual void onFocusKill ();
      virtual void onFocusSet ();
      virtual bool onFontNameSizeChanged ( const GString& fontNameSize );
      virtual bool onForegroundColorChanged ( const GColor& color );
      virtual bool onHide ();
      virtual bool onInitMenu ();
      virtual bool onKeyDown ( const class GKeyMessage& key );
      virtual bool onKey ( const class GKeyMessage& key );
      virtual bool onKeyUp ( const class GKeyMessage& key );
      virtual bool onMenuSelect ( int id );
      virtual bool onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& flags );
      virtual bool onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn );
      virtual bool onPaint ( GGraphics& g, const GRectangle& rect );
      virtual bool onPaintBackground ( GGraphics& g, const GRectangle& rect );
      virtual void onPaintBorder ( class GGraphics& g );
      virtual bool onPaintText ( GGraphics& g, const GRectangle& rect );
      virtual bool onPresParamChanged ( GWindow::PresParamID paramID );
      virtual bool onShow ();
      virtual bool onSize ( int width, int height );
      virtual bool onTimer ( const GString& timerID, GObject* userData );
      virtual bool onUserMessage ( class GUserMessage& msg );

   // -----------------------------------------------------------------------
   // GDialogPanel stuff:

   private:
      mutable int recentSearchIndex;
      int ctrlChangedFilter;
      bool noInitCtrlChanged;
      char hotChar;
      bool hasSent_GM_INITDIALOG;
      GDialogMessageHandler* userMsgProc;

   protected:
      bool filterDlgMessages;
      bool isIn_DmInitDialog;
      bool isInDismiss;
      GWindow* recentFocus;

   protected:
      virtual bool onDialogMessage ( GDialogMessage& msg );

   public:
      virtual int autoAssocAHotKey ( const GString& text );
      virtual void dismiss ( const GString& arg = GString::Empty );
      virtual void enterCtrlChangedFilter ();
      virtual void exitCtrlChangedFilter ();
      virtual GWindow& getComponentByID ( const GString& id );
      virtual GWindow& getComponentByIndex ( int index );
      virtual int getComponentCount () const;
      virtual int getComponentIndex ( const GString& id ) const;
      virtual int getCtrlChangedFilter () const;
      virtual char getHotKeyChar () const;
      virtual class GDialogPanel* getOwnerDialogPanel ();
      virtual GString getProfileSectionName ( const GString& parentSectName );
      virtual GString getTooltipText () const;
      virtual void hotKeyAction ();
      virtual bool isCtrlActivable ();
      virtual bool isEmpty () const;
      virtual bool isExecutingAsDialog () const;
      virtual bool isInitialized () const;
      virtual bool isStatic () const;
      virtual void postDialogMessage ( GDialogMessage* msg, bool autoDeleteMsg );
      virtual bool prepareForExecution ( GObject* initArg, bool* breakSem );
      virtual void sendCtrlChanged ();
      bool sendDialogMessage ( GDialogMessage& msg );
      bool sendDialogMessage ( GDialogMessageHandler::MessageID id, GObject* param1 = null, GObject* param2 = null );
      bool sendDialogMessage ( GDialogMessageHandler::MessageID id, const GString& str1, GObject* param2 = null );
      bool sendDialogMessage ( GDialogMessageHandler::MessageID id, const GString& str1, const GString& str2 );
      virtual bool setComponentFocus ( int index );
      virtual bool setComponentFocus ( const GString& id );
      virtual GDialogMessageHandler* setDialogMessageHandler ( GDialogMessageHandler* msgProc );
      virtual void setNoInitCtrlChanged ( bool flag );
      virtual void setTooltipText ( const GString& tooltipText );
};

#endif
