/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GToggleButton.h"
#include "glib/gui/GPushButton.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/util/GLog.h"

GToggleButton::Peer::Peer ( GToggleButton& parentWin, 
                            bool is3State,
                            bool noFocusPaint )
                    :GWindow("Peer",
                             GBorderLayout::CENTER,
                             &parentWin,
                             &parentWin,
                             WS_VISIBLE | (is3State ? BS_AUTO3STATE : BS_AUTOCHECKBOX) | (noFocusPaint ? BS_NOCURSORSELECT : 0),
                             WS2_DEFAULTPAINT | WS2_NOT_STATIC | WS2_OS2Y | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT,
                             GWindowClass::BUTTON)
{
   setFocusable(true);
}

GToggleButton::Peer::~Peer ()
{
}

GToggleButton::GToggleButton ( const GString& name,
                               const GString& constraints,
                               GWindow& parentWin,
                               long winStyle,
                               long winStyle2,
                               bool is3State,
                               bool noFocusPaint,
                               bool useHotKey )
              :GWindow(name,
                       constraints,
                       &parentWin,
                       &parentWin,
                       winStyle,
                       winStyle2 | WS2_DEFAULTPAINT | WS2_OS2Y | WS2_NOT_STATIC),
               is3State(is3State),
               useHotKey(useHotKey),
               peer(*this, is3State, noFocusPaint)
{
   setLayoutManager(new GBorderLayout(), true);
}

GToggleButton::~GToggleButton ()
{
}

GWindow& GToggleButton::getPeer ()
{
   return peer;
}

void GToggleButton::setText ( const GString& text )
{
   GString oldText = getText();
   if (text == oldText)
      return;

   GWindow::setText(text);
   int pos = 0;
   if (useHotKey)
      pos = autoAssocAHotKey(text);
   GString str = text;
   if (pos >= 0)
      str.insert(GPushButton::MNEMONIC_MARKER_CHAR, pos);
   peer.setText(str);
}

void GToggleButton::grabFocus ( bool force )
{
   peer.grabFocus(force);
}

void GToggleButton::setEnabled ( bool flag, bool repaint )
{
   GWindow::setEnabled(flag, repaint);
   peer.setEnabled(flag, repaint);
}

void GToggleButton::setOily ( bool flag )
{
   peer.setOily(flag);
   GWindow::setOily(flag);
}

void GToggleButton::setToggleStateUndefined ()
{
   enterCtrlChangedFilter();
   setToggleState(2);
   exitCtrlChangedFilter();
   sendCtrlChanged();
}

bool GToggleButton::isToggleStateUndefined () const
{
   int state = getToggleState();
   return state == 2;
}

void GToggleButton::hotKeyAction ()
{
   grabFocus();
   if (is3State)
   {
      if (isToggleStateUndefined())
         setToggleState(false);
      else
      if (getToggleState())
         setToggleStateUndefined();
      else
         setToggleState(true);
   }
   else
   {
      bool state = getToggleState();
      setToggleState(!state);
   }
}

void GToggleButton::setToggleState ( int state )
{
   if (state == 0 || state == 1 || (state == 2 && is3State))
   {
      if (state == getToggleState())
         return;
      peer.sendMessage(BM_SETCHECK, MPARAM(state));
      // Implicitly send GM_CTRLCHANGED.
      fireValueChangeListeners();
   }
   else
   {
      GLog::Log(this, "Unsupported state: %d (%s:%d)", GVArgs(state).add(__FUNCTION__).add(__LINE__));
   }
}

int GToggleButton::getToggleState () const
{
   return int(peer.sendMessage(BM_QUERYCHECK));
}

bool GToggleButton::isEmpty () const
{
   return false;
}

GString GToggleButton::queryValue () const
{
   int state;
   if (is3State && isToggleStateUndefined())
      state = 2;
   else
   if (getToggleState())
      state = 1;
   else
      state = 0;
   return GInteger::ToString(state);
}

void GToggleButton::changeValue ( const GString& newValue, bool notify )
{
   int newval;
   if (is3State && newValue == "2")
      newval = 2;
   else
   if (newValue == "1" || newValue.equalsIgnoreCase("true") || newValue.equalsIgnoreCase("yes") || newValue.equalsIgnoreCase("on"))
      newval = 1;
   else
      newval = 0;

   enterCtrlChangedFilter();
   if (newval == 2)
      setToggleStateUndefined();
   else
      setToggleState(newval != 0);
   exitCtrlChangedFilter();

   if (notify)
      fireValueChangeListeners();
}

bool GToggleButton::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case BN_CLICKED:
      case BN_DBLCLICKED:
      {
         fireValueChangeListeners();
         break;
      }
   }

   return true;
}
