/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include <ctype.h>
#include "glib/gui/GTextViewer.h"
#include "glib/gui/GMessageBox.h"
#include "glib/gui/GGraphics.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/util/GMath.h"
#include "glib/sys/GSystem.h"
#include "glib/GProgram.h"

DEFINE_COMMAND_TABLE(GTextViewer);
   ADD_COMMAND("cmdCopy", cmdCopy);
   ADD_COMMAND("cmdSearch", cmdSearch);
   ADD_COMMAND("cmdSearchNext", cmdSearchNext);
   ADD_COMMAND("cmdGoto", cmdGoto);
   ADD_COMMAND("cmdSelectAll", cmdSelectAll);
   ADD_COMMAND("cmdUnselectAll", cmdUnselectAll);
   ADD_COMMAND("cmdNavigateUp", cmdNavigateUp);
   ADD_COMMAND("cmdNavigateDown", cmdNavigateDown);
   ADD_COMMAND("cmdNavigateLeft", cmdNavigateLeft);
   ADD_COMMAND("cmdNavigateRight", cmdNavigateRight);
   ADD_COMMAND("cmdNavigateHome", cmdNavigateHome);
   ADD_COMMAND("cmdNavigateEnd", cmdNavigateEnd);
   ADD_COMMAND("cmdNavigatePrevWord", cmdNavigatePrevWord);
   ADD_COMMAND("cmdNavigateNextWord", cmdNavigateNextWord);
   ADD_COMMAND("cmdNavigatePageUp", cmdNavigatePageUp);
   ADD_COMMAND("cmdNavigatePageDown", cmdNavigatePageDown);
   ADD_COMMAND("cmdNavigateTop", cmdNavigateTop);
   ADD_COMMAND("cmdNavigateBottom", cmdNavigateBottom);
   ADD_COMMAND("cmdNavigateSelectUp", cmdNavigateSelectUp);
   ADD_COMMAND("cmdNavigateSelectDown", cmdNavigateSelectDown);
   ADD_COMMAND("cmdNavigateSelectLeft", cmdNavigateSelectLeft);
   ADD_COMMAND("cmdNavigateSelectRight", cmdNavigateSelectRight);
   ADD_COMMAND("cmdNavigateSelectHome", cmdNavigateSelectHome);
   ADD_COMMAND("cmdNavigateSelectEnd", cmdNavigateSelectEnd);
   ADD_COMMAND("cmdNavigateSelectPrevWord", cmdNavigateSelectPrevWord);
   ADD_COMMAND("cmdNavigateSelectNextWord", cmdNavigateSelectNextWord);
   ADD_COMMAND("cmdNavigateSelectPageUp", cmdNavigateSelectPageUp);
   ADD_COMMAND("cmdNavigateSelectPageDown", cmdNavigateSelectPageDown);
   ADD_COMMAND("cmdNavigateSelectTop", cmdNavigateSelectTop);
   ADD_COMMAND("cmdNavigateSelectBottom", cmdNavigateSelectBottom);
END_COMMAND_TABLE;

const GString GTextViewer::DefaultFont = "11.System VIO";

GTextViewer::GTextViewer ( GTextViewer::SearchParams* searchParams,
                           const GString& name,
                           const GString& constraints,
                           GWindow& parentWin,
                           bool useAsConsoleMonitor,
                           bool paintFreeTextSpaceAtTop )
            :GDecoratedWindow(name,
                              constraints,
                              &parentWin,
                              GString::Empty,
                              WS_VISIBLE | WS_SYNCPAINT,
                              WS2_OS2Y | WS2_NOT_STATIC),
             lines(512),
             wordSeparatorCharacters(",.-+;:/*\"'%&()=!?|{}[]\\^"),
             searchParams(searchParams),
             idxFirstVisibleColumn(0),
             idxFirstVisibleLine(0),
             winColumnNr(0),
             winLineNr(0),
             widestLine(0),
             fontw(1), // Prevent "divide-by-zero" upon initialization.
             fonth(1), // Ditto.
             verticalFreeSpace(0),
             lastExplicitXPos(0),
             lastExplicitXPosWasEol(false),
             selectionMode(false),
             vlineBuff(1024)
{
   setFocusable(true);

   opts.useAsConsoleMonitor = useAsConsoleMonitor;
   opts.paintFreeTextSpaceAtTop = paintFreeTextSpaceAtTop;

   defaultBackgroundColor = GColor::WHITE;
   defaultForegroundColor = GColor::BLACK;
   defaultFontNameSize = GTextViewer::DefaultFont;
   setBackgroundColor(defaultBackgroundColor);
   setForegroundColor(defaultForegroundColor);
   setFontNameSize(defaultFontNameSize);

   setCaretOn(true);
   setCurrentPos(GTextViewer::Pos(0, 0));
   setHScrollVisible(true);
   setVScrollVisible(true);
}

GTextViewer::~GTextViewer ()
{
}

GTextViewer::Pos::Pos ()
                 :x(0),
                  y(0)
{
}

GTextViewer::Pos::Pos ( int x, int y )
                 :x(GMath::Max(x, 0)),
                  y(GMath::Max(y, 0))
{
}

GTextViewer::Pos::Pos ( const Pos &pos )
                 :x(GMath::Max(pos.x, 0)),
                  y(GMath::Max(pos.y, 0))
{
}

GTextViewer::Pos::~Pos ()
{
}

int GTextViewer::Pos::getX () const
{
   return x;
}

int GTextViewer::Pos::getY () const
{
   return y;
}

GTextViewer::Pos& GTextViewer::Pos::set ( int x, int y )
{
   this->x = x;
   this->y = y;
   return *this;
}

void GTextViewer::Pos::setX ( int x )
{
   this->x = x;
}

void GTextViewer::Pos::setY ( int y )
{
   this->y = y;
}

GString GTextViewer::Pos::toString () const
{
   return GString("[%d, %d]", GVArgs(x).add(y));
}

GTextViewer::Pos& GTextViewer::Pos::operator= ( const Pos& pos )
{
   if (&pos != this)
   {
      x = pos.x;
      y = pos.y;
   }
   return *this;
}

bool GTextViewer::Pos::operator== ( const Pos& pos ) const
{
   if (&pos == this)
      return true;
   else
      return x == pos.x && y == pos.y;
}

bool GTextViewer::Pos::operator!= ( const Pos& pos ) const
{
   return !operator==(pos);
}

bool GTextViewer::Pos::operator> ( const Pos& pos ) const
{
   if (&pos == this)
      return false;
   else
      return y > pos.y || (y == pos.y && x > pos.x);
}

bool GTextViewer::Pos::operator< ( const Pos& pos ) const
{
   if (&pos == this)
      return false;
   else
      return y < pos.y || (y == pos.y && x < pos.x);
}

bool GTextViewer::Pos::operator>= ( const Pos& pos ) const
{
   return operator>(pos) || operator==(pos);
}

bool GTextViewer::Pos::operator<= ( const Pos& pos ) const
{
   return operator<(pos) || operator==(pos);
}

GTextViewer::Selection::Selection ()
{
}

GTextViewer::Selection::~Selection ()
{
}

void GTextViewer::Selection::clear ( const GTextViewer::Pos& newOrigo )
{
   anchor = newOrigo;
   start = newOrigo;
   end = newOrigo;
}

void GTextViewer::Selection::clear ()
{
   GTextViewer::Pos p(0, 0);
   clear(p);
}

bool GTextViewer::Selection::isAnySelection () const
{
   return start != end;
}

bool GTextViewer::Selection::isEmpty () const
{
   return start == end;
}

GTextViewer::ViewerOptions::ViewerOptions ()
                           :tabWidth(8),
                            forceEOLPos(8192), // 2^13=8192
                            useAsConsoleMonitor(false),
                            paintFreeTextSpaceAtTop(false),
                            showForceEOLPosVLine(true),
                            colorOfForceEOLPosVLine(GColor::DGRAY),
                            showTabs(false),
                            showSpaces(false),
                            showLinefeeds(false),
                            chrShowTab('.'),
                            chrShowSpace('.'),
                            chrShowLinefeed('.'),
                            quickScrollMode(false),
                            wordWrap(false),
                            wordWrapChars("-+/*=!?.,:;%&\\^")
{
}

GTextViewer::ViewerOptions::~ViewerOptions ()
{
}

GTextViewer::SearchParams::SearchParams ()
             :searchString(GString::Empty),
                           searchForward(true),
                           searchCaseSen(false),
                           searchWord(false)
{
}

GTextViewer::SearchParams::~SearchParams ()
{
}

bool GTextViewer::SearchParams::userEditSearchParams ( GTextViewer& owner )
{
   return false;
}

void GTextViewer::SearchParams::onSearchStringNotFound ( GTextViewer& owner )
{
}

void GTextViewer::removeAllText ()
{
   lastExplicitXPos = 0;
   lastExplicitXPosWasEol = false;
   idxFirstVisibleColumn = 0;
   widestLine = 0;
   idxFirstVisibleLine = 0;
   selection.clear();
   selectionMode = false;
   lines.removeAll();
   setCurrentPos(GTextViewer::Pos(0, 0));
   updateScrollBarPosAndRange();
   invalidateAll(false);
}

void GTextViewer::updateGuiAfterContentChange ()
{
   updateScrollBarPosAndRange();
   updateWindow();
}

void GTextViewer::appendLine ( GString* text, bool autoDel, bool updtGui )
{
   stopSelectionMode(); // Prevent selection from being automatically extended

   // Because word-wrap might be on, the number of added "lines" might be 
   // more than just one. In that case we need to know the actual number of 
   // new "lines" after the added string has been word-wrapped. 
   // Thus, remember the number of lines as of before we adds the new one.
   int idx1 = lines.getCount();

   // ---
   GString* textToAdd = (autoDel ? text : new GString(*text));
   if (textToAdd->lastChar() != '\n')
      textToAdd->append('\n', 1);
   lines.add(textToAdd, true);
   for (int i=idx1; i<lines.getCount(); i++)
   {
      wrapLineIfNeededAt(i);
      updateWidestLine(i);
   }

   if (updtGui && isVisible() && winLineNr > 0)
   {
      bool keepInvalidated = false;
      int idx2 = lines.getCount();
      if (opts.useAsConsoleMonitor && !hasFocus())
      {
         if (idxFirstVisibleLine >= idx1 - winLineNr)
         {
            int prevFreeLinesAreaAtBottom = winLineNr - (idx1 - idxFirstVisibleLine);
            int linesToScroll = idx2 - idx1 - prevFreeLinesAreaAtBottom;
            if (linesToScroll > 0)
            {
               idxFirstVisibleLine += linesToScroll;
               scrollWindow(0, fonth*linesToScroll, false, null, &rectScrollClip);
            }         
         }
         else
         {
            keepInvalidated = true;
         }
         gotoPos(idx2, 0);
      }

      updateScrollBarPosAndRange();

      if (!keepInvalidated)
      {
         /* // Uncomment this to mark each newly painted line with a red box.
         for (int i=idxFirstVisibleLine; i<idxFirstVisibleLine+winLineNr; i++)
            paintLine(i); */
         for (int i=idx1; i<idx2; i++)
         {
            GRectangle r = calcRectOfIndexedLine(i);
            paintLine(i);
            validateRect(r);
            /* // Ditto
            GGraphics g(this);
            g.setColor(GColor::RED);
            g.drawRectangle(r); */
         }
      }
   }
}

void GTextViewer::appendLine ( const GString& text )
{
   GString& str = const_cast<GString&>(text);
   appendLine(&str, false, true);
}

void GTextViewer::updateLine ( int index, const GString& text )
{
   // Prevent selection from being automatically extended.
   stopSelectionMode(); 

   lines[index] = text;
   updateWidestLine(index);

   if (isVisible())
   {
      if (index >= idxFirstVisibleLine && 
          index < idxFirstVisibleLine + winLineNr)
      {
         paintLine(index);
         // updateScrollBarPosAndRange();
      }
   }
}

const GString& GTextViewer::getIndexedLine ( int index ) const
{
   if (index < 0 || index >= lines.getCount())
      return GString::Empty;
   else
      return lines[index];
}

void GTextViewer::startSelectionMode ()
{
   unselectAll();
   selectionMode = true;
   selection.clear(curPos);
}

void GTextViewer::continueSelection ( bool shift )
{
   if (!shift)
      stopSelectionMode();
   else
   if (!isAnySelectedText())
      startSelectionMode();
   else
   if (curPos == selection.start)
      selectionMode = true;
   else
   if (curPos == selection.end)
      selectionMode = true;
   else
      startSelectionMode();
}

void GTextViewer::stopSelectionMode ()
{
   selectionMode = false;
}

void GTextViewer::dragSelection ( const GTextViewer::Pos& newPos, const GTextViewer::Pos& prevPos )
{
   setCurrentPos(newPos);

   if (!isSelectionMode())
      return;

   bool anySelection = isAnySelectedText();
   if (newPos == selection.anchor && anySelection)
   {
      startSelectionMode();
      stopSelectionMode();
   }
   else
   if (prevPos == selection.anchor && anySelection)
   {
      if (selection.anchor > selection.start)
      {
         selection.anchor = selection.start;
         selection.end = newPos;
      }
      else
      {
         selection.anchor = selection.end;
         selection.start = newPos;
      }
   }
   else
   if (newPos < selection.anchor)
   {
      selection.start = newPos;
      selection.end = selection.anchor;
   }
   else
   if (newPos > selection.anchor)
   {
      selection.end = newPos;
      selection.start = selection.anchor;
   }
   else
   {
      startSelectionMode();
   }
}

void GTextViewer::invalidateLines ( int firstLine, int lastLine )
{
   firstLine = GMath::Min(firstLine, lastLine);
   lastLine = GMath::Max(firstLine, lastLine);
   int starty = GMath::Max(firstLine, idxFirstVisibleLine);
   int endy = GMath::Min(lastLine, idxFirstVisibleLine + winLineNr - 1);

   GRectangle r = getWindowRect();
   r.y = rectScrollClip.y + rectScrollClip.height - fonth*(endy - idxFirstVisibleLine + 1);
   r.height = fonth * (endy - starty + 1);
   invalidateRect(r);
}

void GTextViewer::invalidateLine ( int index )
{
   invalidateLines(index, index);
}

int GTextViewer::getUserSelectedLineNumber () 
{ 
   return 0; 
}

void GTextViewer::setSelectionEndPos ( const GTextViewer::Pos& pos )
{
   GTextViewer::Pos prevStart(selection.start);
   GTextViewer::Pos prevEnd(selection.end);
   bool anyPrevSelection = isAnySelectedText();
   int absColumn = pos.x;
   int absLine = pos.y;

   if (absLine < 0)
      absLine = 0;
   else
   if (absLine > lines.getCount())
      absLine = lines.getCount();
   const GString& line = getIndexedLine(absLine);
   int lineLen = line.length();
   if (absColumn > lineLen)
      absColumn = lineLen;
   else
   if (absColumn < 0)
      absColumn = 0;
   selection.end.set(absColumn, absLine);
   if (prevEnd != selection.end)
   {
      int starty = anyPrevSelection ? GMath::Min(selection.start.y, prevStart.y) : selection.start.y;
      int endy = anyPrevSelection ? GMath::Max(selection.end.y, prevEnd.y) : selection.end.y;
      invalidateLines(starty, endy);
   }
}

void GTextViewer::setSelectionStartPos ( const GTextViewer::Pos& pos )
{
   GTextViewer::Pos prevStart(selection.start);
   GTextViewer::Pos prevEnd(selection.end);
   bool anyPrevSelection = isAnySelectedText();
   int absColumn = pos.x;
   int absLine = pos.y;

   if (absLine < 0)
      absLine = 0;
   else
   if (absLine > lines.getCount())
      absLine = lines.getCount();
   const GString& line = getIndexedLine(absLine);
   int lineLen = line.length();
   if (absColumn > lineLen)
      absColumn = lineLen;
   else
   if (absColumn < 0)
      absColumn = 0;
   selection.start.set(absColumn, absLine);
   if (prevStart != selection.start)
   {
      int starty = anyPrevSelection ? GMath::Min(selection.start.y, prevStart.y) : selection.start.y;
      int endy = anyPrevSelection ? GMath::Max(selection.end.y, prevEnd.y) : selection.end.y;
      invalidateLines(starty, endy);
   }
}

void GTextViewer::navigateUp ( bool shift )
{
   if (curPos.y == 0)
      return;

   Pos pos;
   continueSelection(shift);
   if (!shift && opts.quickScrollMode)
   {
      onVScrollLineUp();
      pos = calcXPosFromPhysicalToVirtual(lastExplicitXPos, idxFirstVisibleLine);
   }
   else
   {
      pos = calcXPosFromPhysicalToVirtual(lastExplicitXPos, curPos.y - 1);
   }
   pos = calcXPosFromVirtualToPhysical(pos);
   gotoPos(pos, false);
}

void GTextViewer::navigateDown ( bool shift )
{
   if (curPos.y == lines.getCount())
      return;

   Pos pos;
   continueSelection(shift);
   if (!shift && opts.quickScrollMode)
   {
      onVScrollLineDown();
      pos = calcXPosFromPhysicalToVirtual(lastExplicitXPos, idxFirstVisibleLine + winLineNr - 1);
   }
   else
   {
      pos = calcXPosFromPhysicalToVirtual(lastExplicitXPos, curPos.y + 1);
   }
   pos = calcXPosFromVirtualToPhysical(pos);
   gotoPos(pos, false);
}

void GTextViewer::navigatePageUp ( bool shift )
{
   continueSelection(shift);
   onVScrollPageUp();
   Pos pos = calcXPosFromPhysicalToVirtual(lastExplicitXPos, curPos.y - winLineNr + 1);
   pos = calcXPosFromVirtualToPhysical(pos);
   gotoPos(pos, false);
}

void GTextViewer::navigatePageDown ( bool shift )
{
   continueSelection(shift);
   onVScrollPageDown();
   Pos pos = calcXPosFromPhysicalToVirtual(lastExplicitXPos, curPos.y + winLineNr - 1);
   pos = calcXPosFromVirtualToPhysical(pos);
   gotoPos(pos, false);
}

void GTextViewer::navigateLeft ( bool shift )
{
   continueSelection(shift);
   if (!shift && opts.quickScrollMode)
   {
      onHScrollLineUp();
      gotoPos(curPos.y, idxFirstVisibleColumn);
   }
   else
   {
      gotoPos(curPos.y, curPos.x - 1);
   }
}

void GTextViewer::navigateRight ( bool shift )
{
   continueSelection(shift);
   if (!shift && opts.quickScrollMode)
   {
      onHScrollLineDown();
      gotoPos(curPos.y, idxFirstVisibleColumn + winColumnNr - 1);
   }
   else
   {
      gotoPos(curPos.y, curPos.x + 1);
   }
}

void GTextViewer::navigateHome ( bool shift )
{
   continueSelection(shift);
   gotoPos(curPos.y, 0);
}

void GTextViewer::navigateEnd ( bool shift )
{
   continueSelection(shift);
   if (!shift && opts.quickScrollMode)
   {
      gotoPos(curPos.y, widestLine);
   }
   else
   {
      GString line = getIndexedLine(curPos.y);
      line.stripTrailingEol();
      int len = line.length();
      gotoPos(curPos.y, len);
   }
   lastExplicitXPosWasEol = true;
   lastExplicitXPos = GInteger::MAX_VALUE;
}

void GTextViewer::navigateTop ( bool shift )
{
   continueSelection(shift);
   gotoPos(0, 0);
}

void GTextViewer::navigateBottom ( bool shift )
{
   continueSelection(shift);
   int lc = lines.getCount();
   gotoPos(lc, 0);
}

void GTextViewer::navigatePrevWord ( bool shift )
{
   continueSelection(shift);

   int startx = curPos.getX() - 1;
   bool hasReachedNextWord = false;

   for (int c1=curPos.getY(); c1>=0; c1--)
   {
      const GString& line = getIndexedLine(c1);
      int lineLen = line.length();
      if (lineLen > 0)
      {
         for (int c2=GMath::Min(lineLen-1, startx); c2>=0; c2--)
         {
            char chr = line[c2];
            if (GCharacter::IsWhiteSpace(chr) || 
                wordSeparatorCharacters.contains(chr))
            {
               if (hasReachedNextWord)
               {
                  gotoPos(c1, c2+1);
                  return;
               }
            }
            else
            {
               hasReachedNextWord = true;
            }
         }

         if (hasReachedNextWord)
         {
            gotoPos(c1, 0);
            return;
         }
      }

      startx = widestLine;
   }

   navigateTop(shift);
}

void GTextViewer::navigateNextWord ( bool shift )
{
   continueSelection(shift);

   int startx = curPos.x;
   bool hasReachedAWhite = (getIndexedLine(curPos.y) == "");

   for (int c1=curPos.y; c1<=lines.getCount(); c1++)
   {
      const GString& line = getIndexedLine(c1);
      int lineLen = line.length();
      if (lineLen > 0)
      {
         for (int c2=startx; c2<lineLen; c2++)
         {
            char chr = line[c2];
            if (!GCharacter::IsWhiteSpace(chr) &&
                !wordSeparatorCharacters.contains(chr))
            {
               if (hasReachedAWhite)
               {
                  gotoPos(c1, c2);
                  return;
               }
            }
            else
            {
               hasReachedAWhite = true;
            }
         }

         hasReachedAWhite = true;
      }

      startx = 0;
   }

   navigateBottom(shift);
}

void GTextViewer::gotoPos ( const Pos& pos, bool updtLastExplicitXPos ) 
{ 
   gotoPos(pos.y, pos.x, updtLastExplicitXPos); 
}

void GTextViewer::gotoPos ( int absLine, int absColumn, bool updtLastExplicitXPos )
{
   // Temporarily hide the caret while we are about to scroll.
   // This is to prevent some uggly flashing of the caret.
   bool caretWasHidden = !isVisiblePos(Pos(absColumn, absLine));
   if (caretWasHidden)
      showCaret(false);

   if (absLine < 0)
      absLine = 0;
   else
   if (absLine > lines.getCount())
      absLine = lines.getCount();

   if (absColumn < 0)
      absColumn = 0;
   else
   if ((absColumn > idxFirstVisibleColumn + winColumnNr - 1) && (winColumnNr >= widestLine))
      absColumn = winColumnNr - idxFirstVisibleColumn - 1;

   GTextViewer::Pos prevPos(curPos);
   GTextViewer::Pos newPos(absColumn, absLine);
   dragSelection(newPos, prevPos);

   int curPosXVirtual = calcXPosFromPhysicalToVirtual(curPos).x;
   if (absLine == curPos.y && isVisibleY(curPos.y))
   {
      // Move caret on current line, which is already within vertical
      // visible area.
      if (!isVisibleVirtualX(curPosXVirtual))
      {
         idxFirstVisibleColumn = curPosXVirtual - (winColumnNr / 2);
         if (idxFirstVisibleColumn < 0)
            idxFirstVisibleColumn = 0;
         updateScrollBarPos();
         invalidateAll(false);
      }
      else
      if (isSelectionMode())
      {
         if (curPos.y < prevPos.y)
         {
            int num = prevPos.y - curPos.y + 1;
            if (curPos.y + num > idxFirstVisibleLine + winLineNr)
               num = idxFirstVisibleLine + winLineNr - curPos.y;
            invalidateLines(curPos.y, curPos.y + num - 1);
         }
         else
         if (curPos.y > prevPos.y)
         {
            int starty = prevPos.y;
            if (starty < idxFirstVisibleLine)
               starty = idxFirstVisibleLine;
            int num = curPos.y - starty + 1;
            if (starty + num > idxFirstVisibleLine + winLineNr)
               num = idxFirstVisibleLine + winLineNr - starty;
            invalidateLines(starty, starty + num - 1);
         }
         else
         {
            invalidateLine(curPos.y);
         }
      }
   }
   else
   if (absLine == idxFirstVisibleLine - 1)
   {
      // Scroll one line UP
      if (!isVisibleVirtualX(curPosXVirtual))
      {
         idxFirstVisibleLine -= 1;
         if (idxFirstVisibleLine < 0)
            idxFirstVisibleLine = 0;
         idxFirstVisibleColumn = curPos.x - (winColumnNr / 2);
         if (idxFirstVisibleColumn < 0)
            idxFirstVisibleColumn = 0;
         updateScrollBarPos();
         invalidateAll(false);
      }
      else
      {
         onVScrollLineUp();
         if (isSelectionMode())
            invalidateLine(idxFirstVisibleLine + 1);
      }
   }
   else
   if (absLine == idxFirstVisibleLine + winLineNr)
   {
      // Scroll one line DOWN
      if (!isVisibleVirtualX(curPosXVirtual))
      {
         idxFirstVisibleLine += 1;
         if (idxFirstVisibleLine > lines.getCount() - winLineNr)
            idxFirstVisibleLine = lines.getCount() - winLineNr;
         if (idxFirstVisibleLine < 0)
            idxFirstVisibleLine = 0;
         idxFirstVisibleColumn = curPos.x - (winColumnNr / 2);
         if (idxFirstVisibleColumn < 0)
            idxFirstVisibleColumn = 0;
         updateScrollBarPos();
         invalidateAll(false);
      }
      else
      {
         onVScrollLineDown();
         if (isSelectionMode())
            invalidateLine(idxFirstVisibleLine + winLineNr - 2);
      }
   }
   else
   if (absLine == 0)
   {
      // Ctrl+HOME
      if (!isVisiblePos(curPos))
      {
         idxFirstVisibleLine = 0;
         idxFirstVisibleColumn = curPos.x - (winColumnNr / 2);
         if (idxFirstVisibleColumn < 0)
            idxFirstVisibleColumn = 0;
         updateScrollBarPos();
         invalidateAll(false);
      }
      else
      if (isSelectionMode())
         invalidateAll(false);
   }
   else
   if (absLine == lines.getCount())
   {
      // Ctrl+END
      if (!isVisiblePos(curPos))
      {
         idxFirstVisibleLine = lines.getCount() - winLineNr + 1;
         if (idxFirstVisibleLine < 0)
            idxFirstVisibleLine = 0;
         if (!isVisibleVirtualX(idxFirstVisibleColumn))
         {
            idxFirstVisibleColumn = curPos.x - (winColumnNr / 2);
            if (idxFirstVisibleColumn < 0)
               idxFirstVisibleColumn = 0;
         }
         updateScrollBarPos();
         invalidateAll(false);
      }
      else
      if (isSelectionMode())
         invalidateAll(false);
   }

   if (!isVisibleVirtualX(curPosXVirtual))
   {
      idxFirstVisibleColumn = curPos.x - (winColumnNr / 2);
      if (idxFirstVisibleColumn < 0)
         idxFirstVisibleColumn = 0;
      updateScrollBarPos();
      invalidateAll(false);
   }

   if (!isVisibleY(curPos.y))
   {
      idxFirstVisibleLine = curPos.y - (winLineNr / 2);
      if (idxFirstVisibleLine < 0)
         idxFirstVisibleLine = 0;
      updateScrollBarPos();
      invalidateAll(false);
   }

   if (updtLastExplicitXPos)
   {
      lastExplicitXPos = curPos.x;
      lastExplicitXPosWasEol = false;
   }

   // Switch the caret back on.
   if (caretWasHidden)
      showCaret(true);
}

void GTextViewer::setCurrentPos ( const GTextViewer::Pos& pos )
{
   curPos = pos;

   // Find the virtual x-position when respecting any tab-characters.
   int virtualX = 0;
   const GString& line = getIndexedLine(curPos.y);
   for (int i=0, len=line.length(); i<len && i<curPos.x; i++)
   {
      char chr = line[i];
      if (chr == '\t')
         virtualX += opts.tabWidth - (virtualX % opts.tabWidth);
      else
         virtualX += 1;
   }

   int x = fontw*(virtualX - idxFirstVisibleColumn);
   int y = rectScrollClip.y + rectScrollClip.height - fonth*(curPos.y - idxFirstVisibleLine + 1);
   setCaretPos(x, y);

   // Call the overridable notification method.
   onCaretPositionChanged();
}

bool GTextViewer::onKeyDown ( const GKeyMessage& key )
{
   // Do the default keyboard cursor navigation.
   // Note that these keyes can be customized. This is typically done either 
   // by overriding this method, or by activating a keyboard accelerator 
   // table on the text viewer vindow.
   switch (key.getCode())
   {
      case GKey::KEY_CTRL_C: 
      case GKey::KEY_CTRL_INSERT: cmdCopy(); return true;
      case GKey::KEY_CTRL_F: cmdSearch(); return true;
      case GKey::KEY_CTRL_L: cmdSearchNext(); return true;
      case GKey::KEY_CTRL_N: cmdSearchNext(); return true;
      case GKey::KEY_CTRL_G: cmdGoto(); return true;
      case GKey::KEY_CTRL_A: cmdSelectAll(); return true;
      case GKey::KEY_CTRL_SPACE: cmdUnselectAll(); return true;
      case GKey::KEY_UP: cmdNavigateUp(); return true;
      case GKey::KEY_DOWN: cmdNavigateDown(); return true;
      case GKey::KEY_LEFT: cmdNavigateLeft(); return true;
      case GKey::KEY_RIGHT: cmdNavigateRight(); return true;
      case GKey::KEY_HOME: cmdNavigateHome(); return true;
      case GKey::KEY_END: cmdNavigateEnd(); return true;
      case GKey::KEY_CTRL_LEFT: cmdNavigatePrevWord(); return true;
      case GKey::KEY_CTRL_RIGHT: cmdNavigateNextWord(); return true;
      case GKey::KEY_PAGEUP: cmdNavigatePageUp(); return true;
      case GKey::KEY_PAGEDOWN: cmdNavigatePageDown(); return true;
      case GKey::KEY_CTRL_HOME: cmdNavigateTop(); return true;
      case GKey::KEY_CTRL_END: cmdNavigateBottom(); return true;
      case GKey::KEY_SHIFT_UP: cmdNavigateSelectUp(); return true;
      case GKey::KEY_SHIFT_DOWN: cmdNavigateSelectDown(); return true;
      case GKey::KEY_SHIFT_LEFT: cmdNavigateSelectLeft(); return true;
      case GKey::KEY_SHIFT_RIGHT: cmdNavigateSelectRight(); return true;
      case GKey::KEY_SHIFT_HOME: cmdNavigateSelectHome(); return true;
      case GKey::KEY_SHIFT_END: cmdNavigateSelectEnd(); return true;
      case GKey::KEY_SHIFT_CTRL_LEFT: cmdNavigateSelectPrevWord(); return true;
      case GKey::KEY_SHIFT_CTRL_RIGHT: cmdNavigateSelectNextWord(); return true;
      case GKey::KEY_SHIFT_PAGEUP: cmdNavigateSelectPageUp(); return true;
      case GKey::KEY_SHIFT_PAGEDOWN: cmdNavigateSelectPageDown(); return true;
      case GKey::KEY_SHIFT_CTRL_HOME: cmdNavigateSelectTop(); return true;
      case GKey::KEY_SHIFT_CTRL_END: cmdNavigateSelectBottom(); return true;
      default: return GDecoratedWindow::onKeyDown(key);
   }
}

bool GTextViewer::onTimer ( const GString& timerID, GObject* userData )
{
   if (timerID != "MOUSESELECTOR")
      return GWindow::onTimer(timerID, userData);

   if (!isMouseCapture())
      return true;

   // ---
   if (mousePos.y >= rectScrollClip.y &&
       mousePos.y <= rectScrollClip.y + rectScrollClip.height - 1)
   {
      // Scroll horizontally?
      if (mousePos.x < rectScrollClip.x)
      {
         if (idxFirstVisibleColumn > 0)
         {
            onHScrollLineUp();
            handleMouseMovement(rectScrollClip.x, mousePos.y, false);
         }
      }
      else
      if (mousePos.x > rectScrollClip.x + rectScrollClip.width - 1)
      {
         if (idxFirstVisibleColumn + winColumnNr < widestLine)
         {
            onHScrollLineDown();
            int xpos = rectScrollClip.x + rectScrollClip.width - 1;
            handleMouseMovement(xpos, mousePos.y, false);
         }
      }
   }
   else
   if (mousePos.x >= rectScrollClip.x &&
       mousePos.x <= rectScrollClip.x + rectScrollClip.width - 1)
   {
      // Scroll vertically?
      if (mousePos.y > rectScrollClip.x + rectScrollClip.height - 1)
      {
         if (idxFirstVisibleLine >= 0)
         {
            if (idxFirstVisibleLine > 0)
               onVScrollLineUp();
            handleMouseMovement(mousePos.x, (fonth*winLineNr)-1, false);
         }
      }
      else
      if (mousePos.y < rectScrollClip.y)
      {
         if (idxFirstVisibleLine <= lines.getCount() - winLineNr + 1)
         {
            onVScrollLineDown();
            handleMouseMovement(mousePos.x, rectScrollClip.y, false);
         }
      }
   }

   return true;
}

void GTextViewer::unselectAll ()
{
   if (!isAnySelectedText())
      return;

   int starty = selection.start.y;
   if (starty < idxFirstVisibleLine)
      starty = idxFirstVisibleLine;

   int endy = selection.end.y;
   if (endy >= idxFirstVisibleLine + winLineNr)
      endy = idxFirstVisibleLine + winLineNr - 1;

   selection.clear();

   if (endy < starty)
      return;

   invalidateLines(starty, endy);
}

GTextViewer::Pos GTextViewer::calcXPosFromPhysicalToVirtual ( int x, int y ) const
{
   Pos pos(x, y);
   return calcXPosFromPhysicalToVirtual(pos);
}

GTextViewer::Pos GTextViewer::calcXPosFromPhysicalToVirtual ( const Pos& pos ) const
{
   const GString& line = getIndexedLine(pos.y);
   int lineLen = line.length();
   if (lastExplicitXPosWasEol)
   {
      GString tmp = line;
      tmp.stripTrailingEol();
      lineLen = tmp.length();
   }

   // Let's keep things simple, if there are no tabs.
   if (line.indexOf('\t') < 0)
      return GTextViewer::Pos(GMath::Min(lineLen, pos.x), pos.y);

   // Find virtualX by adjusting xpos for every tab-character.
   int virtualX = 0;
   for (int i=0; i<pos.x && i<lineLen; i++)
   {
      char ch = line[i];
      if (ch == '\t')
         virtualX += opts.tabWidth - (virtualX % opts.tabWidth);
      else
         virtualX += 1;
   }

   // ---
   return GTextViewer::Pos(virtualX, pos.y);
}

GTextViewer::Pos GTextViewer::calcXPosFromVirtualToPhysical ( const Pos& pos ) const
{
   const GString& line = getIndexedLine(pos.y);
   int lineLen = line.length();
   if (line.indexOf('\t') < 0) // No tabs, so lets keep it simple.
      return GTextViewer::Pos(GMath::Min(lineLen, pos.x), pos.y);

   // Find physicalX by adjusting xpos for every tab-character.
   int physicalX = 0;
   for (int i=0; i<pos.x && physicalX<lineLen; physicalX++)
   {
      char ch = line[physicalX];
      if (ch == '\t')
         i += opts.tabWidth - (i % opts.tabWidth);
      else
         i += 1;
   }

   // ---
   return GTextViewer::Pos(physicalX, pos.y);
}

bool GTextViewer::handleMouseMovement ( int xpos, int ypos, bool updateMousePosRecorder )
// Return: True if the caret was actually moved, or else false.
{
   if (updateMousePosRecorder)
   {
      mousePos.x = xpos;
      mousePos.y = ypos;
   }

   if (xpos < 0)
      xpos = 0;
   if (ypos < 0)
      ypos = 0;

   int winLine = (opts.paintFreeTextSpaceAtTop ? ypos : ypos - verticalFreeSpace) / fonth;
   
   // It is allowed for the user to use the mouse to activate the caret 
   // on the line just below the last visible line.
   if (winLine == 0 && ypos < verticalFreeSpace)
      winLine = -1; 

   // ---
   if (idxFirstVisibleLine + winLineNr - winLine - 1 > lines.getCount())
      winLine = winLineNr - (lines.getCount() - idxFirstVisibleLine) - 1;
   int rawY = GMath::Max(0, idxFirstVisibleLine + winLineNr - winLine - 1);
   int rawX = GMath::Max(0, idxFirstVisibleColumn + (xpos / fontw));
   Pos p(rawX, rawY);
   p = calcXPosFromVirtualToPhysical(p);
   setCurrentPos(p);
   lastExplicitXPos = p.x;

   if (!isMouseCapture())
   {
      unselectAll();
      selection.anchor = p;
      selection.start = p;
      selection.end = p;
   }
   else
   {
      GTextViewer::Pos prevStart = selection.start;
      GTextViewer::Pos prevEnd = selection.end;

      if (p <= selection.anchor)
      {
         selection.start = p;
         selection.end = selection.anchor;
      }
      else
      {
         selection.start = selection.anchor;
         selection.end = p;
      }

      // ---
      if (prevStart.y != selection.start.y && prevEnd.y != selection.end.y)
      {
         if (p.y > prevEnd.y)
            paintLine(prevStart.y, p.y - prevStart.y + 1);
         else
            paintLine(p.y, prevEnd.y - p.y + 1);
      }
      else
      if (prevStart.y == selection.start.y && prevEnd.y == selection.end.y)
      {
         if (prevStart.x != selection.start.x || prevEnd.x != selection.end.x)
            paintLine(p.y, 1);
      }
      else
      if (prevStart.y < selection.anchor.y && p.y < selection.anchor.y)
      {
         if (prevStart.y < p.y)
            paintLine(prevStart.y, p.y - prevStart.y + 1);
         else
         if (prevStart.y > p.y)
            paintLine(p.y, prevStart.y - p.y + 1);
         else
            paintLine(p.y, 1);
      }
      else
      if (prevEnd.y > selection.anchor.y && p.y > selection.anchor.y)
      {
         if (prevEnd.y < p.y)
            paintLine(prevEnd.y, p.y - prevEnd.y + 1);
         else
         if (prevEnd.y > p.y)
            paintLine(p.y, prevEnd.y - p.y + 1);
         else
            paintLine(p.y, 1);
      }
      else
      {
         int start = GMath::Min(prevStart.y, p.y);
         int end = GMath::Max(prevEnd.y, p.y);
         paintLine(start, end - start + 1);
      }
   }

   return true;
}

bool GTextViewer::onButton1DblClk ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (ypos >= fonth * winLineNr)
      return true;

   const GString& line = getIndexedLine(curPos.y);
   int lineLen = line.length();
   if (lineLen <= 0)
      return true;

   if (isWordSeparatorChar(line[curPos.x]))
      return true; // Not a word below the mouse double click position

   // Find the start of the word
   int startx;
   for (startx=curPos.x; startx>=0; startx--)
      if (isWordSeparatorChar(line[startx]))
         break;

   // Find the end of the word
   int endx;
   for (endx=startx+1; endx<lineLen; endx++)
      if (isWordSeparatorChar(line[endx]))
         break;

   // Select the word
   int gotoX = endx;
   gotoPos(curPos.y, gotoX, true); // Make sure that the caret will come into the visible area
   selection.anchor.set(startx+1, curPos.y);
   selection.start.set(startx+1, curPos.y);
   selection.end.set(gotoX, curPos.y);
   invalidateLine(selection.end.y);

   return true;
}

bool GTextViewer::onButton1Down ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   handleMouseMovement(xpos, ypos);

   if (isActive()) // If we are part of the active frame window.
   {
      grabFocus(); // Give keyboard input focus to the text viewer window.

      // Capture the mouse while we wait for the mouse button to be
      // released. We will selected text while the mouse cursor is moved
      // as long as it is captured.
      captureMouse(true);

      startTimer("MOUSESELECTOR", 30);
   }
   else
   {
      setActive(); // Just activate the frame (set it on top of Z-order).
   }

   return true;
}

bool GTextViewer::onButton1Up ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
   {
      stopTimer("MOUSESELECTOR");
      captureMouse(false);
   }
   return true;
}

bool GTextViewer::onButton2Down ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
   {
      // Copy the current selected text to the clipboard.
      cmdCopy();
   }
   else
   {
      // Show the popup-menu.
      grabFocus();
      if (isVisiblePopupMenu())
         setVisiblePopupMenu(false);
      else
         setVisiblePopupMenu(true, xpos, ypos);
   }

   return true;
}

/**
 * Return true if and only if you have actually set the mouse cursor.
 */
bool GTextViewer::onMouseMove ( int xpos, int ypos, const GWindowMessage::InputFlags& /*flags*/ )
{
   if (isMouseCapture())
      handleMouseMovement(xpos, ypos);
   return false;
}

bool GTextViewer::onInitMenu ()
{
   setCommandEnableState("cmdEditCopy", isAnySelectedText());
   setCommandEnableState("cmdEditSearch", lines.getCount() > 0);
   setCommandEnableState("cmdEditSearchNext", lines.getCount() > 0 && isFindNextReady());
   setCommandEnableState("cmdEditGotoLineNum", lines.getCount() > 0);
   setCommandEnableState("cmdEditSelectAll", lines.getCount() > 0);
   return true;
}

bool GTextViewer::isSelectedTextSingleLine () const
{
   if (!isAnySelectedText())
      return false;
   if (selection.start.y != selection.end.y)
      return false;
   return true;
}

GString& GTextViewer::getSelectedText ( GString& returnStr ) const
{
   returnStr = "";
   if (!isAnySelectedText())
      return returnStr;

   int selstartY = selection.start.y; // Make this a fast one
   int selendY = selection.end.y; // Make this a fast one
   int selstartX = selection.start.x; // Make this a fast one
   int selendX = selection.end.x; // Make this a fast one

   // Build the text string of which to copy.
   GString tmpLine(256);
   int lineLen = getIndexedLine(selstartY).length();
   if (selstartX < lineLen)
   {
      int len = (selendY==selstartY ? selendX-selstartX : lineLen-selstartX);
      tmpLine = getIndexedLine(selstartY);
      tmpLine.cutTailFrom(selstartX + len);
      tmpLine.remove(0, selstartX);
      tmpLine.ensureCompatibleEolFormat();
      returnStr += tmpLine;
   }
   for (int i=selstartY+1; i<selendY; i++)
   {
      tmpLine = getIndexedLine(i);
      tmpLine.ensureCompatibleEolFormat();
      returnStr += tmpLine;
   }
   if (selendY > selstartY && selendY < lines.getCount())
   {
      tmpLine = getIndexedLine(selendY);
      tmpLine.cutTailFrom(selendX);
      tmpLine.ensureCompatibleEolFormat();
      returnStr += tmpLine;
   }

   return returnStr;
}

void GTextViewer::cmdCopy ( GAbstractCommand* /*cmd*/ )
{
   GString txt(4096);
   getSelectedText(txt);
   if (txt != "")
      GSystem::CopyTextToClipboard(txt);
}

void GTextViewer::cmdUnselectAll ( GAbstractCommand* /*cmd*/ )
{
   unselectAll();
}

void GTextViewer::cmdSelectAll ( GAbstractCommand* /*cmd*/ )
{
   if (lines.getCount() > 0)
   {
      navigateDown(false);
      selection.anchor.set(0, 0);
      selection.start.set(0, 0);
      int lc = lines.getCount();
      selection.end.set(0, lc);
      invalidateAll(false);
   }
}

void GTextViewer::cmdNavigateUp ( GAbstractCommand* /*cmd*/ )
{
   navigateUp(false);
}

void GTextViewer::cmdNavigateDown ( GAbstractCommand* /*cmd*/ )
{
   navigateDown(false);
}

void GTextViewer::cmdNavigateLeft ( GAbstractCommand* /*cmd*/ )
{
   navigateLeft(false);
}

void GTextViewer::cmdNavigateRight ( GAbstractCommand* /*cmd*/ )
{
   navigateRight(false);
}

void GTextViewer::cmdNavigateHome ( GAbstractCommand* /*cmd*/ )
{
   navigateHome(false);
}

void GTextViewer::cmdNavigateEnd ( GAbstractCommand* /*cmd*/ )
{
   navigateEnd(false);
}

void GTextViewer::cmdNavigatePrevWord ( GAbstractCommand* /*cmd*/ )
{
   navigatePrevWord(false);
}

void GTextViewer::cmdNavigateNextWord ( GAbstractCommand* /*cmd*/ )
{
   navigateNextWord(false);
}

void GTextViewer::cmdNavigatePageUp ( GAbstractCommand* /*cmd*/ )
{
   navigatePageUp(false);
}

void GTextViewer::cmdNavigatePageDown ( GAbstractCommand* /*cmd*/ )
{
   navigatePageDown(false);
}

void GTextViewer::cmdNavigateTop ( GAbstractCommand* /*cmd*/ )
{
   navigateTop(false);
}

void GTextViewer::cmdNavigateBottom ( GAbstractCommand* /*cmd*/ )
{
   navigateBottom(false);
}

void GTextViewer::cmdNavigateSelectUp ( GAbstractCommand* /*cmd*/ )
{
   navigateUp(true);
}

void GTextViewer::cmdNavigateSelectDown ( GAbstractCommand* /*cmd*/ )
{
   navigateDown(true);
}

void GTextViewer::cmdNavigateSelectLeft ( GAbstractCommand* /*cmd*/ )
{
   navigateLeft(true);
}

void GTextViewer::cmdNavigateSelectRight ( GAbstractCommand* /*cmd*/ )
{
   navigateRight(true);
}

void GTextViewer::cmdNavigateSelectHome ( GAbstractCommand* /*cmd*/ )
{
   navigateHome(true);
}

void GTextViewer::cmdNavigateSelectEnd ( GAbstractCommand* /*cmd*/ )
{
   navigateEnd(true);
}

void GTextViewer::cmdNavigateSelectPrevWord ( GAbstractCommand* /*cmd*/ )
{
   navigatePrevWord(true);
}

void GTextViewer::cmdNavigateSelectNextWord ( GAbstractCommand* /*cmd*/ )
{
   navigateNextWord(true);
}

void GTextViewer::cmdNavigateSelectPageUp ( GAbstractCommand* /*cmd*/ )
{
   navigatePageUp(true);
}

void GTextViewer::cmdNavigateSelectPageDown ( GAbstractCommand* /*cmd*/ )
{
   navigatePageDown(true);
}

void GTextViewer::cmdNavigateSelectTop ( GAbstractCommand* /*cmd*/ )
{
   navigateTop(true);
}

void GTextViewer::cmdNavigateSelectBottom ( GAbstractCommand* /*cmd*/ )
{
   navigateBottom(true);
}

bool GTextViewer::incrementPos ( GTextViewer::Pos& pos, bool forward ) const
{
   if (forward)
   {
      int linesCount = lines.getCount();
      if (pos.y >= linesCount)
         return false;
      if (pos.y < 0)
      {
         pos.y = 0;
         pos.x = 0;
         return true;
      }
      else
      if (pos.x < lines[pos.y].length() - 1)
      {
         pos.x++;
         return true;
      }
      else
      {
         if (pos.y >= linesCount - 1)
            return false;
         pos.y++;
         pos.x = 0;
         return true;
      }
   }
   else
   {
      if (pos.y < 0)
         return false;
      int linesCount = lines.getCount();
      if (pos.y >= linesCount)
      {
         pos.y = linesCount - 1;
         pos.x = lines[pos.y].length() - 1;
         return true;
      }
      else
      if (pos.x > 0)
      {
         pos.x--;
         return true;
      }
      else
      {
         if (pos.y <= 0)
            return false;
         pos.y--;
         pos.x = lines[pos.y].length() - 1;
         return true;
      }
   }
}

bool GTextViewer::isWordSeparatorChar ( char chr ) const
{
   if (chr <= 32 || opts.wordWrapChars.indexOf(chr) >= 0)
      return true;
   else
      return false;
}

bool GTextViewer::isCharactersTheSame ( char chr1, char chr2, bool caseSen ) const
{
   if (caseSen)
      return chr1 == chr2;
   else
      return tolower(chr1) == tolower(chr2);
}

char GTextViewer::getCharAt ( const GTextViewer::Pos& pos ) const
{
   return lines[pos.y][pos.x];
}

bool GTextViewer::matchStringAt ( const GString& str,
                                  const GTextViewer::Pos& pos,
                                  bool searchWord, 
                                  bool searchCaseSen,
                                  bool searchForeward ) const
{
   GTextViewer::Pos p = pos;
   int strLen = str.length();
   if (strLen <= 0)
      return false;

   if (lines.getCount() <= 0 ||
       lines.getCount() == 1 && lines[0].length() <= 0)
   {
      return false;
   }

   bool hasMore = false;
   for (int i=0; i<strLen; i++) 
   {
      char chr = getCharAt(p);
      if (!isCharactersTheSame(str[i], chr, searchCaseSen))
         return false;
      hasMore = incrementPos(p, searchForeward);
      if (!hasMore)
         return false;
   }

   if (searchWord && hasMore)
   {
      char chr = getCharAt(p);
      if (!isWordSeparatorChar(chr))
         return false;
   }

   return true;
}

bool GTextViewer::findNextText ( GTextViewer::Pos& pos ) const
{
   if (!isFindNextReady())
      return false;

   GString str = searchParams->searchString;   // Make this a fast one.
   bool caseSen = searchParams->searchCaseSen; // Ditto.
   bool wholeWord = searchParams->searchWord;  // Ditto.
   bool forward = searchParams->searchForward; // Ditto.
   if (!forward)
      str.reverse();

   if (pos.y >= lines.getCount() || pos.x >= lines[pos.y].length())
   {
      if (!incrementPos(pos, forward))
         return false;
   }

   for (;;)
   {
      if (wholeWord)
      {
         // Find the beginning of the next "word".
         bool anyWhite = false;
         for (;;)
         {
            char chr = getCharAt(pos);
            bool isWhite = isWordSeparatorChar(chr);
            if (!isWhite && anyWhite)
               break;
            if (!incrementPos(pos, forward))
               return false;
            if (isWhite)
               anyWhite = true;
         }
      }

      if (matchStringAt(str, pos, wholeWord, caseSen, forward))
         return true;
      if (!incrementPos(pos, forward))
         return false;
   }
}

void GTextViewer::cmdGoto ( GAbstractCommand* /*cmd*/ )
{
   int line = getUserSelectedLineNumber();
   if (line >= 1 && line <= getLinesCount())
   {
      int column = getIndexOfFirstVisibleColumn();
      gotoPos(line - 1, column);
   }
}

void GTextViewer::cmdSearch ( GAbstractCommand* /*cmd*/ )
{
   if (searchParams == null)
      return;
   if (searchParams->userEditSearchParams(*this))
      searchNextImpl();
}

void GTextViewer::cmdSearchNext ( GAbstractCommand* /*cmd*/ )
{
   if (searchParams == null)
      return;
   if (searchParams->searchString == "")
      return cmdSearch();
   searchNextImpl();
}

bool GTextViewer::searchNextImpl ()
{
   GTextViewer::Pos pos = curPos;
   if (findNextText(pos))
   {
      // Select the found text, and put the caret at the correct position
      // with respect to forward or backward search.
      if (searchParams->searchForward)
      {
         GTextViewer::Pos pos2 = pos;
         for (int i=0, len=searchParams->searchString.length(); i<len; i++)
            incrementPos(pos2, true);
         gotoPos(pos2);
         setSelectionStartPos(pos);
         setSelectionEndPos(pos2);
      }
      else
      {
         incrementPos(pos);
         GTextViewer::Pos pos2 = pos;
         for (int i=0, len=searchParams->searchString.length(); i<len; i++)
            incrementPos(pos2, false);
         gotoPos(pos2);
         setSelectionStartPos(pos2);
         setSelectionEndPos(pos);
      }
      return true;
   }
   else
   {
      searchParams->onSearchStringNotFound(*this);
      return false;
   }
}

void GTextViewer::setQuickScrollMode ( bool flag )
{
   if (!flag == !opts.quickScrollMode)
      return;
   opts.quickScrollMode = flag;
}

void GTextViewer::setShowLinefeeds ( bool flag )
{
   if (!flag == !opts.showLinefeeds)
      return;
   opts.showLinefeeds = flag;
   invalidateAll(false);
}

void GTextViewer::setShowSpaces ( bool flag )
{
   if (!flag == !opts.showSpaces)
      return;
   opts.showSpaces = flag;
   invalidateAll(false);
}

void GTextViewer::setShowTabs ( bool flag )
{
   if (!flag == !opts.showTabs)
      return;
   opts.showTabs = flag;
   invalidateAll(false);
}

void GTextViewer::updateWidestLine ( int index )
{
   if (index < 0 || index >= lines.getCount())
      return;
   const GString& line = lines[index];
   const GString& textToDraw = makeVirtualLine(line, 0);
   int len = textToDraw.length();
   if (len > widestLine)
      widestLine = len;
}

bool GTextViewer::wordWrapLineAt ( int index )
{
   const int columns = (opts.forceEOLPos>0 ? GMath::Min(opts.forceEOLPos+1, winColumnNr) : winColumnNr);
   const int lastVisibleColumnIdx = GMath::Max(0, columns);
   const GString& l = lines.get(index);
   int columnIdxOnScreen = 0;
   int idxOfLastWhite = -1;
   int columnIdxOnScreenOfLastWhite = -1;
   for (int i=0, llen=l.length(); i<llen; i++)
   {
      char chr = l[i];
      if (chr == '\t')
      {
         columnIdxOnScreen += opts.tabWidth - (columnIdxOnScreen % opts.tabWidth);
         idxOfLastWhite = i;
         columnIdxOnScreenOfLastWhite = columnIdxOnScreen;
      }
      else
      if (isWordSeparatorChar(chr))
      {
         columnIdxOnScreen += 1;
         idxOfLastWhite = i;
         columnIdxOnScreenOfLastWhite = columnIdxOnScreen;
      }
      else
      {
         columnIdxOnScreen += 1;
      }

      // ---
      if (columnIdxOnScreen >= lastVisibleColumnIdx)
      {
         if (idxOfLastWhite >= 0)
         {
            // Include all consecutive trailing whitespace characters,
            // but make sure not to exceed the "max line length" limit.
            while (idxOfLastWhite<llen &&
                   columnIdxOnScreenOfLastWhite<=opts.forceEOLPos)
            {
               chr = l[idxOfLastWhite];
               if (!isWordSeparatorChar(chr))
                  break;
               else
               if (chr == '\t')
                  columnIdxOnScreenOfLastWhite += opts.tabWidth - (columnIdxOnScreenOfLastWhite % opts.tabWidth);
               else
                  columnIdxOnScreenOfLastWhite += 1;
               idxOfLastWhite++;
            }

            // Word break index.
            i = idxOfLastWhite;
         }
         else
         {
            // No whitespace found, so we must do a "hard" break,
            // possibly in the middle of a (probably very long-) word.
         }

         if (i > 0 && i < llen-1)
         {
            GString* rest = new GString(l.substring(i));
            lines.insert(rest, index+1);
            lines[index] = l.substring(0, i);
            return true; // Yes, the line was wrapped.
         }

         break;
      }
   }

   return false; // No, the line was not wrapped.
}

bool GTextViewer::respectForceEOLPosAt ( int lineIndex )
{
   const GString& line = lines.get(lineIndex);
   for (int endIdx=0, virtualX=0, llen=line.length();
        endIdx<llen;
        endIdx++)
   {
      if (line[endIdx] == '\t')
         virtualX += opts.tabWidth - (virtualX % opts.tabWidth);
      else
         virtualX += 1;

      if (virtualX > opts.forceEOLPos)
      {
         GString* tail = new GString(line.substring(endIdx));
         lines.insert(tail, lineIndex + 1);
         lines[lineIndex] = line.substring(0, endIdx);
         return true; // Yes, the line was wrapped.
      }
   }

   return false; // No, the line was not wrapped.
}

bool GTextViewer::wrapLineIfNeededAt ( int lineIndex )
{
   if (opts.wordWrap)
      return wordWrapLineAt(lineIndex);
   else
      return respectForceEOLPosAt(lineIndex);
}

void GTextViewer::setWordWrapImpl ( bool flag, bool temp )
{
   navigateTop(false);

   // First, convert all lines into physical lines with no wrapping at all,
   // not even respecting the "maximum line length" setting.
   if (opts.wordWrap)
   {
      for (int lineIdx=0; lineIdx<lines.getCount()-1; )
      {
         char lastChar = lines[lineIdx].lastChar();
         if (lastChar != '\n' && lastChar != '\r')
         {
            int nextLineIdx = lineIdx + 1;
            lines[lineIdx] += lines[nextLineIdx];
            lines.remove(nextLineIdx);
         }
         else
         {
            lineIdx++;
         }
      }
   }

   widestLine = 0;
   opts.wordWrap = flag;

   // Then, wrap the lines.
   for (int i=0; i<lines.getCount(); i++)
   {
      wrapLineIfNeededAt(i);
      if (!temp)
         updateWidestLine(i);
   }

   // ---
   stopSelectionMode();
   unselectAll();
   if (!temp)
   {
      updateScrollBarPosAndRange();
      invalidateAll(false);
   }
}

void GTextViewer::setWordWrap ( bool flag )
{
   if (!flag == !opts.wordWrap)
      return;

   setWordWrapImpl(flag, false);
}

bool GTextViewer::onPaint ( GGraphics& g, const GRectangle& rect )
{
   GRectangle r = getWindowRect();

   int ypos = r.y + r.height - 1;
   if (opts.paintFreeTextSpaceAtTop)
   {
      ypos -= verticalFreeSpace;
      // Clear the small possible topmost area of no text.
      GRectangle space = r;
      space.y = space.height - verticalFreeSpace;
      GColor bck = getBackgroundColor();
      g.drawFilledRectangle(space, bck);
   }

   // Paint all text lines of the visible space.
   const int numl = lines.getCount();
   for (int i=idxFirstVisibleLine, lasti=idxFirstVisibleLine+winLineNr-1;
        i>=0 && i<=lasti && i<numl;
        i++, ypos -= fonth)
   {
      if (ypos >= rect.y &&
          ypos < rect.y + rect.height - 1 + fonth)
      {
         paintLineImpl(g, i, r.width, ypos);
      }
   }

   // Fill rest of window with background.
   if (ypos >= 0)
   {
      r.y = 0;
      r.height = ypos + 1;
      GColor bck = getBackgroundColor();
      g.drawFilledRectangle(r, bck);
   }

   return true;
}

bool GTextViewer::onFontNameSizeChanged ( const GString& fontNameSize )
{
   // Make sure that "fontw" and "fonth" is not zero, to prevent divide-by-zero.
   // This should normally not be needed, but I have seen an exception due
   // to this in some very rare cases. Therefore, keep this code just in case.
   GString charsBase("ABCDEFGHIJKLMNOPQRSTUWWXYZabcdefghijklmnopqrstuvwxyz0123456789");
   fontw = getWidthOfString(charsBase) / charsBase.length();
   fonth = getHeightOfString("X");
   if (fontw == 0)
      fontw = 1;
   if (fonth == 0)
      fonth = 1;

   // Update the size of the caret to respect the new font size.
   int caretW = fontw / 10;
   if (caretW < 2)
      caretW = 2;
   setCaretSize(caretW, fonth);

   // Recalculate everything that has to do with the
   // presentation of the new font.
   invalidateAll(false);
   GDimension dim = getWindowSize();
   onSize(dim.width, dim.height);

   return true;
}

bool GTextViewer::onSize ( int width, int height )
{
   winColumnNr = width / fontw;
   winLineNr = height / fonth;
   verticalFreeSpace = height % fonth;

   const GRectangle newSize = getWindowRect();

   // ---
   rectScrollClip = newSize;
   rectScrollClip.height -= verticalFreeSpace;
   if (!opts.paintFreeTextSpaceAtTop)
      rectScrollClip.y += verticalFreeSpace;

   if (opts.useAsConsoleMonitor && !hasFocus())
   {
      int lc = lines.getCount();
      idxFirstVisibleLine = GMath::Max(0, lc - winLineNr + 1);
      gotoPos(lc, 0);
   }

   updateScrollBarPosAndRange();

   // Make sure to update the bottom area of the window as needed.
   int newH = newSize.height;
   int prevH = previousSize.height;
   if (opts.paintFreeTextSpaceAtTop)
   {
      // Must invalidate the whole rectangle,
      // but only if the window height has changed.
      if (newH != prevH)
         invalidateRect(newSize);
   }
   else
   {
      GRectangle r = newSize;
      if (newH < prevH)
         r.height = fonth;
      else
      if (newH > prevH)
         r.height = newH - prevH + fonth;
      invalidateRect(r);
   }

   // Remember the size until the next time the window is resized.
   previousSize = newSize;

   // Update word-wraps if needed.
   if (opts.wordWrap)
   {
      setWordWrapImpl(false, true); // Set word wrap off (temporary).
      setWordWrapImpl(true, false); // Set word wrap on.
   }

   // ---
   return true;
}

void GTextViewer::onFocusSet ()
{
   setCurrentPos(curPos);
   GDecoratedWindow::onFocusSet();
}

void GTextViewer::updateScrollBarPos ()
{
   setCurrentPos(curPos);
   setHScrollPos(idxFirstVisibleColumn);
   setVScrollPos(idxFirstVisibleLine);
}

void GTextViewer::updateScrollBarRange ()
{
   // The vertical scrollbar
   int thumbLen = winLineNr;
   int scrollLen = lines.getCount() - thumbLen + 1;
   setVScrollRange(scrollLen, thumbLen);

   // The horizontal scrollbar
   thumbLen  = winColumnNr;
   scrollLen = widestLine - thumbLen + 1;
   setHScrollRange(scrollLen, thumbLen);
}

void GTextViewer::updateScrollBarPosAndRange ()
{
   setCurrentPos(curPos);

   // The vertical scrollbar
   int thumbLen = winLineNr;
   int scrollLen = lines.getCount() - thumbLen + 1;
   setVScrollPosAndRange(idxFirstVisibleLine, scrollLen, thumbLen);

   // The horizontal scrollbar
   thumbLen  = winColumnNr;
   scrollLen = widestLine - thumbLen + 1;
   setHScrollPosAndRange(idxFirstVisibleColumn, scrollLen, thumbLen);
}

bool GTextViewer::onVScrollLineUp ()
{
   int prevLine1 = idxFirstVisibleLine;

   int newLine1 = GMath::Clip(idxFirstVisibleLine-1, 0, lines.getCount() - winLineNr + 1);
   if (newLine1 >= 0 && newLine1 != prevLine1)
   {
      idxFirstVisibleLine = newLine1;
      scrollWindow(0, -fonth, true, null, &rectScrollClip);
      updateScrollBarPos();
   }

   return true;
}

bool GTextViewer::onVScrollLineDown ()
{
   int prevLine1 = idxFirstVisibleLine;

   int newLine1 = GMath::Clip(idxFirstVisibleLine+1, 0, lines.getCount() - winLineNr + 1);
   if (newLine1 >= 0 && newLine1 != prevLine1)
   {
      idxFirstVisibleLine = newLine1;
      scrollWindow(0, fonth, true, null, &rectScrollClip);
      updateScrollBarPos();
   }

   return true;
}

bool GTextViewer::onVScrollPageUp ()
{
   int prevLine1 = idxFirstVisibleLine;

   int sub = winLineNr - 1;
   if (sub <= 0)
      sub = 1;

   int newLine1 = GMath::Clip(idxFirstVisibleLine-sub, 0, lines.getCount() - winLineNr + 1);
   if (newLine1 >= 0 && newLine1 != prevLine1)
   {
      idxFirstVisibleLine = newLine1;
      updateScrollBarPos();
      invalidateAll(false);
   }

   return true;
}

bool GTextViewer::onVScrollPageDown ()
{
   int prevLine1 = idxFirstVisibleLine;

   int add = winLineNr - 1;
   if (add <= 0)
      add = 1;

   int newLine1 = GMath::Clip(idxFirstVisibleLine+add, 0, lines.getCount() - winLineNr + 1);
   if (newLine1 >= 0 && newLine1 != prevLine1)
   {
      idxFirstVisibleLine = newLine1;
      updateScrollBarPos();
      invalidateAll(false);
   }

   return true;
}

bool GTextViewer::onVScrollSliderTrack ( int pos )
{
   int prevLine1 = idxFirstVisibleLine;
   idxFirstVisibleLine = GMath::Clip(pos, 0, lines.getCount() - winLineNr + 1);
   if (idxFirstVisibleLine != prevLine1)
   {
      int diff = idxFirstVisibleLine - prevLine1;
      if (GMath::Abs(diff) >= winLineNr)
         invalidateAll(false);
      else
         scrollWindow(0, fonth*diff, true, null, &rectScrollClip);
      setVScrollPos(idxFirstVisibleLine); // In case we are called programatically.
   }

   return true;
}

bool GTextViewer::onHScrollSliderTrack ( int pos )
{
   int prevCol1 = idxFirstVisibleColumn;
   idxFirstVisibleColumn = GMath::Clip(pos, 0, widestLine - winColumnNr + 1);
   if (idxFirstVisibleColumn != prevCol1)
   {
      int diff = prevCol1 - idxFirstVisibleColumn;
      if (GMath::Abs(diff) >= winColumnNr)
         invalidateAll(false);
      else
         scrollWindow(fontw*diff, 0, true, null, &rectScrollClip);
      setHScrollPos(idxFirstVisibleColumn); // In case we are called programatically.
   }

   return true;
}

bool GTextViewer::onHScrollLineUp ()
{
   if (idxFirstVisibleColumn > 0)
   {
      int prevCol1 = idxFirstVisibleColumn;
      idxFirstVisibleColumn = GMath::Clip(idxFirstVisibleColumn-1, 0, widestLine - winColumnNr + 1);
      if (idxFirstVisibleColumn != prevCol1)
      {
         updateScrollBarPos();
         scrollWindow(fontw, 0, true, null, &rectScrollClip);
      }
   }

   return true;
}

bool GTextViewer::onHScrollLineDown ()
{
   if (idxFirstVisibleColumn + winColumnNr < widestLine)
   {
      int prevCol1 = idxFirstVisibleColumn;
      idxFirstVisibleColumn = GMath::Clip(idxFirstVisibleColumn+1, 0, widestLine - winColumnNr + 1);
      if (idxFirstVisibleColumn != prevCol1)
      {
         updateScrollBarPos();
         scrollWindow(-fontw, 0, true, null, &rectScrollClip);
      }
   }

   return true;
}

bool GTextViewer::onHScrollPageUp ()
{
   int prevCol1 = idxFirstVisibleColumn;

   int sub = 10;
   if (sub >= winColumnNr)
      sub = winColumnNr - 1;
   if (sub <= 0)
      sub = 1;

   idxFirstVisibleColumn = GMath::Clip(idxFirstVisibleColumn-sub, 0, widestLine - winColumnNr + 1);
   if (idxFirstVisibleColumn != prevCol1)
   {
      updateScrollBarPos();
      invalidateAll(false);
   }

   return true;
}

bool GTextViewer::onHScrollPageDown ()
{
   int prevCol1 = idxFirstVisibleColumn;

   int add = 10;
   if (add >= winColumnNr)
      add = winColumnNr - 1;
   if (add <= 0)
      add = 1;

   idxFirstVisibleColumn = GMath::Clip(idxFirstVisibleColumn+add, 0, widestLine - winColumnNr + 1);
   if (idxFirstVisibleColumn != prevCol1)
   {
      updateScrollBarPos();
      invalidateAll(false);
   }

   return true;
}

const GString& GTextViewer::makeVirtualLine ( const GString& line, int fromX ) const
{
   int lineLen = line.length();
   if (lineLen == 0)
      return line;

   vlineBuff.clear();

   for (int i=0, pos=fromX; i<lineLen; i++)
   {
      char c = line[i];
      if (c == '\t')
      {
         int num = opts.tabWidth - (pos % opts.tabWidth);
         if (opts.showTabs)
         {
            vlineBuff += opts.chrShowTab;
            if (num > 1)
               vlineBuff.append(' ', num - 1);
         }
         else
         {
            vlineBuff.append(' ', num);
         }
         pos += num;
      }
      else
      {
         if (c == ' ' && opts.showSpaces)
            vlineBuff += opts.chrShowSpace;
         else
            vlineBuff += c;
         pos += 1;
      }
   }

   if (vlineBuff.endsWithEol())
   {
      int len1 = vlineBuff.length();
      vlineBuff.stripTrailingEol();
      int num = len1 - vlineBuff.length(); // Number of characters removed.
      if (opts.showLinefeeds)
      {
         // Show each CR/LF as a single specific character.
         vlineBuff += opts.chrShowLinefeed;
         num -= 1;
      }
      // Show a normal space in place of each normal CR/LF respectively.
      if (num > 0)
         vlineBuff.append(' ', num);
   }

   return vlineBuff;
}

bool GTextViewer::isAnySelectedText () const
{
   return selection.isAnySelection();
}

int GTextViewer::calcYPosOfIndexedLine ( int idx ) const
{
   return rectScrollClip.y + rectScrollClip.height - 1 - 
          fonth * (idx - idxFirstVisibleLine);
}

GRectangle GTextViewer::calcRectOfIndexedLine ( int idx ) const
{
   GRectangle r = rectScrollClip;
   r.y = calcYPosOfIndexedLine(idx) - fonth + 1;
   r.height = fonth;
   return r;
}

void GTextViewer::PaintText ( GGraphics& g,
                              GRectangle& r,
                              const GString& txt,
                              const GColor& bck,
                              const GColor& frg )
{
   g.drawText(txt, r, frg, bck, GGraphics::LEFT, GGraphics::TOP);
}

void GTextViewer::paintLineImpl ( GGraphics& g,
                                  int lineIndex,
                                  int width,
                                  int ypos )
{
   // First, separate the three substrings of the indexed line, that must
   // be painted separately with different color.
   const GString& line = lines.get(lineIndex);
   GString txt1; // First part, to be painted with normal colors.
   GString txt2; // Second part, to be painted with "selection" colors.
   GString txt3; // Third part, to be painted with normal colors.
   if (selection.isEmpty() ||
       lineIndex < selection.start.y || lineIndex > selection.end.y)
   {
      // Line is not selected at all. 
      // So paint the whole line using normal colors.
      txt1 = makeVirtualLine(line, 0);
   }
   else
   if (lineIndex > selection.start.y && lineIndex < selection.end.y)
   {
      // All parts of the line is selected.
      // So paint the whole line using selection colors.
      txt2 = makeVirtualLine(line, 0);
   }
   else
   if (lineIndex == selection.start.y && lineIndex == selection.end.y)
   {
      // The indexed line contains all parts of the selection.
      // Split into prefix (txt1), middle (txt2) and postfix (txt3) text.
      int pos = 0;
      if (selection.start.x > 0)
      {
         GString pre = line.substring(0, selection.start.x);
         txt1 = makeVirtualLine(pre, pos);
         pos += txt1.length();
      }
      GString mid = line.substring(selection.start.x, selection.end.x);
      txt2 = makeVirtualLine(mid, pos);
      char lastCharOfMid = mid.lastChar();
      bool lastCharOfMidIsEOL = (lastCharOfMid == '\r' || lastCharOfMid == '\n');
      if (selection.end.x < line.length() && !lastCharOfMidIsEOL)
      {
         pos += txt2.length();
         GString post = line.substring(selection.end.x);
         txt3 = makeVirtualLine(post, pos);
      }
   }
   else
   if (lineIndex == selection.start.y)
   {
      // The indexed lines contains the beginning of the selection.
      int pos = 0;
      if (selection.start.x > 0)
      {
         GString pre = line.substring(0, selection.start.x);
         txt1 = makeVirtualLine(pre, pos);
         pos += txt1.length();
      }
      GString post = line.substring(selection.start.x);
      txt2 = makeVirtualLine(post, pos);
   }
   else
   if (lineIndex == selection.end.y)
   {
      // The indexed lines contains the end of the selection.
      int pos = 0;
      txt1 = "";
      GString pre = line.substring(0, selection.end.x);
      txt2 = makeVirtualLine(pre, pos);
      pos += txt2.length();
      char lastCharOfPre = pre.lastChar();
      bool lastCharOfPreIsEOL = (lastCharOfPre == '\r' || lastCharOfPre == '\n');
      if (selection.end.x < line.length() && !lastCharOfPreIsEOL)
      {
         GString post = line.substring(selection.end.x);
         txt3 = makeVirtualLine(post, pos);
      }
   }
   else
   {
      // This could really never happen. Logically impossible!
   }

   // Paint the three parts of the line.
   GRectangle r;
   r.x = -(idxFirstVisibleColumn * fontw);
   r.width = width - r.x + 1;
   r.y = ypos - fonth + 1;
   r.height = fonth;

   GColor bck = getBackgroundColor();
   GColor frg = getForegroundColor();

   if (txt1 != "")
   {
      r.width = g.getWidthOfString(txt1);
      PaintText(g, r, txt1, bck, frg);
      r.x += r.width;
   }

   if (txt2 != "")
   {
      r.width = g.getWidthOfString(txt2);
      PaintText(g, r, txt2, frg, bck);
      r.x += r.width;
   }

   if (txt3 != "")
   {
      r.width = g.getWidthOfString(txt3);
      PaintText(g, r, txt3, bck, frg);
      r.x += r.width;
   }

   // Clear the rest of the line (empty area).
   r.width = width - r.x;
   PaintText(g, r, GString::Empty, bck, frg);

   // Draw a thin vertical line at the "max line length" margine.
   if (opts.showForceEOLPosVLine && opts.forceEOLPos > 0)
   {
      g.setColor(opts.colorOfForceEOLPosVLine);
      int x = (opts.forceEOLPos * fontw) - (idxFirstVisibleColumn * fontw);
      g.drawLine(x, ypos, x, ypos - fonth);
   }
}

void GTextViewer::paintLine ( int idx, int count )
{
   if (idx < idxFirstVisibleLine)
   {
      int delta = idxFirstVisibleLine - idx;
      idx += delta;
      count -= delta;
   }

   if (count <= 0)
      return;

   showCaret(false);

   GGraphics g(this);
   GRectangle r = getWindowRect();
   int lastIdx = idxFirstVisibleLine + winLineNr - 1;
   int ypos = calcYPosOfIndexedLine(idx);
   for (int i=idx, num=idx+count; i<num; i++, ypos-=fonth)
   {
      // Don't spend time painting outside visible area!
      if (i > lastIdx)
         break; 

      // ---
      if (i >= lines.getCount()) // If indexed line is outside range
      {
         GRectangle rc(0, ypos - fonth, r.width, fonth);
         GColor bck = getBackgroundColor();
         g.drawFilledRectangle(rc, bck); // Paint the line by clearing its area
      }
      else
      {
         paintLineImpl(g, i, r.width, ypos);
      }
   }

   showCaret(true);
}

void GTextViewer::onCaretPositionChanged ()
{
}

void GTextViewer::restoreCaretPositioningAndView ( const GTextViewer::Pos& pos, 
                                                   int firstVisibleLine_,
                                                   int firstVisibleColumn_ )
{
   gotoPos(pos.y, pos.x);
   if (firstVisibleLine_ >= 0 && firstVisibleLine_ < getLinesCount())
      idxFirstVisibleLine = firstVisibleLine_;
   if (firstVisibleColumn_ >= 0 && firstVisibleColumn_ < widestLine)
      idxFirstVisibleColumn = firstVisibleColumn_;
   updateScrollBarPosAndRange();
   invalidateAll(true);
}

GTextViewer::Pos GTextViewer::getCurrentPos () const
{
   return curPos;
}

int GTextViewer::getCurrentPosX () const
{
   return curPos.getX();
}

int GTextViewer::getCurrentPosY () const
{
   return curPos.getY();
}

int GTextViewer::getFontHeight () const
{
   return fonth;
}

int GTextViewer::getIndexOfFirstVisibleLine () const
{
   return idxFirstVisibleLine;
}

int GTextViewer::getLinesCount () const
{
   return lines.getCount();
}

int GTextViewer::getIndexOfFirstVisibleColumn () const
{
   return idxFirstVisibleColumn;
}

int GTextViewer::getWidestLine () const
{
   return widestLine;
}

int GTextViewer::getWindowVisibleColumnsCount () const
{
   return winColumnNr;
}

int GTextViewer::getWindowVisibleLinesCount () const
{
   return winLineNr;
}

bool GTextViewer::isFindNextReady () const
{
   return searchParams != null &&
          searchParams->searchString.length() > 0;
}

bool GTextViewer::isQuickScrollMode () const
{
   return opts.quickScrollMode;
}

bool GTextViewer::isSelectionMode () const
{
   return selectionMode;
}

bool GTextViewer::isShowLinefeeds () const
{
   return opts.showLinefeeds;
}

bool GTextViewer::isShowSpaces () const
{
   return opts.showSpaces;
}

bool GTextViewer::isShowTabs () const
{
   return opts.showTabs;
}

char GTextViewer::getChrShowLinefeed () const
{
   return opts.chrShowLinefeed;
}

char GTextViewer::getChrShowSpace () const
{
   return opts.chrShowSpace;
}

char GTextViewer::getChrShowTab () const
{
   return opts.chrShowTab;
}

void GTextViewer::setChrShowLinefeed ( char chr )
{
   if (!chr == !opts.chrShowLinefeed)
      return;
   opts.chrShowLinefeed = chr;
   invalidateAll(false);
}

void GTextViewer::setChrShowSpace ( char chr )
{
   if (!chr == !opts.chrShowSpace)
      return;
   opts.chrShowSpace = chr;
   invalidateAll(false);
}

void GTextViewer::setChrShowTab ( char chr )
{
   if (!chr == !opts.chrShowTab)
      return;
   opts.chrShowTab = chr;
   invalidateAll(false);
}

bool GTextViewer::isVisibleY ( int y ) const
{
   return y >= idxFirstVisibleLine &&
          y < idxFirstVisibleLine + winLineNr;
}

bool GTextViewer::isVisibleVirtualX ( int virtualX ) const
{
   return virtualX >= idxFirstVisibleColumn &&
          virtualX < idxFirstVisibleColumn + winColumnNr;
}

bool GTextViewer::isVisiblePos ( const GTextViewer::Pos& absPos ) const
{
   if (!isVisibleY(absPos.y))
      return false;
   int virtualX = calcXPosFromPhysicalToVirtual(absPos).x;
   return isVisibleVirtualX(virtualX);
}

bool GTextViewer::isWordWrap () const
{
   return opts.wordWrap;
}
