/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GTextEntry.h"
#include "glib/gui/GDialogFrame.h"
#include "glib/gui/event/GKeyMessage.h"
#include "glib/gui/layout/GBorderLayout.h"
#include "glib/sys/GSystem.h"
#include "glib/util/GLog.h"

GTextEntry::Peer::Peer ( GTextEntry& theEntry )
                 :GWindow(GWindowClass::ENTRYFIELD, GString::Empty, true),
                  theEntry(theEntry)
{
}

GTextEntry::Peer::~Peer ()
{
}

void GTextEntry::Peer::onFocusKill ()
{
   theEntry.moveCursorHome();
   theEntry.setSelection(0, 0);
}

void GTextEntry::Peer::onFocusSet ()
{
   theEntry.moveCursorEnd();
   theEntry.selectAll();
}

bool GTextEntry::Peer::onKeyDown ( const GKeyMessage& key )
{
   GKey::KeyCode code = key.getCode();
   switch (code)
   {
      case GKey::KEY_RIGHT:
      case GKey::KEY_LEFT:
         if (dynamic_cast<GDialogFrame*>(&getTopLevelWindow()) == null)
            return GWindow::onKeyDown(key);
         // Give this keyboard event directly to the system.
         return callDefaultKeyHandler(key, true, true);

      default:
         return GWindow::onKeyDown(key);
   }
}

GTextEntry::GTextEntry ( const GString& name, 
                         const GString& constraints,
                         GWindow& parentWin, 
                         long winStyle, 
                         long winStyle2,
                         int maxTextLength, 
                         ADJUST adjust,
                         bool border, 
                         bool password, 
                         bool readOnly )
           :GAbstractTextField(parentWin, name, 
                               constraints, winStyle, 
                               winStyle2),
            peer(*this)
{
   int f = (WS_VISIBLE | ES_AUTOSCROLL |
           (adjust == RIGHT ? ES_RIGHT : (adjust == CENTER ? ES_CENTER : ES_LEFT)) |
           (border ? ES_MARGIN : 0) |
           (password ? ES_UNREADABLE : 0) |
           (readOnly ? ES_READONLY : 0));
   peer.init("Peer", this, this, f, WS2_DEFAULTPAINT | WS2_IGNORE_COLORS_PROFILE | WS2_NOT_STATIC | WS2_OS2Y | WS2_USE_SAME_PROFILE_SECTION_NAME_AS_PARENT);
   // Make room for the border that is painted by the underlying system
   // around the child window of ours (that is; the window that actually
   // forms the Text Entry Field).
   setInsets(new GInsets(3, 3, 3, 3), true);

   peer.setFocusable(true);
   setLayoutManager(new GBorderLayout(), true);
   setMaxTextLength(maxTextLength);
   setBackgroundColor(GSystem::GetSystemColor(GSystem::SCID_TEXTENTRYBCK));
   setForegroundColor(GSystem::GetSystemColor(GSystem::SCID_TEXTENTRYTXT));
}

GTextEntry::~GTextEntry ()
{
}

bool GTextEntry::onNotify ( int ctrlID, int notifyID, int data, int& sysAnswerToReturn )
{
   if (ctrlID != peer.getWindowID())
      return false;

   switch (notifyID)
   {
      case EN_CHANGE: // The Entry Field Text has been changed.
         fireValueChangeListeners();
         break;
   }

   return true;
}

void GTextEntry::copy () const
{
   peer.sendMessage(EM_COPY);
}

GWindow& GTextEntry::getPeer () const
{
   return peer;
}

void GTextEntry::cut ()
{
   peer.sendMessage(EM_CUT);
}

void GTextEntry::paste ()
{
   peer.sendMessage(EM_PASTE);
}

void GTextEntry::setSelection ( int selStart, int selEnd )
{
   peer.sendMessage(EM_SETSEL, MPFROM2SHORT(selStart, selEnd));
}

int GTextEntry::getSelectionEnd () const
{
   int sel = int(peer.sendMessage(EM_QUERYSEL));
   return sel & 0xFFFF;
}

int GTextEntry::getSelectionStart () const
{
   int sel = int(peer.sendMessage(EM_QUERYSEL));
   return (sel >> 16) & 0xFFFF;
}

int GTextEntry::getTextLength () const
{
   GString text = getText();
   return text.length();
}

void GTextEntry::setMaxTextLength ( int maxLength )
{
   if (maxLength <= 0)
      maxLength = 32000;
   peer.sendMessage(EM_SETTEXTLIMIT, MPFROMSHORT(maxLength));
}

void GTextEntry::setReadOnly ( bool flag )
{
   peer.sendMessage(EM_SETREADONLY, MPFROMSHORT(flag));
}

void GTextEntry::appendText ( const GString& text )
{
   GString str = getText();
   str += text;
   setText(str);
   moveCursorEnd();
}

void GTextEntry::setText ( const GString& text )
{
   peer.setText(text);
}

int GTextEntry::getPreferredHeight () const
{
   return getHeightOfString("X") + 8;
}

int GTextEntry::getPreferredWidth () const
{
   return (32 * getWidthOfString("x")) + 8;
}
