/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GStaticText.h"
#include "glib/gui/GDialogPanel.h"
#include "glib/gui/GGraphics.h"
#include "glib/util/GLog.h"

const GString GStaticText::DefaultFont = "9.WarpSans Bold";

GStaticText::GStaticText ( const GString& name,
                           const GString& constraints,
                           GWindow& parentWin,
                           long winStyle,
                           long winStyle2,
                           ADJUST adjust,
                           bool leader,
                           bool hotKey,
                           GWindow::LeaderStyle style,
                           bool wordWrap,
                           VADJUST vadjust )
            :GWindow(name, 
                     constraints, 
                     &parentWin,
                     &parentWin,
                     winStyle,
                     winStyle2 | WS2_AUTO_SHOW_TOOLTIP | WS2_OS2Y,
                     GWindowClass::GENERIC,
                     GGraphics::DUMMY_COLOR,
                     leader ? GColor::DBLUE : GGraphics::DUMMY_COLOR,
                     leader ? GStaticText::DefaultFont : GWindow::SysDefFontNameSize),
             adjust(adjust),
             vadjust(vadjust),
             wordWrap(wordWrap),
             useLeader(leader),
             useHotKey(hotKey),
             hotKeyPos(-1),
             overriddenTooltipPos(GTooltip::PosInheritFromParent)
{
   setLeaderStyle(style);
}

GStaticText::~GStaticText ( void )
{
}

GString GStaticText::getText () const
{
   return text;
}

void GStaticText::setText ( const GString& txt_ )
{
   changeValue(txt_);
}

int GStaticText::getPreferredWidth () const
{
   // Return the explicit preferred width, if set by user code.
   if (GWindow::preferredSize != null)
      return GWindow::getPreferredWidth();

   // Return the default preferred width.
   const GInsets& ins = getInsets();
   int ret = getWidthOfString(text) + 6;
   return ret + ins.left + ins.right;
}

bool GStaticText::isEmpty () const
{
   return text == GString::Empty;
}

GString GStaticText::queryValue () const
{
   return text;
}

void GStaticText::changeValue ( const GString& newValue, bool /*notify*/ )
{
   GStringl str(newValue); // Load text, if it is loadable.
   if (str == text)
      return;
   text = str;
   GWindow::setText(text);
   if (useHotKey)
      hotKeyPos = autoAssocAHotKey(text);
   invalidateAll(false);
}

void GStaticText::paintTextLine ( GGraphics& g, const GColor& frgColor, const GString& textLine, int ypos, int lineCount, int curLine, int displace )
{
   GString text = textLine; // Working copy, in case we need to manipulate it.
   bool leader = (useLeader && (curLine == lineCount-1)); // Leader (dots) only on last line of text.
   int hotPos = ((curLine == 0) ? hotKeyPos : -1); // Mnemonic only on first line of text.
   ADJUST adjust = this->adjust;

   if (useLeader)
   {
      switch (getLeaderStyle())
      {
         case LeaderStyle_CLASSIC:
              break;

         case LeaderStyle_LEFT:
              adjust = LEFT;
              leader = false;
              if (curLine == lineCount-1)
              {
                 if (text.lastChar() != ':')
                    text += ":";
              }
              break;

         case LeaderStyle_RIGHT:
              adjust = RIGHT;
              leader = false;
              if (curLine == lineCount-1)
              {
                 if (text.lastChar() != ':')
                    text += ":";
              }
              break;

         case LeaderStyle_FANCY1:
         default:
              leader = false;
              if (adjust == LEFT)
              {
                 text = GString::Blank + text;
                 if (hotPos >= 0)
                    hotPos += 1;
              }
              break;
      }
   }

   GDimension dm = getWindowSize();
   int iTextYPos = ypos - displace;
   int iTextWidth = g.getWidthOfString(text);
   int iTextXPos;

   switch (adjust)
   {
      case CENTER:
           iTextXPos = (dm.width / 2) - (iTextWidth / 2);
           if (iTextXPos < 0)
              iTextXPos = 0;
           break;

      case RIGHT:
           iTextXPos = dm.width - iTextWidth;
           if (iTextXPos < 0)
              iTextXPos = 0;
           break;

      case LEFTMARGINE:
           iTextXPos = 4;
           break;

      case LEFT:
      default:
           iTextXPos = 0;
           break;
   }

   iTextXPos += displace;

   g.setColor(frgColor);
   g.drawTextMnemonic(iTextXPos, iTextYPos, text, hotPos);

   if (leader)
   {
      int remainingWidth = dm.width - iTextWidth - 2;
      int leaderStartXPos = iTextWidth + 1;
      int currentXPos = leaderStartXPos;
      int widthOfDot = g.getWidthOfString(".");
      int widthOfColon = g.getWidthOfString(":");
      while (remainingWidth > 0)
      {
         if (remainingWidth <= widthOfColon)
         {
            // Draw the ending colon
            g.drawText(currentXPos, iTextYPos, ":");
            remainingWidth -= widthOfColon;
            widthOfDot += widthOfColon;
         }
         else
         {
            // Draw next dot
            g.drawText(currentXPos, iTextYPos, ".");
            remainingWidth -= widthOfDot;
            currentXPos += widthOfDot;
         }
      }
   }
}

void GStaticText::paintText ( GGraphics& g, const GColor& frgColor, int displace )
{
   GDimension dim = getWindowSize();
   int fontHeight = g.getFontHeight();

   if (wordWrap)
   {
      GArray<GString> strings(16);
      g.getWordWrappedText(text, dim.width, strings);
      const int numLines = strings.getCount();

      int ypos;
      switch (vadjust)
      {
         case TOP:
              ypos = dim.height - fontHeight;
              break;

         case BOTTOM:
              ypos = fontHeight * (numLines - 1);
              if (ypos > dim.height - fontHeight)
                 ypos = dim.height - fontHeight;
              break;

         case CENTER:
         default:
              ypos = (dim.height / 2) + ((fontHeight * (numLines - 1)) / 2);
              if (ypos > dim.height - fontHeight)
                 ypos = dim.height - fontHeight;
              break;
      }

      for (int i=0; i<numLines && ypos >= 0; i++, ypos -= fontHeight)
      {
         const GString& textLine = strings[i];
         paintTextLine(g, frgColor, textLine, ypos, numLines, i, displace);
      }
   }
   else
   {
      int ypos;
      switch (vadjust)
      {
         case TOP:
              ypos = dim.height - fontHeight;
              break;

         case BOTTOM:
              ypos = 0;
              break;

         case CENTER:
         default:
              ypos = (dim.height / 2) - (fontHeight / 2);
              break;
      }
      paintTextLine(g, frgColor, text, ypos, 1, 0, displace);
   }
}

bool GStaticText::onPaintBackground ( GGraphics& g, const GRectangle& rect )
{
   if (useLeader && getLeaderStyle() == LeaderStyle_FANCY1)
   {
      // We are going to use our own background color to paint the
      // fancy arrow, upon <i>onPaint()</i>. Thefore, we are better
      // using the background color of our parent dialog (probably light
      // gray) to fill our background area that is "underneath" the arrow.
      GWindow* pwin = getParentWindow();
      GColor bck = (pwin == null ? GColor::GRAY : pwin->getBackgroundColor());
      g.setColor(bck);
      g.drawFilledRectangle(rect);
   }
   else
   {
      g.setColor(getBackgroundColor());
      g.drawFilledRectangle(rect);
   }

   return true;
}

bool GStaticText::onPaint ( GGraphics& g, const GRectangle& /*rect*/ )
{
   GColor bckColor = getBackgroundColor();

   // Paint the fancy arrow, if any
   if (useLeader && getLeaderStyle() == LeaderStyle_FANCY1)
   {
      GColor parentBckColor = getParentWindow()->getBackgroundColor();
      if (bckColor == parentBckColor)
         bckColor = GColor(255, 255, 153); // A kind of yellow

      GDimension dm = getWindowSize();
      int fontWidth = g.getWidthOfString("x");

      g.setColor(bckColor);
      GArray<GPoint> polygon(6);
      polygon.add(new GPoint(0, dm.height - 1));
      polygon.add(new GPoint(dm.width - fontWidth, dm.height - 1));
      polygon.add(new GPoint(dm.width, ((dm.height - 2) / 2) + 1));
      polygon.add(new GPoint(dm.width - fontWidth, 1));
      polygon.add(new GPoint(0, 1));
      g.drawFilledPolygon(polygon);

      g.setColor(GColor::BLACK);
      g.setPosition(1, 0);
      g.setLineType(GGraphics::LTSolid);
      g.drawLineTo(dm.width - fontWidth, 0);
      g.drawLineTo(dm.width, ((dm.height - 2) / 2) + 1);
   }

   // Paint the text string and the leading dots, if any
   if (isEnabled())
   {
      GColor frg = getForegroundColor();
      paintText(g, frg, 0);
   }
   else
   {
      GColor col1 = bckColor.getLighter();
      GColor col2 = bckColor.getDarker();
      paintText(g, col1, 1);
      paintText(g, col2, 0);
   }

   return true;
}

GString GStaticText::getTooltipText () const
{
   // By default, use the tooltip position as set in our super class.
   overriddenTooltipPos = GTooltip::PosInheritFromParent;

   // ---
   GString tt = GWindow::getTooltipText();
   if (wordWrap)
      return tt;

   GGraphics g(this);
   GDimension dim = getWindowSize();
   GDimension textDim = g.getTextDim(text);
   int faceWidth = dim.width;
   if (useLeader)
      faceWidth -= g.getWidthOfString("x");

   // Show the current text as a tooltip if the text is too long to be all 
   // visible within the window face area.
   if (textDim.height >= dim.height || textDim.width >= faceWidth)
   {
      // Show the full length of the text in a tooltip box that is 
      // overlapping our self.
      overriddenTooltipPos = GTooltip::PosOverlap;
      return text;
   }

   // The component is wide enough to show all parts of the text, so show 
   // tooltip only if explicitly set.
   return tt;
}

GTooltip::Position GStaticText::getTooltipPosition () const
{
   if (overriddenTooltipPos != GTooltip::PosInheritFromParent)
      return overriddenTooltipPos;
   return GWindow::getTooltipPosition();
}
