/* --------------------------------------------------------------------------
 *
 * Copyright (C) 2007 Leif Erik Larsen, Kjerringvik, Norway.
 *
 * This file is part of the Open Source Edition of Larsen Commander, as
 * available from http://home.online.no/~leifel/lcmd/.  This code is free 
 * software; you can redistribute it and/or modify it under the terms of 
 * the GNU General Public License version 3 only, as published by the 
 * Free Software Foundation.  
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 3 at http://www.gnu.org/licenses/gpl-3.0.txt for more details 
 * (a copy is included in the LICENSE file that accompanied this code).
 *
 * ------------------------------------------------------------------------ */

#include "glib/gui/GScrollbar.h"
#include "glib/util/GMath.h"
#include "glib/util/GLog.h"
#include "glib/sys/GSystem.h"

GScrollbar::GScrollbar ( const GString& name,
                         const GString& constraints,
                         GWindow& parentWin,
                         bool vertical,
                         long winStyle,
                         long winStyle2 )
           :GWindow(name,
                    constraints,
                    &parentWin,
                    &parentWin,
                    winStyle | SBS_AUTOTRACK | (vertical ? SBS_VERT : SBS_HORZ),
                    winStyle2 | WS2_DEFAULTPAINT | 
                                WS2_IGNORE_COLORS_AND_FONT_PROFILE | 
                                WS2_OS2Y |
                                WS2_NOT_STATIC,
                    GWindowClass::SCROLLBAR),
            MAX_SYSTEM_SCROLL_RANGE(GShort::MAX_VALUE),
            realWorldScrolRange(0),
            thumbLen(0)
{
}

GScrollbar::~GScrollbar ()
{
}

int GScrollbar::GetDefaultVScrollbarWidth ()
{
   return GSystem::GetSystemMetrics(GSystem::SMID_CXVSCROLL);
}

int GScrollbar::GetDefaultHScrollbarHeight ()
{
   return GSystem::GetSystemMetrics(GSystem::SMID_CYHSCROLL);
}

int GScrollbar::GetDefaultScrollbarColor ()
{
   return GSystem::GetSystemColor(GSystem::SCID_SCROLLBARBCK);
}

int GScrollbar::convertFromSysPos ( int syspos ) const
{
   syspos = GMath::Max(0, syspos);
   if (realWorldScrolRange <= GScrollbar::MAX_SYSTEM_SCROLL_RANGE)
      return syspos;
   double temp = double(realWorldScrolRange * double(syspos));
   double pos = temp / double(GScrollbar::MAX_SYSTEM_SCROLL_RANGE);
   return int(pos);
}

int GScrollbar::convertToSysPos ( int pos ) const
{
   pos = GMath::Max(0, pos);
   if (realWorldScrolRange <= GScrollbar::MAX_SYSTEM_SCROLL_RANGE)
      return pos;
   if (pos == 0)
      return 0;
   double each = double(realWorldScrolRange) / double(pos);
   if (each == 0.0)
      return 0;
   double syspos = double(GScrollbar::MAX_SYSTEM_SCROLL_RANGE) / each;
   return int(syspos + 0.5);
}

int GScrollbar::getScrollPos () const
{
   int syspos = getSysScrollPos();
   int pos = convertFromSysPos(syspos);
   return pos;
}

void GScrollbar::setScrollPos ( int pos )
{
   int syspos = convertToSysPos(pos);
   setSysScrollPos(syspos);
}

int GScrollbar::getScrollRange () const
{
   return realWorldScrolRange;
}

int GScrollbar::getThumbLength () const
{
   return thumbLen;
}

void GScrollbar::setScrollRange ( int scrolLen, int thumbLen )
{
   if (thumbLen > scrolLen)
      thumbLen = scrolLen;
   this->thumbLen = GMath::Max(0, thumbLen);
   this->realWorldScrolRange = GMath::Max(0, scrolLen);
   int len = GMath::Min(scrolLen, GScrollbar::MAX_SYSTEM_SCROLL_RANGE - thumbLen);
   setSysScrollRange(len, thumbLen);
}

void GScrollbar::setScrollPosAndRange ( int pos, int scrolLen, int thumbLen )
{
   if (thumbLen > scrolLen)
      thumbLen = scrolLen;
   this->thumbLen = GMath::Max(0, thumbLen);
   this->realWorldScrolRange = GMath::Max(0, scrolLen);
   int syspos = convertToSysPos(pos);
   int len = GMath::Min(scrolLen, GScrollbar::MAX_SYSTEM_SCROLL_RANGE - thumbLen);
   setSysScrollPosAndRange(syspos, len, thumbLen);
}

int GScrollbar::getSysScrollPos ( bool currentThumbTrackPos ) const
{
   // We don't have any way to differ between the current thumb track and 
   // the current scroll position, in OS/2. We must therefore ignore 
   // the "currentThumbTrackPos" parameter, and always return the current
   // scroll position only.
   GScrollbar& self = const_cast<GScrollbar&>(*this); // Cast away const.
   return int(self.sendMessage(SBM_QUERYPOS));
}

void GScrollbar::setSysScrollPos ( int pos )
{
   sendMessage(SBM_SETPOS, MPFROM2SHORT(pos, 0));
}

void GScrollbar::setSysScrollRange ( int scrolLen, int thumbLen )
{
   sendMessage(SBM_SETSCROLLBAR, MPFROMSHORT(0), MPFROM2SHORT(0, scrolLen));
   sendMessage(SBM_SETTHUMBSIZE, MPFROM2SHORT(thumbLen, scrolLen + thumbLen));
}

void GScrollbar::setSysScrollPosAndRange ( int pos, int scrolLen, int thumbLen )
{
   sendMessage(SBM_SETSCROLLBAR, MPFROMSHORT(pos), MPFROM2SHORT(0, scrolLen));
   sendMessage(SBM_SETTHUMBSIZE, MPFROM2SHORT(thumbLen, scrolLen + thumbLen));
}
